﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_DeviceCode.h>
#include <nn/nn_SdkAssert.h>

#include <nn/i2c/driver/i2c_I2cDeviceProperty.h>
#include <nn/i2c/driver/i2c_II2cDriver.h>
#include <nn/i2c/i2c_Type.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/util/util_IntrusiveList.h>

namespace nnd { namespace i2c { namespace skeleton { namespace detail {

class I2cBusAccessor : public nn::i2c::driver::II2cDriver
{
    NN_DDSF_CAST_SAFE_DECL;

    NN_DISALLOW_COPY(I2cBusAccessor);
    NN_DISALLOW_MOVE(I2cBusAccessor);

private:
    // I2cBusAccessorManager で管理するために必要な定義
    nn::util::IntrusiveListNode m_ListNode;

    // I2cBusAccessorManager で管理するために必要な定義
    friend class nn::util::IntrusiveListMemberNodeTraits<I2cBusAccessor, &I2cBusAccessor::m_ListNode>;

public:
    // I2cBusAccessorManager で管理するために必要な定義
    typedef nn::util::IntrusiveList<I2cBusAccessor, nn::util::IntrusiveListMemberNodeTraits<I2cBusAccessor, &I2cBusAccessor::m_ListNode>> List;

public:
    I2cBusAccessor() NN_NOEXCEPT;

    void RegisterDeviceCode(nn::DeviceCode deviceCode) NN_NOEXCEPT;

    virtual void InitializeDriver() NN_NOEXCEPT NN_OVERRIDE;

    virtual nn::Result InitializeDevice(nn::i2c::driver::I2cDeviceProperty* pDevice) NN_NOEXCEPT NN_OVERRIDE;

    virtual void FinalizeDevice(nn::i2c::driver::I2cDeviceProperty* pDevice) NN_NOEXCEPT NN_OVERRIDE;

    virtual nn::Result Send(nn::i2c::driver::I2cDeviceProperty* pDevice, const void* pInData,
        size_t dataBytes, nn::i2c::TransactionOption inOption) NN_NOEXCEPT NN_OVERRIDE;

    virtual nn::Result Receive(void* pOutData, nn::i2c::driver::I2cDeviceProperty* pDevice,
        size_t dataBytes, nn::i2c::TransactionOption inOption) NN_NOEXCEPT NN_OVERRIDE;

    virtual nn::os::SdkMutex& GetTransactionOrderMutex() NN_NOEXCEPT NN_OVERRIDE;

    virtual void SuspendBus() NN_NOEXCEPT NN_OVERRIDE;

    virtual void SuspendPowerBus() NN_NOEXCEPT NN_OVERRIDE;

    virtual void ResumeBus() NN_NOEXCEPT NN_OVERRIDE;

    virtual void ResumePowerBus() NN_NOEXCEPT NN_OVERRIDE;

    virtual const nn::DeviceCode& GetDeviceCode() const NN_NOEXCEPT NN_OVERRIDE;

    // I2cBusAccessorManager で管理するために必要なメンバ関数
    void RemoveFrom(List* pList) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pList);
        pList->erase(pList->iterator_to(*this));
    }

    // I2cBusAccessorManager で管理するために必要なメンバ関数
    bool IsLinkedToList() const NN_NOEXCEPT
    {
        return m_ListNode.IsLinked();
    }

private:
    //! 一連の送受信処理を行う間にロックするべきミューテックス
    nn::os::SdkMutex m_TransactionOrderMutex;

    //! バスに対応するデバイスコード
    nn::DeviceCode m_DeviceCode { 0 };
};

}}}} // nnd::i2c::skeleton::detail
