﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_BitTypes.h>
#include <nn/nn_Common.h>

#include <nn/dt.h>

#include <nn/i2c/detail/i2c_Log.h>
#include <nn/i2c/driver/i2c_DriverService.h>
#include <nn/i2c/driver/i2c_I2cDeviceProperty.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include "i2cSkeleton_I2cBusAccessor.h"
#include "i2cSkeleton_I2cBusAccessorManager.h"
#include "i2cSkeleton_I2cDevicePropertyManager.h"
#include "i2cSkeleton_DeviceCodeNodeParser.h"

namespace nnd { namespace i2c { namespace skeleton { namespace detail {

namespace {

// TODO: Add mutex if needed
const size_t MaxNodesSize = 0x1000;
char g_Nodes[MaxNodesSize];

}

nn::Result DeviceCodeBusParser::OnCompatibleNodeBegin(nn::dt::Node node) NN_NOEXCEPT
{
    bool hasNodes;
    NN_RESULT_DO(nn::dt::IsPropertyExist(&hasNodes, &node, "nodes"));
    NN_ABORT_UNLESS(hasNodes, "i2c-subsys doesn't have \"nodes\".");

    size_t nodesSize = 0;
    NN_RESULT_DO(nn::dt::GetPropertyString(&nodesSize, g_Nodes, MaxNodesSize, &node, "nodes"));

    nn::dt::StringForwardList nodes(g_Nodes, g_Nodes + nodesSize);

    m_Iterator = nodes.cbegin();
    m_End = nodes.cend();

    NN_RESULT_SUCCESS;
}

nn::Result DeviceCodeBusParser::OnDeviceCodeFound(nn::DeviceCode deviceCode, nn::dt::Node node) NN_NOEXCEPT
{
    NN_UNUSED(node);

    NN_ABORT_UNLESS(m_Iterator < m_End, "number of \"nodes\" is lesser than number of \"device-codes\".");

    auto nodePath = *m_Iterator;
    m_Iterator++;

    nn::dt::Node i2cBusNode;
    auto result = nn::dt::FindNodeByPath(&i2cBusNode, nodePath);
    NN_RESULT_TRY(result)
    NN_RESULT_CATCH(nn::dt::ResultNodeNotExist)
    {
        NN_DETAIL_I2C_WARN("node not exist on device tree. Skipping parsing\n");
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_CATCH_ALL
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
    NN_RESULT_END_TRY

    // I2cBusAccessor の生成もしくは取得
    I2cBusAccessor* pI2cBusAccessor = nullptr;
    result = ParseI2cBusAccessor(&pI2cBusAccessor, &i2cBusNode);
    NN_RESULT_TRY(result)
    NN_RESULT_CATCH(nn::dt::ResultPropertyNotExist)
    {
        NN_DETAIL_I2C_ERROR("parent node (%s) did not have expected property!\n", nodePath);
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_CATCH(nn::dt::ResultPropertyTypeError)
    {
        NN_DETAIL_I2C_ERROR("parent node (%s) did not have expected type property!\n", nodePath);
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_CATCH_ALL
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
    NN_RESULT_END_TRY

    pI2cBusAccessor->RegisterDeviceCode(deviceCode);

    NN_RESULT_SUCCESS;
}

nn::Result DeviceCodeBusParser::ParseI2cBusAccessor(I2cBusAccessor** ppOutI2cBusAccessor, const nn::dt::Node* pI2cBusNode) NN_NOEXCEPT
{
    // implement me
    NN_UNUSED(pI2cBusNode);

    // 既に I2cBusAccessor が初期化されリストに登録されているか確認する
    *ppOutI2cBusAccessor = m_I2cBusAccessorAllocator.FindOnAllocateList(
        [&](const I2cBusAccessor* pI2cBusAccessor) NN_NOEXCEPT -> bool
        {
            // implement me
            // Check parsed BusAccessor already exists in AllocateList
            // e.g. pI2cBusAccessor->GetBaseAddress() == baseAddress;
            NN_UNUSED(pI2cBusAccessor);
            return true;
        }
    );

    if ( *ppOutI2cBusAccessor )
    {
        // implement me
    }
    else
    {
        *ppOutI2cBusAccessor = m_I2cBusAccessorAllocator.Alloc();

        NN_ABORT_UNLESS(*ppOutI2cBusAccessor, "Could not allocate I2cBusAccessor buffer for new i2c controller. Please consider increasing max buses.");

        // implement me (initialize I2cBusAccessor)

        nn::i2c::driver::RegisterDriver(*ppOutI2cBusAccessor);
    }

    NN_RESULT_SUCCESS;
}

nn::Result DeviceCodeDeviceParser::OnDeviceCodeFound(nn::DeviceCode deviceCode, nn::dt::Node node) NN_NOEXCEPT
{
    NN_UNUSED(node);

    NN_ABORT_UNLESS(m_Iterator < m_End, "number of \"nodes\" is lesser than number of \"device-codes\".");

    auto nodePath = *m_Iterator;
    m_Iterator++;

    nn::dt::Node i2cDeviceNode;
    auto result = nn::dt::FindNodeByPath(&i2cDeviceNode, nodePath);
    NN_RESULT_TRY(result)
    NN_RESULT_CATCH(nn::dt::ResultNodeNotExist)
    {
        NN_DETAIL_I2C_WARN("node not exist on device tree. Skipping parsing\n");
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_CATCH_ALL
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
    NN_RESULT_END_TRY

    // I2cDeviceProperty の生成
    nn::i2c::driver::I2cDeviceProperty* pI2cDeviceProperty = nullptr;
    result = ParseI2cDeviceProperty(&pI2cDeviceProperty, &i2cDeviceNode);
    NN_RESULT_TRY(result)
    NN_RESULT_CATCH(nn::dt::ResultPropertyNotExist)
    {
        NN_DETAIL_I2C_ERROR("node (%s) did not have expected property!\n", nodePath);
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_CATCH(nn::dt::ResultPropertyTypeError)
    {
        NN_DETAIL_I2C_ERROR("node (%s) did not have expected type property!\n", nodePath);
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_CATCH_ALL
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
    NN_RESULT_END_TRY

    // I2cDeviceProperty の解放
    bool needFreeI2cDeviceProperty = true;

    NN_UTIL_SCOPE_EXIT
    {
        if ( needFreeI2cDeviceProperty )
        {
            m_I2cDevicePropertyAllocator.Free(pI2cDeviceProperty);
        }
    };

    nn::dt::Node i2cBusNode;

    result = nn::dt::GetParentNode(&i2cBusNode, &i2cDeviceNode);
    NN_RESULT_TRY(result)
    NN_RESULT_CATCH(nn::dt::ResultNodeNotExist)
    {
        NN_DETAIL_I2C_WARN("parent node not exist on device tree. Skipping parsing\n");
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_CATCH_ALL
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
    NN_RESULT_END_TRY

    // I2cBusAccessor の生成もしくは取得
    I2cBusAccessor* pI2cBusAccessor = nullptr;
    result = ParseI2cBusAccessor(&pI2cBusAccessor, &i2cBusNode);
    NN_RESULT_TRY(result)
    NN_RESULT_CATCH(nn::dt::ResultPropertyNotExist)
    {
        NN_DETAIL_I2C_ERROR("parent node (%s) did not have expected property!\n", nodePath);
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_CATCH(nn::dt::ResultPropertyTypeError)
    {
        NN_DETAIL_I2C_ERROR("parent node (%s) did not have expected type property!\n", nodePath);
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_CATCH_ALL
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
    NN_RESULT_END_TRY

    pI2cBusAccessor->RegisterDevice(pI2cDeviceProperty);
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::driver::RegisterDeviceCode(deviceCode, pI2cDeviceProperty));

    needFreeI2cDeviceProperty = false;
    NN_RESULT_SUCCESS;
}

nn::Result DeviceCodeDeviceParser::ParseI2cDeviceProperty(nn::i2c::driver::I2cDeviceProperty** ppOutI2cDeviceProperty, const nn::dt::Node* pI2cDeviceNode) NN_NOEXCEPT
{
    // implement me
    NN_UNUSED(pI2cDeviceNode);
    uint32_t slaveAddress = 0;
    auto addressingMode = nn::i2c::AddressingMode_BitWidth7;

    *ppOutI2cDeviceProperty = m_I2cDevicePropertyAllocator.Alloc();

    NN_ABORT_UNLESS(*ppOutI2cDeviceProperty, "Could not allocate I2cDevicePropertySkeleton buffer for new device code. Please consider increasing device code entry.");

    *ppOutI2cDeviceProperty = new (*ppOutI2cDeviceProperty) nn::i2c::driver::I2cDeviceProperty(static_cast<nn::Bit16>(slaveAddress), addressingMode);

    NN_RESULT_SUCCESS;
}

}}}} // nnd::i2c::skeleton::detail
