﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/dt.h>

#include <nn/gpio/detail/gpio_Log.h>
#include <nn/gpio/driver/gpio_DriverService.h>

#include "gpioTegra_DeviceCodeNodeParser.h"
#include "gpioTegra_Pad.h"

// #define NN_DETAIL_GPIO_DUMP_DEVICE_ENTRY

namespace nnd { namespace gpio { namespace tegra { namespace detail {

namespace {
    // "pads" プロパティの取り出し
    nn::Result GetInternalGpioPadNumber(int* pOutInternalGpioPadNumber, int* pCurrentPropertyIndex, nn::dt::Node node) NN_NOEXCEPT
    {
        NN_RESULT_DO(nn::dt::GetProperty(pOutInternalGpioPadNumber, &node, "pads", *pCurrentPropertyIndex));
        ++*pCurrentPropertyIndex;
        NN_RESULT_SUCCESS;
    }

    // "nvidia,pmc-wakeup" プロパティの取り出し
    nn::Result GetNvidiaPmcWakeup(decltype(PadInfo::wakeEvent) * pOutWakeEvent, int* pCurrentPropertyIndex, nn::dt::Node node) NN_NOEXCEPT
    {
        NN_RESULT_DO(nn::dt::GetProperty(pOutWakeEvent, &node, "nvidia,pmc-wakeup", *pCurrentPropertyIndex));
        ++*pCurrentPropertyIndex;
        NN_RESULT_SUCCESS;
    }
}

nn::Result DeviceCodeNodeParserImpl::OnCompatibleNodeBegin(nn::dt::Node node) NN_NOEXCEPT
{
    NN_RESULT_DO(nn::dt::IsPropertyExist(&m_HasGpiosProperty, &node, "pads"));
    NN_RESULT_DO(nn::dt::IsPropertyExist(&m_HasNvidiaPmcWakeupProperty, &node, "nvidia,pmc-wakeup"));

    // 各プロパティの中身を舐めるためのインデックスの初期化
    m_PropertyGpiosIndex = 0;
    m_PropertyNvidiaPmcWakeupIndex = 0;
    NN_RESULT_SUCCESS;
}

nn::Result DeviceCodeNodeParserImpl::OnDeviceCodeFound(nn::DeviceCode deviceCode, nn::dt::Node node) NN_NOEXCEPT
{
    bool needsRollback = true;

    int padNumber = InternalGpioPadNumberNone;
    if ( m_HasGpiosProperty )
    {
        NN_RESULT_DO(GetInternalGpioPadNumber(&padNumber, &m_PropertyGpiosIndex, node));
    }

    PadInfo padInfo;
    padInfo.wakeEvent = WakeEventNone;
    if ( m_HasNvidiaPmcWakeupProperty )
    {
        NN_RESULT_DO(GetNvidiaPmcWakeup(&padInfo.wakeEvent, &m_PropertyNvidiaPmcWakeupIndex, node));
    }

    // TORIAEZU: ちゃんと汎用的実装にするなら、 DeviceTree path を見て物理アドレス等の設定を取得するようにする

    // 同一の pad number / wake event の組み合わせのものは同一として、確保済の pad があれば再利用する
    auto pPad = m_pDriver->FindIdenticalPad(padNumber, padInfo);
    bool wasPadNewAllocated = false;
    if ( !pPad )
    {
        // すでに確保されている同一のデバイスがなかったので、新規に確保する
        pPad = m_PadAllocator.Alloc();
        if ( !pPad )
        {
            NN_DETAIL_GPIO_ERROR("Could not allocate PadTegra buffer for new device code. Please consider increasing device code entry.\n");
            return nn::gpio::ResultTooManyDeviceCodesOnDeviceTree();
        }
        wasPadNewAllocated = true;

        pPad->SetParam(padNumber, padInfo);
        m_pDriver->AddPad(*pPad);
    }
    NN_UTIL_SCOPE_EXIT
    {
        // 新規に確保しつつ処理に失敗した場合はロールバック時に解放
        if ( needsRollback && wasPadNewAllocated )
        {
            m_PadAllocator.Free(pPad);
        }
    };

    NN_RESULT_DO(nn::gpio::driver::RegisterDeviceCode(deviceCode, pPad));

#if defined(NN_DETAIL_GPIO_DUMP_DEVICE_ENTRY)
    NN_DETAIL_GPIO_TRACE_V1("DeviceCode 0x%08x : Pad Number=%d, Wake Event=%d\n",
        deviceCode.GetInternalValue(),
        padNumber,
        padInfo.wakeEvent
    );
#endif

    needsRollback = false;
    NN_RESULT_SUCCESS;
}

}}}} // nnd::gpio::tegra::detail
