﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       タッチコントローラ ftm4cd60d のファームウェアアップデータ用の定義です。
 */

#pragma once

#include <nn/i2c/i2c.h>
#include <nnd/ftm/ftm.h>

namespace nnd { namespace ftm { namespace detail {

class Ftm4cd60dUpdater final
{
    NN_DISALLOW_COPY(Ftm4cd60dUpdater);
    NN_DISALLOW_MOVE(Ftm4cd60dUpdater);

private:
    //!< i2c のセッション
    ::nn::i2c::I2cSession m_Session;

    //!< ファームウェア入力用関数と引数データをまとめた構造体
    struct FirmwareInputCallback final
    {
        //!< ファームウェアバイナリ入力用の関数ポインタ
        FirmwareInputFunctionPointer pFunction;

        //!< 入力用関数に与える引数データのポインタ
        void* pParameter;
    };

    //!< ファームウェアバイナリ入力用のコールバック
    FirmwareInputCallback m_FirmwareInputCallback;

public:
    Ftm4cd60dUpdater() NN_NOEXCEPT;

    //!< i2c のセッションをセットします。
    void SetI2cSession(::nn::i2c::I2cSession session) NN_NOEXCEPT;

    //!< ファームウェアバイナリ入力用のコールバックを登録します。
    void RegisterFirmwareInputCallback(
        FirmwareInputFunctionPointer const pFunction, void* const pParameter) NN_NOEXCEPT;

    //!< ファームウェアバイナリ入力用のコールバックを解除します。
    void UnregisterFirmwareInputCallback() NN_NOEXCEPT;

    //!< ファームウェアアップデートを実行します。
    ::nn::Result Proceed(size_t fileSize) const NN_NOEXCEPT;

    //!< フラッシュ上のファームウェアデータを消去します。
    ::nn::Result EraseFirmware() const NN_NOEXCEPT;

private:
    //!< ファイルサイズが適切か確認します。
    ::nn::Result ConfirmFileSize(size_t fileSize) const NN_NOEXCEPT;

    //!< ファイルの署名が適切か確認します。
    ::nn::Result ConfirmSignature() const NN_NOEXCEPT;

    //!< 書き込みデータの CRC チェックを有効にします。
    ::nn::Result EnableWarmBootCrcCheck() const NN_NOEXCEPT;

    //!< フラッシュのロックを解除します。
    ::nn::Result UnlockFlash() const NN_NOEXCEPT;

    //!< Erase コマンドを発行します。
    ::nn::Result IssueEraseFlashCommand() const NN_NOEXCEPT;

    //!< Erase コマンドの完了を待ちます。
    ::nn::Result WaitForCompletionOfErasing() const NN_NOEXCEPT;

    //!< Erase 処理の状態を読みます。
    ::nn::Result ReadEraseState(char* pOutEraseState, size_t size) const NN_NOEXCEPT;

    //!< Program データを書き込みます。
    ::nn::Result WriteProgram() const NN_NOEXCEPT;

    //!< Configuration データを書き込みます。
    ::nn::Result WriteConfiguration() const NN_NOEXCEPT;

    //!< データが書かれている部分のサイズを取得します。
    ::nn::Result GetActualProgramDataSize(uint32_t* pOutSize, uint32_t offset, uint32_t dataSizeMax) const NN_NOEXCEPT;

    //!< フラッシュにデータを書き込みます。
    ::nn::Result WriteOnFlashMemory(uint32_t offset, uint32_t dataSize, uint16_t address) const NN_NOEXCEPT;

    //!< ファームウェアデータを読み込みます。
    ::nn::Result ReadFirmwareData(char* pOutBuffer, uint32_t address, size_t size) const NN_NOEXCEPT;

    //!< ファームウェアデータを RAM にロードします。
    ::nn::Result LoadFirmwareIntoRam(const char* pBuffer, uint32_t size) const NN_NOEXCEPT;

    //!< RAM からフラッシュへの DMA 転送を開始します。
    ::nn::Result StartFlashDma(uint16_t offset, uint16_t sectorCount) const NN_NOEXCEPT;

    //!< DMA の転送設定を行います。
    ::nn::Result ConfigureFlashDma(uint16_t offset, uint16_t sectorCount) const NN_NOEXCEPT;

    //!< DMA の開始コマンドを発行します。
    ::nn::Result IssueStartFlashDmaCommand() const NN_NOEXCEPT;

    //!< DMA 転送の完了を待ちます。
    ::nn::Result WaitForCompletionOfDma() const NN_NOEXCEPT;

    //!< DMA の転送状態を読みます。
    ::nn::Result ReadDmaState(char* pOutDmaState, size_t size) const NN_NOEXCEPT;

    //!< レジスタにデータを書き込みます。
    ::nn::Result WriteRegister(const void* pInData, size_t dataBytes) const NN_NOEXCEPT;

    //!< レジスタからデータを読み出します。
    ::nn::Result ReadRegister(void* pBuffer, size_t receiveDataBytes, const nn::Bit8* pInData, size_t sendDataBytes) const NN_NOEXCEPT;
};

}}} // namespace nnd::ftm::detail
