﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   NXFP2 向けの ftm 関連のペリフェラル設定値です。
 */

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/nn_SdkAssert.h>

#include <nn/i2c/i2c.h>
#include <nn/gpio/gpio.h>

#include "ftm_BusConfig.h"
#include "ftm_Util.h"
#include "ftm_Debug.h"

namespace nnd { namespace ftm { namespace detail {

namespace {

const ::nn::i2c::I2cDevice    I2cDeviceName       = ::nn::i2c::I2cDevice_Ftm3bd56;
const ::nn::gpio::GpioPadName GpioInterruptName   = ::nn::gpio::GpioPadName_TpIrq;
const ::nn::gpio::GpioPadName GpioResetName       = ::nn::gpio::GpioPadName_TpReset;

// Low レベルになったとき割り込みが入るようにする
const ::nn::gpio::InterruptMode   GpioInterruptMode   = ::nn::gpio::InterruptMode::InterruptMode_LowLevel;

// バスの状態
enum class BusState : uint8_t
{
    Uninitialized, // 未初期化
    Initialized,   // 初期化済み
};
BusState g_BusState = BusState::Uninitialized;

} // namespace

void InitializeBus(::nn::i2c::I2cSession* pOutI2c, ::nn::gpio::GpioPadSession* pOutIrq, ::nn::gpio::GpioPadSession* pOutReset) NN_NOEXCEPT
{
    // すでに初期化済みの場合は何もしない
    if (g_BusState == BusState::Initialized)
    {
        return;
    }

    // 割り込み用 GPIO を入力に設定する
    ::nn::gpio::Initialize();
    ::nn::gpio::OpenSession(pOutIrq, GpioInterruptName);
    ::nn::gpio::SetDirection(pOutIrq, ::nn::gpio::Direction_Input);
    ::nn::gpio::SetInterruptMode(pOutIrq, GpioInterruptMode);

    // リセット用 GPIO に Low 出力
    ::nn::gpio::OpenSession(pOutReset, GpioResetName);
    ::nn::gpio::SetDirection(pOutReset, ::nn::gpio::Direction_Output);
    ::nn::gpio::SetValue(pOutReset, ::nn::gpio::GpioValue_Low);

    // 12us 以上待つ（リセットを反映させるため）
    ::nn::os::SleepThread(::nn::TimeSpan::FromMicroSeconds(20));

    // 2.9V ラインに電源を入れる
    ::nn::i2c::Initialize();
    ::nn::i2c::OpenSession(pOutI2c, I2cDeviceName);

    // 1.8V ラインの上昇を待つため 1ms の待機が必要
    ::nn::os::SleepThread(::nn::TimeSpan::FromMilliSeconds(1));

    // リセット解除（High 出力）
    ::nn::gpio::SetValue(pOutReset, ::nn::gpio::GpioValue_High);

    g_BusState = BusState::Initialized;
}

void FinalizeBus(::nn::i2c::I2cSession* pI2c, ::nn::gpio::GpioPadSession* pIrq, ::nn::gpio::GpioPadSession* pReset) NN_NOEXCEPT
{
    // すでにファイナライズ済みの場合は何もしない
    if (g_BusState == BusState::Uninitialized)
    {
        return;
    }

    // 割り込み用 GPIO のクローズ
    ::nn::gpio::CloseSession(pIrq);

    // リセット用 GPIO を Low 出力
    ::nn::gpio::SetDirection(pReset, ::nn::gpio::Direction_Output);
    ::nn::gpio::SetValue(pReset, ::nn::gpio::GpioValue_Low);

    // 12us 以上待つ（リセットを反映させるため）
    ::nn::os::SleepThread(::nn::TimeSpan::FromMicroSeconds(20));

    // リセット用 GPIO のクローズ
    ::nn::gpio::CloseSession(pReset);
    ::nn::gpio::Finalize();

    // 2.9V ラインの電源を落とす
    ::nn::i2c::CloseSession(*pI2c);
    ::nn::i2c::Finalize();

    g_BusState = BusState::Uninitialized;
}

void SleepInBus(::nn::i2c::I2cSession* pI2c, ::nn::gpio::GpioPadSession* pIrq, ::nn::gpio::GpioPadSession* pReset) NN_NOEXCEPT
{
    FinalizeBus(pI2c, pIrq, pReset);
}

void SleepOutBus(::nn::i2c::I2cSession* pOutI2c, ::nn::gpio::GpioPadSession* pOutIrq, ::nn::gpio::GpioPadSession* pOutReset) NN_NOEXCEPT
{
    InitializeBus(pOutI2c, pOutIrq, pOutReset);
}

void SetPowerEnabled(bool enabled) NN_NOEXCEPT
{
    // nx では nn::i2c::OpenSession() と ::nn::i2c::CloseSession(); により電源制御される
    NN_UNUSED(enabled);
}

void DeassertReset(::nn::gpio::GpioPadSession* pOutReset) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(g_BusState, BusState::Uninitialized);

    // リセット解除
    ::nn::gpio::OpenSession(pOutReset, GpioResetName);
    ::nn::gpio::SetDirection(pOutReset, ::nn::gpio::Direction_Output);
    ::nn::gpio::SetValue(pOutReset, ::nn::gpio::GpioValue_High);
}

}}} // namespace nnd::ftm::detail
