﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>

#include <nn/util/util_BitPack.h>

#include <nnd/bh1730fvc/bh1730fvc.h>

namespace nnd {
namespace bh1730fvc {
namespace detail {


const int   CycleCalcurateConstantValue = 256; // サイクル数計算用の定数
const float OneCycleTime = 2.7f;  // 1サイクル分の時間 [ms]
inline nn::Bit8 ConvertCycleAndTime( volatile nn::Bit8 data)
{
    return static_cast<nn::Bit8>(CycleCalcurateConstantValue - static_cast<int>(data));
}


const size_t    RegSize         = 1;    // レジスタ１つ分のサイズ
const int       InvalidIndex    = -1;   // 無効値

const int       IntegralCycleMin    = 1;   // 最小値
const int       IntegralCycleMax    = 255;   // 最大値

// アドレス
enum class RegAddr : uint8_t
{
    Control                 = 0x00,
    Timing                  = 0x01,
    Interrupt               = 0x02,
    LowThresholdLowByte     = 0x03,
    LowThresholdHighByte    = 0x04,
    HighThresholdLowByte    = 0x05,
    HighThresholdHighByte   = 0x06,
    Gain                    = 0x07,
    Id                      = 0x12,
    Data0LowByte            = 0x14,
    Data0HighByte           = 0x15,
    Data1LowByte            = 0x16,
    Data1HighByte           = 0x17,
};

// Special Commands
enum class SpecialCommand : uint8_t
{
    ResetInterrupt          = 0x01,
    StopManualMeasurment    = 0x02,
    StartManualMeasurment   = 0x03,
    ResetSystem             = 0x04,
};

// Gainの列挙体と、ゲイン値のテーブル
struct GainNameValueCombination
{
    Gain    name;
    int     val;
};
const GainNameValueCombination GainNameValueCombinationList[] = {
    { Gain::X1,      1 },
    { Gain::X2,      2 },
    { Gain::X64,    64 },
    { Gain::X128,  128 },
};
const int NumberOfGain = sizeof(GainNameValueCombinationList) / sizeof(GainNameValueCombinationList[0]);


// センサーデータ
struct SensorData
{
    typedef nn::util::BitPack16::Field< 0,  8, nn::Bit8>    LowByte;    // センサデータの下位 8 bit
    typedef nn::util::BitPack16::Field< 8,  8, nn::Bit8>    HighByte;   // センサデータの上位 8 bit
    typedef nn::util::BitPack16::Field< 0, 16, nn::Bit16>   All;        // 全領域
};

const int SensorDataValueMax = 0xffff; // センサデータの上限値

// コマンド
struct Command
{
    typedef nn::util::BitPack8::Field< 0, 5, int>       AddressSpecialCommand;  // アドレスもしくはスペシャルコマンド
    typedef nn::util::BitPack8::Field< 5, 2, int>       Transaction;            // 処理モード
    typedef nn::util::BitPack8::Field< 7, 1, bool>      Cmd;                    // コマンド
    typedef nn::util::BitPack8::Field< 0, 8, nn::Bit8>  All;                    // 全領域
};

// CONTROL レジスタ
struct Control
{
    typedef nn::util::BitPack8::Field< 0, 1, bool>      Power;      // パワーUp・Down
    typedef nn::util::BitPack8::Field< 1, 1, bool>      AdcEn;      // 計測停止・開始
    typedef nn::util::BitPack8::Field< 2, 1, bool>      DataSel;    // 計測するデータを選択
    typedef nn::util::BitPack8::Field< 3, 1, bool>      OneTime;    // 連続・一回計測モード
    typedef nn::util::BitPack8::Field< 4, 1, bool>      AdcValid;   // 計測結果の更新有無
    typedef nn::util::BitPack8::Field< 5, 1, bool>      AdcIntr;    // 割り込みのアクティブ・インアクティブ
//    typedef nn::util::BitPack8::Field< 6, 2, int>   Res;        // 予約領域。
    typedef nn::util::BitPack8::Field< 0, 8, nn::Bit8>  All;        // 全領域
};

// INTERRUPT レジスタ
struct Interrupt
{
    typedef nn::util::BitPack8::Field< 0, 4, int>       Persist;    // 割り込みをする連続回数の設定
    typedef nn::util::BitPack8::Field< 4, 1, bool>      IntEn;      // 1: 割り込み使用
//    typedef nn::util::BitPack8::Field< 5, 1, bool>  Res;        // 予約領域。
    typedef nn::util::BitPack8::Field< 6, 1, bool>      IntStop;    // 1: 割り込みが発生すると計測を止める
//    typedef nn::util::BitPack8::Field< 4, 1, bool>  Res;        // 予約領域。
    typedef nn::util::BitPack8::Field< 0, 8, nn::Bit8>  All;        // 全領域
};

// PART_ID レジスタ
struct PartId
{
    typedef nn::util::BitPack8::Field< 0, 4, nn::Bit8>  RevId;      // Revision ID
    typedef nn::util::BitPack8::Field< 4, 4, nn::Bit8>  PartNum;    // パートナンバー [0111固定]
    typedef nn::util::BitPack8::Field< 0, 8, nn::Bit8>  All;        // 全領域
};
const nn::Bit8   PartNumber    = 0x7;   // パートナンバーの固定値

} // detail
} // bh1730fvc
} // nnd
