﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_BitTypes.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_BitPack.h>
#include <nn/TargetConfigs/build_Base.h>
#include <nn/i2c/i2c.h>
#include <nnd/bh1730fvc/bh1730fvc.h>

#include "bh1730fvc_Driver.h"
#include "bh1730fvc_Specification.h"
#include "bh1730fvc_BusConfig.h"

#include "bh1730fvc_Debug.h"

namespace nnd {
namespace bh1730fvc {
namespace detail {

// -----------------------
// 内部変数
// -----------------------
namespace {

static Gain             g_Gain                              = Gain::X1;                     // ゲインの設定 (初期値: X1)
static int              g_MeasurmentCycleCount              = ConvertCycleAndTime(0xDA);    // ITIME のサイクル数 (初期値: 0xDA)
static int              g_CycleCountBeforeManualMeasurment  = ConvertCycleAndTime(0xDA);    // ITIME のサイクル数 (初期値: 0xDA)
static MeasurementMode  g_MeasurmentMode                    = MeasurementMode::Standby;     // モード (初期値: 0xDA)

const float             InvalidLux  = -1.0f;
} // namespace


// -----------------------
// 内部関数
// -----------------------
namespace{

nn::Bit8 GenerateCommandRegister(nn::i2c::I2cSession i2cSession, nn::Bit8 data, bool isSpecialCommand) NN_NOEXCEPT
{
    NND_BH1730FVC_DETAIL_LOG("START\n");
    nn::util::BitPack8 cmd;
    cmd.SetAllBitOff(Command::All::Mask); // クリア

    cmd.SetBit(Command::Cmd::Pos, true);

    if(isSpecialCommand)
    {
        cmd.SetAllBitOn(Command::Transaction::Mask);
    }
    else
    {
        cmd.SetAllBitOff(Command::Transaction::Mask);

    }

    cmd.SetMaskedBits(Command::AddressSpecialCommand::Mask, data);
    NND_BH1730FVC_DETAIL_LOG("%3d(0x%02x): Generated command [data=%3d(0x%02x)]\n", cmd, cmd, data, data);

    return cmd.Get<Command::All>();
}

nn::Result ReadRegister(nn::Bit8* pOutDate, nn::i2c::I2cSession i2cSession, RegAddr addr, int readSize) NN_NOEXCEPT
{
    NND_BH1730FVC_DETAIL_LOG("START\n");
    uint8_t commandList[nn::i2c::CommandListLengthCountMax];

    nn::i2c::CommandListFormatter  commandListFormatter(commandList, sizeof(commandList));
    std::memset(commandList, 0, sizeof(commandList));

    nn::Bit8 cmd = GenerateCommandRegister(i2cSession, static_cast<nn::Bit8>(addr), false);
    commandListFormatter.EnqueueSendCommand( I2cTransStartStop, &cmd, sizeof(cmd) );
    commandListFormatter.EnqueueReceiveCommand( I2cTransStartStop, readSize );
    return CheckI2cError(nn::i2c::ExecuteCommandList(pOutDate, readSize, i2cSession, commandList, commandListFormatter.GetCurrentLength()));
}

nn::Result SendCommand(nn::i2c::I2cSession i2cSession, RegAddr addr, nn::Bit8 data) NN_NOEXCEPT
{
    NND_BH1730FVC_DETAIL_LOG("START\n");

    nn::Bit8 sendData[2] = {
            GenerateCommandRegister(i2cSession, static_cast<nn::Bit8>(addr), false),
            data,
    };

    NND_BH1730FVC_DETAIL_LOG("%3d(0x%02x): sendData[0]\n", sendData[0], sendData[0]);
    NND_BH1730FVC_DETAIL_LOG("%3d(0x%02x): sendData[1]\n", sendData[1], sendData[1]);
    NN_RESULT_DO(CheckI2cError(nn::i2c::Send(i2cSession, sendData, sizeof(sendData) / sizeof(sendData[0]), I2cTransStartStop)));
    NN_RESULT_SUCCESS;
}

nn::Result SendSpecialCommand(nn::i2c::I2cSession i2cSession, SpecialCommand spCmd) NN_NOEXCEPT
{
    NND_BH1730FVC_DETAIL_LOG("START\n");

    nn::Bit8 sendData = GenerateCommandRegister(i2cSession, static_cast<nn::Bit8>(spCmd), true);
    NN_RESULT_DO(CheckI2cError(nn::i2c::Send(i2cSession, &sendData, sizeof(sendData), I2cTransStartStop)));
    NN_RESULT_SUCCESS;
}

nn::Result ResetDevice(nn::i2c::I2cSession session) NN_NOEXCEPT
{
    NND_BH1730FVC_DETAIL_LOG("START\n");
    NN_RESULT_DO(SendSpecialCommand(session, SpecialCommand::ResetSystem));
    NN_RESULT_SUCCESS;
}

bool HasDevice(nn::i2c::I2cSession* pI2cSession) NN_NOEXCEPT
{
    NND_BH1730FVC_DETAIL_LOG("START\n");

    // パートID 部分を Read
    // 内容自体はチップリビジョンで変わる可能性がありそうなので、比較しない
    nn::Bit8 readReg;
    auto result = ReadRegister(&readReg, *pI2cSession, RegAddr::Id, RegSize);

    NND_BH1730FVC_DETAIL_LOG("ID register value = 0x%02x\n",readReg);

    return result.IsSuccess();
}

} // namespace



bool Initialize(nn::i2c::I2cSession* pOutI2cSession) NN_NOEXCEPT
{
    NND_BH1730FVC_DETAIL_LOG("START\n");

    // バスの初期化
    InitializeBus(pOutI2cSession);

    // デバイスの有無の確認
    if(!HasDevice(pOutI2cSession))
    {
        // バスの開放
        FinalizeBus(pOutI2cSession);
        return false;
    }

    // デバイスの初期化: 全レジスタの初期化とPowerDownへの遷移
    if(!ResetDevice(*pOutI2cSession).IsSuccess())
    {
        // バスの開放
        FinalizeBus(pOutI2cSession);
        return false;
    }

    return true;
}


void Finalize(nn::i2c::I2cSession* pI2cSession) NN_NOEXCEPT
{
    NND_BH1730FVC_DETAIL_LOG("START\n");

    // デバイスの初期化: 全レジスタの初期化とPowerDownへの遷移
    ResetDevice(*pI2cSession);

    // バスの開放
    FinalizeBus(pI2cSession);
}


nn::Result GetMeasurementMode(nn::i2c::I2cSession session, MeasurementMode* pOutMode) NN_NOEXCEPT
{
    NND_BH1730FVC_DETAIL_LOG("START\n");
    nn::Bit8 readCtrlReg;

    NN_RESULT_DO(ReadRegister(&readCtrlReg, session, RegAddr::Control, RegSize));
//    NND_BH1730FVC_DETAIL_LOG("0x%02x : readCtrlReg\n",readCtrlReg);

    int cycle;
    NN_RESULT_DO(GetIntegralCycle(session, &cycle));
//    NND_BH1730FVC_DETAIL_LOG("0x%02x : cycle\n",cycle);

    nn::util::BitPack8 data;
    data.Set<Control::All>(readCtrlReg);

    auto isGetMode = false;
    if (data.GetBit(Control::Power::Pos) == false && data.GetBit(Control::AdcEn::Pos) == false)
    {
        *pOutMode = MeasurementMode::Standby;
        NND_BH1730FVC_DETAIL_LOG("MeasurementMode::Standby (%d)\n",*pOutMode);
        isGetMode = true;
    }
    else if (data.GetBit(Control::Power::Pos) == true && data.GetBit(Control::AdcEn::Pos) == true)
    {
        if (cycle == 0)
        {
            *pOutMode = MeasurementMode::Manual;
            NND_BH1730FVC_DETAIL_LOG("MeasurementMode::Manual (%d)\n",*pOutMode);
            isGetMode = true;
        }
        else if (data.GetBit(Control::OneTime::Pos) == true)
        {
            *pOutMode = MeasurementMode::OneShot;
            NND_BH1730FVC_DETAIL_LOG("MeasurementMode::OneShot (%d)\n",*pOutMode);
            isGetMode = true;
        }
        else
        {
            *pOutMode = MeasurementMode::Continuous;
            NND_BH1730FVC_DETAIL_LOG("MeasurementMode::Continuous (%d)\n",*pOutMode);
            isGetMode = true;
        }
    }
    NN_SDK_REQUIRES(isGetMode, "Invalid state\n");

    // SetMeasurementMode() で直前にセットしたモードを確認するため、
    // 内部では g_MeasurmentMode を更新しない

    NN_RESULT_SUCCESS;
}



nn::Result SetMeasurementMode(nn::i2c::I2cSession session, MeasurementMode mode) NN_NOEXCEPT
{
    NND_BH1730FVC_DETAIL_LOG("START\n");

    // TODO: 計測の途中で中断した場合の挙動によっては Sensing などの中間状態を用意する？
    // TODO: ベンダに確認する

    nn::Result result;

    // 既に同じモードなら成功
    NN_RESULT_DO(GetMeasurementMode(session, &g_MeasurmentMode));
    if(mode == g_MeasurmentMode)
    {
        NND_BH1730FVC_DETAIL_LOG("Already set mode: %d\n", g_MeasurmentMode);
        NN_RESULT_SUCCESS;
    }

    // Standby 以外の状態への以降は許可しない
    if(g_MeasurmentMode != MeasurementMode::Standby && mode != MeasurementMode::Standby)
    {
        NND_BH1730FVC_DETAIL_LOG("ResultInvalidState!! : current=%d, input=%d\n", g_MeasurmentMode, mode);
        return ResultInvalidState();
    }

    // Manual -> Standby の操作時のみの追加処理
    if (g_MeasurmentMode == MeasurementMode::Manual && mode == MeasurementMode::Standby)
    {
//        NND_BH1730FVC_DETAIL_LOG("g_MeasurmentMode == MeasurementMode::Manual && mode == MeasurementMode::Standby\n");

        // スペシャルコマンドで計測停止
        NN_RESULT_DO(SendSpecialCommand(session, SpecialCommand::StopManualMeasurment));

        // タイミングレジスタを Manual 実行前の設定に戻す
//        NND_BH1730FVC_DETAIL_LOG("0x%02x : g_CycleCountBeforeManualMeasurment\n",g_CycleCountBeforeManualMeasurment);
        NN_RESULT_DO(SetIntegralCycle(session, g_CycleCountBeforeManualMeasurment));
    }

    nn::util::BitPack8 data;
    data.SetAllBitOff(Command::All::Mask); // クリア

    switch (mode)
    {
    case MeasurementMode::Continuous:
        {
            NND_BH1730FVC_DETAIL_LOG("MeasurementMode::Continuous\n");
            data.SetBit(Control::OneTime::Pos,  false); // 連続計測モード
            data.SetBit(Control::AdcEn::Pos,    true);  // 計測開始
            data.SetBit(Control::Power::Pos,    true);  // 電源オン
        }
        break;
    case MeasurementMode::OneShot:
        {
            NND_BH1730FVC_DETAIL_LOG("MeasurementMode::OneShot\n");
            data.SetBit(Control::OneTime::Pos,  true);  // 一回計測モード
            data.SetBit(Control::AdcEn::Pos,    true);  // 計測開始
            data.SetBit(Control::Power::Pos,    true);  // 電源オン
        }
        break;
    case MeasurementMode::Manual:
        {
            NND_BH1730FVC_DETAIL_LOG("MeasurementMode::Manual\n");
            // HW的にはどちらでも良いが、計測停止後に PowerDown 状態にするために一回計測モードにしておく
            // FIXME: のはずだが、OneShot を有効にしているとエラーになる
            data.SetBit(Control::OneTime::Pos,  false);  // 連続計測モード
//            data.SetBit(Control::OneTime::Pos,  true);  // 一回計測モード
            data.SetBit(Control::AdcEn::Pos,    true);  // 計測開始
            data.SetBit(Control::Power::Pos,    true);  // 電源オン
        }
        break;
    case MeasurementMode::Standby:
        {
            NND_BH1730FVC_DETAIL_LOG("MeasurementMode::Standby\n");
//            data.SetBit(Control::OneTime::Pos,  true);  // どちらでも良いので、何もしない
            data.SetBit(Control::AdcEn::Pos,    false); // 計測停止
            data.SetBit(Control::Power::Pos,    false); // 電源オフ
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
    NN_RESULT_DO(SendCommand(session, RegAddr::Control, data.Get<Control::All>()));

    // Manual モード時のみの追加処理
    if (mode == MeasurementMode::Manual)
    {
//        NND_BH1730FVC_DETAIL_LOG("mode == MeasurementMode::Manual\n");

        // 現在のサイクル数を保存しておく
        NN_RESULT_DO(GetIntegralCycle(session, &g_CycleCountBeforeManualMeasurment));

        // タイミングレジスタで 積分時間マニュアルモードにセット
        NN_RESULT_DO(SetIntegralCycle(session, 0));

        // スペシャルコマンドで計測開始
        NN_RESULT_DO(SendSpecialCommand(session, SpecialCommand::StartManualMeasurment));
    }

    // ライブラリのモード情報を更新
    g_MeasurmentMode = mode;
    NN_RESULT_SUCCESS;
}


nn::Result GetMeasurementValue(nn::i2c::I2cSession session, MeasurementValue* pOutValue) NN_NOEXCEPT
{
    NND_BH1730FVC_DETAIL_LOG("START\n");
    const int ReadDataSize = 4;

    nn::Bit8 readData[ReadDataSize];

    NN_RESULT_DO(ReadRegister(readData, session, RegAddr::Data0LowByte, RegSize*ReadDataSize));

    // ２つの値から照度値を計算
    nn::util::BitPack16 visible;
    nn::util::BitPack16 infrared;

    visible.SetAllBitOff(SensorData::All::Mask); // クリア
    visible.Set<SensorData::LowByte>(readData[0]);
    visible.Set<SensorData::HighByte>(readData[1]);

    infrared.SetAllBitOff(SensorData::All::Mask); // クリア
    infrared.Set<SensorData::LowByte>(readData[2]);
    infrared.Set<SensorData::HighByte>(readData[3]);

    NND_BH1730FVC_DETAIL_LOG("readData: %2x %2x %2x %2x\n",readData[0],readData[1],readData[2],readData[3]);
    pOutValue->visible = visible.Get<SensorData::All>();
    pOutValue->infrared = infrared.Get<SensorData::All>();
    bool isOverflown = false;
    pOutValue->lux = detail::CalculateLux(pOutValue->visible, pOutValue->infrared, g_Gain, static_cast<nn::Bit8>(g_MeasurmentCycleCount), &isOverflown);
    pOutValue->isOverflown = isOverflown;
    NN_RESULT_SUCCESS;
}

float CalculateLux(int visible, int infrared, Gain gain, nn::Bit8 cycle, bool* pOutOverflown) NN_NOEXCEPT
{
    NND_BH1730FVC_DETAIL_LOG("START\n");

    // Manual モードの場合は、無効値を返す
    if (cycle == 0)
    {
        return InvalidLux;
    }
    else
    {
        return CalculateLuxImpl(visible, infrared, gain, cycle, pOutOverflown);
    }
}


nn::Result IsLuxUpdated(nn::i2c::I2cSession session, bool* pOutUpdated) NN_NOEXCEPT
{
    NND_BH1730FVC_DETAIL_LOG("START\n");

    nn::Bit8 readReg;
    NN_RESULT_DO(ReadRegister(&readReg, session, RegAddr::Control, RegSize));

    nn::util::BitPack8 data;
    data.Set<Control::All>(readReg);

    *pOutUpdated = data.GetBit(Control::AdcValid::Pos);

    NN_RESULT_SUCCESS;
}


nn::Result ReadRevisionId(nn::i2c::I2cSession session, nn::Bit8* pOutId) NN_NOEXCEPT
{
    NND_BH1730FVC_DETAIL_LOG("START\n");

    nn::Bit8 readReg;
    NN_RESULT_DO(ReadRegister(&readReg, session, RegAddr::Id, RegSize));

    // パートID 部分が正しいか確認
    nn::util::BitPack8 data;
    data.Set<PartId::All>(readReg);
    NN_SDK_REQUIRES((data.Get<PartId::PartNum>() == PartNumber), "Invalid part number. : %d",data.Get<PartId::PartNum>());
    *pOutId = data.Get<PartId::All>();
    NN_RESULT_SUCCESS;
}


nn::Result GetIntegralCycle(nn::i2c::I2cSession session, int* pOutCycle) NN_NOEXCEPT
{
    NND_BH1730FVC_DETAIL_LOG("START\n");

    nn::Bit8 readReg;
    NN_RESULT_DO(ReadRegister(&readReg, session, RegAddr::Timing, RegSize));
//    NND_BH1730FVC_DETAIL_LOG("%3d(0x%02x) : readReg\n",readReg, readReg);

    // 積分時間マニュアルモードの場合
    if (readReg != 0)
    {
        *pOutCycle = static_cast<int>(ConvertCycleAndTime(readReg));
    }
    else
    {
        *pOutCycle = readReg;
    }
//    NND_BH1730FVC_DETAIL_LOG("%3d(0x%02x) : pOutCycle\n",*pOutCycle, *pOutCycle);

    g_MeasurmentCycleCount = *pOutCycle;

    NN_RESULT_SUCCESS;
}


nn::Result SetIntegralCycle(nn::i2c::I2cSession session, int cycle) NN_NOEXCEPT
{
    NND_BH1730FVC_DETAIL_LOG("START\n");

    // 内部呼び出しの場合は 0 (マニュアルモード) を許す
    NN_SDK_ASSERT_RANGE(cycle, 0, IntegralCycleMax + 1);

//    NND_BH1730FVC_DETAIL_LOG("%3d(0x%02x) : cycle\n", cycle, cycle);
    nn::Bit8 sendData;
    if (cycle != 0)
    {
        sendData = ConvertCycleAndTime(static_cast<nn::Bit8>(cycle));
    }
    else
    {
        sendData = cycle;
    }
//    NND_BH1730FVC_DETAIL_LOG("%3d(0x%02x) : sendData\n", sendData, sendData);
    NN_RESULT_DO(SendCommand(session, RegAddr::Timing, sendData));

    g_MeasurmentCycleCount = cycle;

    NN_RESULT_SUCCESS;
}


nn::Result GetGain(nn::i2c::I2cSession session, Gain* pOutGain) NN_NOEXCEPT
{
    NND_BH1730FVC_DETAIL_LOG("START\n");

    nn::Bit8 readReg;
    NN_RESULT_DO(ReadRegister(&readReg, session, RegAddr::Gain, RegSize));
    bool isMatched = false;
    for (auto gainTable : GainNameValueCombinationList)
    {
        if (static_cast<nn::Bit8>(readReg) == static_cast<nn::Bit8>(gainTable.name))
        {
            *pOutGain = gainTable.name;
            g_Gain = *pOutGain;
            isMatched = true;
            break;
        }
    }
    NN_SDK_REQUIRES(isMatched, "GetGain() failed.");

    NN_RESULT_SUCCESS;
}


nn::Result SetGain(nn::i2c::I2cSession session, Gain gain) NN_NOEXCEPT
{
    NND_BH1730FVC_DETAIL_LOG("START\n");

    nn::Bit8 writeReg = 0;
    bool isMatched = false;
    for (auto gainTable : GainNameValueCombinationList)
    {
        if (gainTable.name == gain)
        {
            writeReg = static_cast<nn::Bit8>(gain);
            isMatched = true;
        }
    }
    NN_SDK_REQUIRES(isMatched, "SetGain() failed.");

    NN_RESULT_DO(SendCommand(session, RegAddr::Gain, writeReg));
    g_Gain = gain;

    NN_RESULT_SUCCESS;
}


} // detail
} // bh1730fvc
} // nnd
