﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       lsm6ds3 ライブラリの API リストです。
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_TimeSpan.h>

#include <nnd/lsm6ds3/lsm6ds3_Types.h>
#include <nnd/lsm6ds3/lsm6ds3_Result.h>

namespace nnd {
namespace lsm6ds3 {

//! @name ライブラリ全体に関連した処理
//! @{

/**
 * @brief       lsm6ds3 ドライバライブラリを初期化します。
 *
 * @return      処理の結果を返します。
 * @retval      ResultSuccess                        成功しました。
 * @retval      nnd::lsm6ds3::ResultNotImplemented   デバイスが実装されていません。
 *
 * @details     lsm6ds3 のドライバライブラリを初期化します。
 *              依存するライブラリの初期化処理とリソースの確保が行われます。
 *              SPI ドライバの割り込み監視スレッドが以下の設定で立ち上がります。
 *
 *              - スレッド優先度は nn::os::DefaultThreadPriority
 *              - コア番号は自スレッドが動作しているコア
 *
 *              デバイスの実装有無の確認のため、SPI のトランザクションが発生します。
 *              各センサーは以下の設定になります。
 *              加速度センサー : FSR = ±2[G], ODR = 1660[Hz]
 *              ジャイロセンサー : FSR = ±245[dps], ODR = 208[Hz]
 *              6軸センサーのノイズ低減機能はデフォルトで OFF 状態です。
 *
 *              なお、デバイスが実装されていない場合は、確保されたリソースは全て解放され、
 *              本関数呼出し前の未初期化状態に戻ります。
 *
 * @post        lsm6ds3 ライブラリは初期化済の状態になり、デバイス識別子がキャッシュされます。
 *
 */
nn::Result Initialize() NN_NOEXCEPT;

/**
 * @brief       lsm6ds3 ドライバライブラリを初期化します。
 *
 * @param[in]   priority          スレッドの優先度
 * @param[in]   idealCoreNumber   スレッドを作成するコア番号
 *
 * @return      処理の結果を返します。
 * @retval      ResultSuccess                        成功しました。
 * @retval      nnd::lsm6ds3::ResultNotImplemented   デバイスが実装されていません。
 *
 *  @pre
 *              - nn::os::HighestThreadPriority <= priority <= nn::os::LowestThreadPriority
 *              - idealCoreNumber がターゲット動作環境毎に定義された有効な値である
 *
 * @details     ライブラリ内部で立てる SPI ドライバの割り込み監視スレッドの優先度、動作コアを指定して
 *              lsm6ds3 のドライバライブラリを初期化します。
 *              評価目的での利用に限定されます。
 *              依存するライブラリの初期化処理とリソースの確保が行われます。
 *              デバイスの実装有無の確認のため、SPI のトランザクションが発生します。
 *              各センサーは以下の設定になります。
 *              加速度センサー : FSR = ±2[G], ODR = 1660[Hz]
 *              ジャイロセンサー : FSR = ±245[dps], ODR = 208[Hz]
 *              6軸センサーのノイズ低減機能はデフォルトで OFF 状態です。
 *
 *              なお、デバイスが実装されていない場合は、確保されたリソースは全て解放され、
 *              本関数呼出し前の未初期化状態に戻ります。
 *
 * @post        lsm6ds3 ライブラリは初期化済の状態になり、デバイス識別子がキャッシュされます。
 *
 */
nn::Result Initialize(int priority, int idealCoreNumber) NN_NOEXCEPT;

/**
 * @brief       lsm6ds3 ドライバライブラリを終了します。
 *
 * @details     lsm6ds3 ドライバライブラリを終了します。
 *              関連するライブラリの終了処理とリソースの解放が行われます。
 *              lsm6ds3 が初期化済でない場合は何もしません。
 *
 * @post        lsm6ds3 ライブラリが初期化前の状態になります。
 */
void Finalize() NN_NOEXCEPT;

/**
 * @brief       lsm6ds3 をリセットします。
 *
 * @details     lsm6ds3 をリセットします。
 *              TODO: リセットの定義の記述
 *              リセット復帰後は以下の状態になります。
 *              TODO: 復帰後の状態の記述
 * @pre         - lsm6ds3 ライブラリは初期化済である必要があります。
 */
void ResetDevice() NN_NOEXCEPT;

/**
 * @brief       lsm6ds3 の入力状態を取得します。
 *
 * @details     lsm6ds3 の加速度センサー/ジャイロセンサーの入力状態を取得します
 *              入力状態の取得に失敗した場合、バッファ上の値は維持されます。
 *              lsm6ds3 のデバイス自体が未初期化の場合、本関数呼出し時に暗黙的に
 *              初期化処理が行われます。
 *              各センサーの電源が入っていない場合は、ゼロ埋めされた入力状態が取得されます。
 *
 * @param[out]  pOutState                       入力状態取得用のバッファ
 *
 * @pre         - lsm6ds3 ライブラリは初期化済である必要があります。
 *              - pOutState != nullptr
 *
 */
void GetState(SixAxisSensorCountState* pOutState) NN_NOEXCEPT;

/**
* @brief       lsm6ds3 の入力状態を定期的に取得する処理を開始します。
*
* @details     lsm6ds3 の加速度センサー/ジャイロセンサーの入力状態を定期的に取得する処理を開始します。
*              システムコアで定期的に取得し、渡されたバッファへ直接書き出すため、連続して GetState() を呼ぶよりも処理負荷が軽くなります。
*              入力状態の取得に失敗した場合、バッファ上の値は維持されます。
*              lsm6ds3 のデバイス自体が未初期化の場合、本関数呼出し時に暗黙的に
*              初期化処理が行われます。
*              各センサーの電源が入っていない場合は、ゼロ埋めされた入力状態が取得されます。
*
* @param[in]   interval                       入力状態を取得する頻度
* @param[in]   receiveBuffer                  定期取得用バッファ
* @param[in]   interval                       定期取得用バッファイサイズ
*
* @pre         - lsm6ds3 ライブラリは初期化済である必要があります。
*              - pOutState != nullptr
*
*/
void StartPeriodicReceiveMode(nn::TimeSpan interval, char* receiveBuffer, size_t receiveBufferLength) NN_NOEXCEPT;

/**
* @brief       lsm6ds3 の入力状態を定期的に取得する処理を停止します。
*
* @details     lsm6ds3 の加速度センサー/ジャイロセンサーの入力状態を定期的に取得する処理を停止します。
*
* @pre         - lsm6ds3 ライブラリは初期化済である必要があります。
*              - pOutState != nullptr
*
*/
void StopPeriodicReceiveMode() NN_NOEXCEPT;

/**
* @brief       定期取得している共有メモリから lsm6ds3 の入力状態を取得します。
*
* @details     定期取得している共有メモリから lsm6ds の入力状態を取得します。
*              各センサーの電源が入っていない場合は、ゼロ埋めされた入力状態が取得されます。
*
* @param[out]  pOutState                       入力状態取得用のバッファ
* @param[in]   count                           取得するサンプル数
*
* @return      取得したサンプル数
*
* @pre         - lsm6ds3 ライブラリは初期化済である必要があります。
*              - pOutState != nullptr
*
*/
int GetStatesForPeriodicReceiveMode(SixAxisSensorCountState* pOutState, int count) NN_NOEXCEPT;

/**
 * @brief       lsm6ds3 のノイズ低減フィルター機能の状態を取得します。
 *
 * @details     lsm6ds3 のノイズ低減フィルター機能の状態を取得します。
 *
 * @param[in]   pOutEnable                   フィルター機能の状態
 *
 * @pre         - lsm6ds3 ライブラリは初期化済である必要があります。
 *              - pOutEnable != nullptr
 */
void GetNoiseReductionFilter(bool* pOutEnable) NN_NOEXCEPT;

/**
 * @brief       lsm6ds3 のノイズ低減フィルター機能を設定します。
 *
 * @details     lsm6ds3 のノイズ低減フィルター機能を設定します。
 *
 *              ノイズ低減フィルター使用時:
 *              - アンチエイリアシングフィルタのカットオフ周波数 = 400Hz
 *              - ディジタル信号にローパスフィルタを適用
 *
 *              ノイズ低減フィルター不使用時:
 *              - アンチエイリアシングフィルタのカットオフ周波数 = 100Hz
 *
 * @param[in]   enable                   フィルター機能の設定
 *
 * @pre         - lsm6ds3 ライブラリは初期化済である必要があります。
 *
 */
void SetNoiseReductionFilter(bool enable) NN_NOEXCEPT;

//! @}


//! @name 加速度センサー固有の処理
//! @{

/**
 * @brief       加速度センサーの電源を入れます。
 *
 * @details     加速度センサーの動作設定は以下の状態になります。
 *              TODO: アプリ側のユースケースをヒアリングして、尤もらしい設定を初期状態とする。
 * @pre         - lsm6ds3 ライブラリは初期化済である必要があります。
 */
void StartAccelerometer() NN_NOEXCEPT;

/**
 * @brief       加速度センサーの電源を切ります。
 *
 * @details     加速度センサーの動作設定は以下の状態になります。
 *
 * @pre         - lsm6ds3 ライブラリは初期化済である必要があります。
 */
void StopAccelerometer() NN_NOEXCEPT;

 /**
 * @brief      加速度センサーの現在のFSRを取得します。
 *
 * @details    加速度センサーの現在のFSRを取得します。
 *
 * @param[out] pOutAccelerometerFsr          取得された加速度センサーのFSR
 *
 * @pre         - lsm6ds3 ライブラリは初期化済である必要があります。
 *              - pOutAccelerometerFsr != nullptr
 */
void GetAccelerometerFsr(AccelerometerFsr* pOutAccelerometerFsr) NN_NOEXCEPT;

 /**
 * @brief      加速度センサーの現在のFSRを設定します。
 *
 * @details    加速度センサーの現在のFSRを設定します
 *             FSRを変更すると感度が変わるため、GetAccelerometerCalibrationState()で再取得をしてください
 *
 * @param[out] accelerometerFsr             設定する加速度センサーのFSR
 *
 * @pre        - lsm6ds3 ライブラリは初期化済である必要があります。
 */
void SetAccelerometerFsr(AccelerometerFsr accelerometerFsr) NN_NOEXCEPT;

 /**
 * @brief      加速度センサーの現在のODRを取得します。
 *
 * @details    加速度センサーの現在のODRを取得します。
 *
 * @param[out] pOutAccelerometerOdr          取得された加速度センサーのODR
 *
 * @pre         - lsm6ds3 ライブラリは初期化済である必要があります。
 *              - pOutAccelerometerOdr != nullptr
 */
void GetAccelerometerOdr(AccelerometerOdr* pOutAccelerometerOdr) NN_NOEXCEPT;

 /**
 * @brief      加速度センサーの現在のODRを設定します。
 *
 * @details    加速度センサーの現在のODRを設定します
 *
 * @param[out] accelerometerOdr             設定する加速度センサーのODR
 *
 * @pre        - lsm6ds3 ライブラリは初期化済である必要があります。
 */
void SetAccelerometerOdr(AccelerometerOdr accelerometerOdr) NN_NOEXCEPT;

//! @}


//! @name ジャイロセンサー固有の処理
//! @{

/**
 * @brief       ジャイロセンサーの電源を入れます。
 *
 * @details     ジャイロセンサーの動作設定は以下の状態になります。
 *              TODO: アプリ側のユースケースをヒアリングして、尤もらしい設定を初期状態とする。
 * @pre         - lsm6ds3 ライブラリは初期化済である必要があります。
 */
void StartGyroscope() NN_NOEXCEPT;

/**
 * @brief       ジャイロセンサーの電源を切ります。
 *
 * @details     ジャイロセンサーの動作設定は以下の状態になります。
 *
 * @pre         - lsm6ds3 ライブラリは初期化済である必要があります。
 */
void StopGyroscope() NN_NOEXCEPT;

 /**
 * @brief      ジャイロセンサーの現在のFSRを取得します。
 *
 * @details    ジャイロセンサーの現在のFSRを取得します
 *
 * @param[out] pOutGyroscopeFsr          取得されたジャイロセンサーのFSR
 *
 * @pre         - lsm6ds3 ライブラリは初期化済である必要があります。
 *              - pOutGyroscopeFsr != nullptr
 */
void GetGyroscopeFsr(GyroscopeFsr* pOutGyroscopeFsr) NN_NOEXCEPT;

 /**
 * @brief      ジャイロセンサーの現在のFSRを設定します。
 *
 * @details    ジャイロセンサーの現在のFSRを設定します
 *             FSRを変更すると感度が変わるため、GetGyroscopeCalibrationState()で再取得をしてください
 *
 * @param[out] gyroscopeFsr             設定するジャイロセンサーのFSR
 *
 * @pre        - lsm6ds3 ライブラリは初期化済である必要があります。
 */
void SetGyroscopeFsr(GyroscopeFsr gyroscopeFsr) NN_NOEXCEPT;

 /**
 * @brief      ジャイロセンサーの現在のODRを取得します。
 *
 * @details    ジャイロセンサーの現在のODRを取得します。
 *
 * @param[out] pOutGyroscopeOdr          取得されたジャイロセンサーのODR
 *
 * @pre         - lsm6ds3 ライブラリは初期化済である必要があります。
 *              - pOutGyroscopeOdr != nullptr
 */
void GetGyroscopeOdr(GyroscopeOdr* pOutGyroscopeOdr) NN_NOEXCEPT;

 /**
 * @brief      ジャイロセンサーの現在のODRを設定します。
 *
 * @details    ジャイロセンサーの現在のODRを設定します
 *
 * @param[out] gyroscopeOdr             設定するジャイロセンサーのODR
 *
 * @pre        - lsm6ds3 ライブラリは初期化済である必要があります。
 */
void SetGyroscopeOdr(GyroscopeOdr gyroscopeOdr) NN_NOEXCEPT;

//! @}

} // lsm6ds3
} // nnd
