﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       ftm ライブラリの検査工程開発者向けのデータ型や定数の定義です。
 */

#pragma once

namespace nnd {
namespace ftm {

const int FtmForceChannelCountMax = 28;   //!< FTM シリーズがサポートする Force チャンネルの最大ノード数を表します。
const int FtmSenseChannelCountMax = 32;   //!< FTM シリーズがサポートする Sense チャンネルの最大ノード数を表します。

/**
 * @brief       ftm から取得できる相互補償データに関する情報をまとめた構造体です。
 */
struct MutualCompensationData
{
    char forceChannelCount;               //!< Force チャンネルのノード数
    char senseChannelCount;               //!< Sense チャンネルのノード数
    char cx1;                             //!< Mutual Cx1 データ
    char cx2[FtmForceChannelCountMax][FtmSenseChannelCountMax];    //!< Mutual Cx2 データ
};

/**
 * @brief       ftm から取得できる自己補償データに関する情報をまとめた構造体です。
 */
struct SelfCompensationData
{
    char forceChannelCount;              //!< Force チャンネルのノード数
    char senseChannelCount;              //!< Sense チャンネルのノード数
    char forceIx1;                       //!< Self Force Ix1 データ
    char senseIx1;                       //!< Self Sense Ix1 データ
    char forceCx1;                       //!< Self Force Cx1 データ
    char senseCx1;                       //!< Self Sense Cx1 データ
    char ix2[FtmForceChannelCountMax + FtmSenseChannelCountMax]; //!< Self Ix2 データ
    char cx2[FtmForceChannelCountMax + FtmSenseChannelCountMax]; //!< Self Cx2 データ
};

/**
 * @brief       ftm から取得できる生の相互データに関する情報をまとめた構造体です。
 */
struct MutualRawData
{
    char forceChannelCount;               //!< Force チャンネルのノード数
    char senseChannelCount;               //!< Sense チャンネルのノード数
    int  data[FtmForceChannelCountMax][FtmSenseChannelCountMax]; //!< Mutual Raw データ
};

/**
 * @brief       ftm から取得できる生の自己データに関する情報をまとめた構造体です。
 */
struct SelfRawData
{
    char forceChannelCount;                  //!< Force チャンネルのノード数
    char senseChannelCount;                  //!< Sense チャンネルのノード数
    int  forceData[FtmForceChannelCountMax]; //!< Force チャンネルの Raw データ
    int  senseData[FtmSenseChannelCountMax]; //!< Sense チャンネルの Raw データ
};

/**
 * @brief       ITO テストで検出可能なエラーの種類です。
 */
enum class ItoErrorType : uint8_t
{
    NoError,                        //!< エラーは発見されませんでした
    PanelForceOpen,                 //!< パネルの Force チャンネルがオープンになっています。
    PanelSenseOpen,                 //!< パネルの Sense チャンネルがオープンになっています。
    ForceGroundShort,               //!< Force チャンネルが GND とショートしています。
    SenseGroundShort,               //!< Sense チャンネルが GND とショートしています。
    ForceVcmShort,                  //!< Force チャンネルが電源とショートしています。
    SenseVcmShort,                  //!< Sense チャンネルが電源とショートしています。
    ForceForceShort,                //!< Force チャンネル同士がショートしています。
    SenseSenseShort,                //!< Sense チャンネル同士がショートしています。
    FpcForceOpen,                   //!< FPC の Force チャンネルがオープンになっています。
    FpcSenseOpen,                   //!< FPC の Sense チャンネルがオープンになっています。
    ItoScan,                        //!< ITO スキャンに失敗しました。
    MaxErrorReached,                //!< エラー検出できる上限に達しました。
    Undefined,                      //!< 未定義
};

/**
 * @brief       ftm から取得できる ITO テスト結果のレポートです。
 */
struct ItoEventReport
{
    ItoErrorType type;
    int          number;
};

} // ftm
} // nnd
