﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

/**
 * @file
 * @brief       bh1730fvc ライブラリの初期化と終了処理の API リストです。
 */

#include <nn/nn_Common.h>
#include <nn/os.h>

#include "bh1730fvc_Types.h"

namespace nnd {
namespace bh1730fvc {

//! @name 初期化と終了処理
//! @{

/**
 * @brief       ドライバライブラリを初期化します。
 *
 * @post        Standby 状態に遷移します。
 *
 * @return      処理の結果を返します。
 *
 * @details     デバイスの接続確認、初期化、及び関連するライブラリの初期化処理が行われます。@n
 *              デバイスとの接続確認や初期化ができない場合、false を返します。
 */
bool Initialize() NN_NOEXCEPT;


/**
 * @brief       ドライバライブラリの終了処理をします。
 *
 * @pre         初期化済み状態である必要があります。
 *
 * @post        未初期化状態に遷移します。
 *
 * @details     関連するライブラリの終了処理、デバイスの停止処理が行われます。
 */
void Finalize() NN_NOEXCEPT;

//! @}



//! @name 照度計測の有効化と読み取り専用情報の取得
//! @{

/**
 * @brief       現在の動作モードを取得します。
 *
 * @param[out]  pOutMode    動作モード
 *
 * @return      処理の結果を返します。
 * @retval      nn::ResultSuccess
 *                  成功しました。
 * @retval      nn::i2c::ResultBusBusy
 *                  I2C に関するエラー(参考：@ref nn::i2c)が発生しました。
 *                  処理が完了していないため、やり直すためには再度関数を呼び直す必要があります。
 *
 * @pre
 *      - 初期化済み状態である必要があります。
 *      - pOutMode != nullptr
 */
nn::Result GetMeasurementMode(MeasurementMode* pOutMode) NN_NOEXCEPT;


/**
 * @brief       指定する動作モードに設定します。
 *
 * @param[in]   mode    開始する動作モード
 *
 * @return      処理の結果を返します。
 * @retval      nn::ResultSuccess
 *                     指定する動作モードへの遷移に成功しました。
 * @retval      nn::ResultInvalidState
 *                     不正なステートであるため、処理を実行出来ませんでした。
 *                     設定したいモードに遷移するための前提状態を満たしていない可能性があります。
 *
 * @pre         初期化済み状態である必要があります。 @n
 *              また、本関数により各動作モードに遷移させるためには、
 *              本関数を実行する直前の状態が以下に示す状態である必要があります。
 *              @li OneShot に遷移可能な状態: Standby 状態
 *              @li Continuous に遷移可能な状態: Standby 状態
 *              @li Manual に遷移可能な状態: Standby 状態
 *              @li Standby に遷移可能な状態: Continuous 状態、Manual 状態
 *
 * @post        設定する動作モードに遷移します。
 */
nn::Result SetMeasurementMode(MeasurementMode mode) NN_NOEXCEPT;


/**
 * @brief       センサーによる計測結果を取得します。
 *
 * @param[out]  pOutValue   計測結果
 *                  lux         照度値 [lux]
 *                  visible     可視光センサーの計測値
 *                  infrared    赤外光センサーの計測値
 *
 * @return      処理の結果を返します。
 * @retval      nn::ResultSuccess
 *                  成功しました。
 * @retval      nn::i2c::ResultBusBusy
 *                  I2C に関するエラー(参考：@ref nn::i2c)が発生しました。
 *                  処理が完了していないため、やり直すためには再度関数を呼び直す必要があります。
 *
 * @pre
 *      - 初期化済み状態である必要があります。
 *      - pOutValue != nullptr
 *
 * @details     デバイスから得られる可視光センサーデータの生値(MeasurmentValue.visible)と
 *              赤外光センサーデータの生値(MeasurmentValue.infrared)と、
 *              生値と動作設定から算出した照度値(MeasurmentValue.lux)を取得します。@n
 *              照度値は、 @ref nnd::bh1730fvc::CalcurateLux() に生値と動作設定を入力した結果と
 *              同じです。@n
 *              Manual モード実行後の照度値の算出は、未サポートのため無効な値を返します。
 *              照度値の取得には @ref nnd::bh1730fvc::CalculateLux() を使用してください。
 */
nn::Result GetMeasurementValue(MeasurementValue* pOutValue) NN_NOEXCEPT;


/**
 * @brief       センサーデータの生値と計測設定から照度値を算出します。
 *
 * @param[in]   visible     可視光センサーデータの生値
 * @param[in]   infrared    赤外光センサーデータの生値
 * @param[in]   gain        ゲイン
 * @param[in]   cycle       サイクル数 [1～255]
 *
 * @pre         初期化済み状態である必要があります。
 *
 * @return      照度値 [lux]
 */
float CalculateLux(int visible, int infrared, Gain gain, uint8_t cycle) NN_NOEXCEPT;


/**
 * @brief       計測値が最後に取得した状態から更新されているかどうかを取得します。
 *
 * @param[out]  pOutUpdated     計測結果が更新されたかどうか
 *
 * @return      処理の結果を返します。
 * @retval      nn::ResultSuccess
 *                  成功しました。
 * @retval      nn::i2c::ResultBusBusy
 *                  I2C に関するエラー(参考：@ref nn::i2c)が発生しました。
 *                  処理が完了していないため、やり直すためには再度関数を呼び直す必要があります。
 *
 * @pre
 *      - 初期化済み状態である必要があります。
 *      - pOutUpdated != nullptr
 *
 * @details     @ref nnd::bh1730fvc::GetMeasurmentValue() により計測値を最後に取得した状態から
 *              計測結果が更新されているかどうかを取得します。
 */
nn::Result IsLuxUpdated(bool* pOutUpdated) NN_NOEXCEPT;


/**
 * @brief       リビジョンIDを読み取ります。
 *
 * @param[out]  pOutStatus  ステータス
 *
 * @return      処理の結果を返します。
 * @retval      nn::ResultSuccess
 *                  成功しました。
 * @retval      nn::i2c::ResultBusBusy
 *                  I2C に関するエラー(参考：@ref nn::i2c)が発生しました。
 *                  処理が完了していないため、やり直すためには再度関数を呼び直す必要があります。
 *
 * @pre
 *      - 初期化済み状態である必要があります。
 *      - pOutId != nullptr
 */
nn::Result ReadRevisionId(nn::Bit8* pOutId) NN_NOEXCEPT;

//! @}



//! @name   計測に関する動作設定
//! @{

/**
 * @brief       センシング結果を積分するサイクルを取得します。
 *
 * @param[out]  pOutCycle   サイクル数 [1～255]
 *
 * @return      処理の結果を返します。
 * @retval      nn::ResultSuccess
 *                  成功しました。
 * @retval      nn::i2c::ResultBusBusy
 *                  I2C に関するエラー(参考：@ref nn::i2c)が発生しました。
 *                  処理が完了していないため、やり直すためには再度関数を呼び直す必要があります。
 *
 * @pre
 *      - 初期化済み状態である必要があります。
 *      - pOutCycle != nullptr
 *
 * @details     センシング結果を積分するサイクルを取得します。 @n
 *              1 cycle は 2.7 ms です。
 */
nn::Result GetIntegralCycle(int* pOutCycle) NN_NOEXCEPT;


/**
 * @brief       センシング結果を積分するサイクルを設定します。
 *
 * @param[in]   cycle    サイクル数 [1～255]
 *
 * @return      処理の結果を返します。
 * @retval      ResultSuccess
 *                  成功しました。
 * @retval      nn::i2c::ResultBusBusy
 *                  I2C に関するエラー(参考：@ref nn::i2c)が発生しました。
 *                  処理が完了していないため、やり直すためには再度関数を呼び直す必要があります。
 *
 * @pre         Standby 状態である必要があります。
 *
 * @details     センシング結果を積分するサイクルを設定します。 @n
 *              1 cycle は 2.7 ms です。
 */
nn::Result SetIntegralCycle(int cycle) NN_NOEXCEPT;


/**
 * @brief       ゲインを取得します。
 *
 * @param[out]  pOutGain    ゲイン
 *
 * @return      処理の結果を返します。
 * @retval      nn::ResultSuccess
 *                  成功しました。
 * @retval      nn::i2c::ResultBusBusy
 *                  I2C に関するエラー(参考：@ref nn::i2c)が発生しました。
 *                  処理が完了していないため、やり直すためには再度関数を呼び直す必要があります。
 *
 * @pre
 *      - 初期化済み状態である必要があります。
 *      - pOutGain != nullptr
 */
nn::Result GetGain(Gain* pOutGain) NN_NOEXCEPT;


/**
 * @brief       ゲインを設定します。
 *
 * @param[in]   gain    ゲイン
 *
 * @return      処理の結果を返します。
 * @retval      nn::ResultSuccess
 *                  成功しました。
 * @retval      nn::i2c::ResultBusBusy
 *                  I2C に関するエラー(参考：@ref nn::i2c)が発生しました。
 *                  処理が完了していないため、やり直すためには再度関数を呼び直す必要があります。
 *
 * @pre         Standby 状態である必要があります。
 */
nn::Result SetGain(Gain gain) NN_NOEXCEPT;

//! @}


} // bh1730fvc
} // nnd
