﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       ペアリングに関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/bluetooth/bluetooth_AddressTypes.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/xcd/xcd_Device.h>
#include <nn/xcd/xcd_DeviceState.h>

namespace nn { namespace xcd {

const int RegisteredDeviceCountMax = 10;   //!< デバイス情報として登録できる最大数

const int ClassOfDeviceLength = 3;   //!< ClassOfDevice の長さ

/**
 * @brief       ClassOfDevice
 */
struct ClassOfDevice
{
    uint8_t cod[ClassOfDeviceLength];
};

const int LinkkeyLength = 16;   //!< LinkKey の長さ

/**
 * @brief       LinkKey
 */
struct LinkKey
{
    uint8_t key[LinkkeyLength];
};

static const int BdNameLength = 20;   //!< Bluetooth Device Name の長さ

/**
 * @brief       Bluetooth Device Name
 */
struct BdName
{
    uint8_t name[BdNameLength];  //NULL終端
};

/**
 * @brief       登録されているデバイスの情報を表す構造体です
 */
struct BluetoothHidDeviceInfo
{
    uint16_t vid;
    uint16_t pid;
};

/**
 * @brief       Bluetooth のデバイス情報
 */
struct BluetoothDeviceInfo
{
    ::nn::bluetooth::Address address;              //!< Bluetooth Address 6 バイト
    NN_PADDING2;
    LinkKey linkKey;                               //!< LinkKey 16 バイト
    BdName  bdName;                                //!< Bluetooth Device Name 20 バイト
    ClassOfDevice classOfDevice;                   //!< Class Of Device 3 バイト
    uint8_t featureSet;                            //!< FeatureSet 1 バイト
    BluetoothHidDeviceInfo hid;                    //!< Hid に関連する情報 4バイト
    NN_PADDING4;
};

/**
 * @brief       登録されているデバイスの情報を表す構造体です
 */
struct RegisteredDeviceInfo
{
    int64_t registerationCount;                    //!< デバイスが登録されるたびにインクリメントされる数字
    BluetoothDeviceInfo bluetooth;                 //!< Bluetooth に関する情報 56 バイト
    IdentificationCode identificationCode;         //!< シリアルナンバー 16 バイト
    DeviceColor color;                             //!< 色情報 16 バイト
    DeviceType deviceType;                         //!< デバイスタイプ 1バイト
    DeviceSubType subType;                         //!< サブタイプ 1バイト
    InterfaceType interfaceType;                   //!< ペアリングに利用されたインタフェース 1バイト
    bool isIncomplete;                             //!< 色情報などの登録がまだで不完全なデータである (NxAddon 1.0.0 以前に登録されたコントローラーの情報)
    bool isUpdated;                                //!< もともと登録済みのデータが更新されただけかどうか。false の場合は新規に登録されたデバイス
    uint8_t designVariation;                       //!< デザインのバリエーションの情報
    NN_PADDING2;
};

/**
 * @brief       コントローラーとのボタンペアリングを開始します
 *              ペアリングは 10秒間継続し、その後終了します。
 *
 */
void StartButtonPairing() NN_NOEXCEPT;

/**
 * @brief       新規にデバイスがが登録された際に、通知を受けるためのイベントオブジェクトを登録します
 *
 * @details     有線ペアリングが完了した際に、登録されたイベントオブジェクトに対して通知を行います
 *              ペアリングされたデバイスの情報については、GetPairedDeviceInfo() で取得してください
 *
 * @param[in]   pSystemEvent                  有線ペアリングの完了通知を受けるイベント
 *
 * @pre         - pSystemEvent != nullptr
 */
void BindPairingCompletedEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

/**
 * @brief       新規にデバイスが登録された際に、通知を受けるためのイベントオブジェクトを解除します
 *
 * @param[in]   pSystemEvent                  解除するイベントオブジェクト
 *
 * @pre         - pSystemEvent != nullptr
 */
void UnbindPairingCompletedEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

/**
 * @brief      登録されているデバイスの情報を取得します
 *
 * @details    取得できるデバイスの情報の最大数は　RegisteredDeviceCountMax 個です
 *
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 *
 * @return     登録されているデバイスの数です
 *
 */
size_t GetRegisteredDevices(RegisteredDeviceInfo* pOutValues, size_t length) NN_NOEXCEPT;

/**
 * @brief      デバイスを手動で登録します。
 *
 * @details    デバイスを手動で登録します。
 *             最大数に達すると自動的にもっとも古いデバイスが削除されます。
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 *
 */
void AddRegisteredDeviceInfo(const RegisteredDeviceInfo& info) NN_NOEXCEPT;

}} // namespace nn::xcd
