﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       xcd ライブラリに関係する型の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/os/os_MultipleWait.h>
#include "xcd_Device.h"

namespace nn { namespace xcd {

/**
 * @brief   タグの ID (UID) の最大長です。
 */
const int NfcUidLengthMax = 10;

/**
 * @brief   Read コマンドのブロック最大数です。
 */
const int NtagReadBlockCountMax = 4;

/**
 * @brief   Write コマンドのブロック最大数です。
 */
const int NtagWriteBlockCountMax = 4;

/**
 * @brief   パススルーデータの最大サイズです。
 */
const size_t NfcPassThruDataSizeMax = 512;

/**
 * @brief   Type2 タグの認証データのサイズです。
 */
const size_t Type2TagSignatureSize = 32;

/**
 * @brief   Type2 タグのページサイズです。
 */
const size_t Type2TagPageSize = 4;

/**
 * @brief   1 ブロックで読み込める最大ページ数です。
 */
const int NtagReadBlockPageCountMax = 60;

/**
 * @brief   1 ブロックで読み込める最大バイト数です。
 */
const size_t NtagReadBlockSizeMax = NtagReadBlockPageCountMax * Type2TagPageSize;

/**
 * @brief   一度に受信できる NFC データの最大値です。
 */
const size_t NfcReceiveDataSizeMax = 2048;

/**
 * @brief   アクティベーション情報のサイズです。
 */
const size_t NtagActivationAreaSize = 4;

/**
 * @brief   1 ブロックで書き込める最大ページ数です。
 */
const int NtagWriteBlockPageCountMax = 60;

/**
 * @brief   1 ブロックで書き込める最大サイズです。
 */
const size_t NtagWriteBlockSizeMax = NtagWriteBlockPageCountMax * Type2TagPageSize;

/**
 * @brief   MIFARE 生鍵の長さです。
 */
const int MifareKeyLength = 6;

/**
 * @brief   MIFARE 鍵の登録に使用する、暗号化された鍵の長さです。
 */
const int MifareEncryptedKeyLength = 8;

/**
 * @brief   登録できる MIFARE 鍵の最大数です。
 */
const int MifareKeyCountMax = 12;

/**
 * @brief   MIFARE データブロックのバイト数です。
 */
const int MifareBlockBytes = 16;

/**
 * @brief   MIFARE Read コマンドのブロック最大数です。
 */
const int MifareReadBlockCountMax = 16;

/**
 * @brief   MIFARE Write コマンドのブロック最大数です。
 */
const int MifareWriteBlockCountMax = 16;


/**
 * @brief   NFC 関連のイベントの発生理由です。発生したイベントに応じて、@ref NfcInfo の内部フォーマットを特定するために使用します。
 */
enum NfcEventReason : uint32_t
{
    NfcEventReason_None,                    //!< イベントは発生していません。
    NfcEventReason_Detected,                //!< タグを検出しました。
    NfcEventReason_Deactivated,             //!< タグを喪失しました。
    NfcEventReason_ReadFinish,              //!< タグからの読み込みが終了しました。
    NfcEventReason_WriteFinish,             //!< タグへの書き込みが終了しました。
    NfcEventReason_PassThruResult,          //!< パススルーコマンドの結果が取得されました。
    NfcEventReason_ControllerDisconnected,  //!< コントローラが切断されました。
    NfcEventReason_Error,                   //!< エラーが発生しました。
    NfcEventReason_MifareResult,            //!< MIFARE タグアクセスの結果が取得されました。
    NfcEventReason_MifareKeyWriteFinish     //!< MIFARE 鍵書き込みが完了しました。
};

/**
 * @brief   NFC コマンドのリザルトコードです。
 */
enum NfcResultCode : uint32_t
{
    NfcResultCode_Ok                      =  0,
    NfcResultCode_FunctionError           = 60,   //!< NFC ハードウェアに異常があります。
    NfcResultCode_ResetRequired           = 61,   //!< NFC スタックが異常な状態です。
    NfcResultCode_ReadError               = 62,   //!< データ読み込み中にエラーが発生しました。
    NfcResultCode_WriteError              = 63,   //!< データ書き込み中にエラーが発生しました。
    NfcResultCode_ArgumentError           = 64,   //!< コマンドの引数が不正です。
    NfcResultCode_Timeout                 = 65,   //!< タグが時間内に検知されませんでした。
    NfcResultCode_InvalidUid              = 66,   //!< 指定した UID と検出したタグの UID が異なります。
    NfcResultCode_Type2PasswordInvalidTag = 68,   //!< パスワード情報が不正です。
    NfcResultCode_VerifyError             = 69,   //!< Verify 処理に失敗しました。
    NfcResultCode_ActivationError         = 70,   //!< Activation 処理に失敗しました。
    NfcResultCode_InvalidTag              = 71,   //!< サポート外のタグを検知しました。
    NfcResultCode_InvalidFormat           = 72,   //!< タグのフォーマットが不正です。
    NfcResultCode_AuthenticationError     = 73,   //!< 認証処理に失敗しました。
    NfcResultCode_SequenceError           = 74,   //!< NFC 内部ステートに不適なコマンドを受信しました。
    NfcResultCode_CommandTimeout          = 75,   //!< タグへのコマンド送信がタイムアウトしました。
    NfcResultCode_MifareError             = 76    //!< MIFARE アクセスに失敗しました。
};

/**
 * @brief   NFC タグのプロトコルを表す列挙型です。
 */
enum NfcProtocol : uint32_t
{
    NfcProtocol_TypeA     = 0x00,   //!< NFC-A
    NfcProtocol_TypeB     = 0x01,   //!< NFC-B
    NfcProtocol_TypeF     = 0x02,   //!< NFC-F
    NfcProtocol_Type15693 = 0x06    //!< ISO-15693
};

/**
 * @brief   NFC タグのタイプを表す列挙型です。
 */
enum NfcTagType : uint32_t
{
    NfcTagType_Error    = 0,            //!< 想定していないタイプのタグです。
    NfcTagType_Type1    = 1,            //!< TYPE1
    NfcTagType_Type2    = 1 << 1,       //!< TYPE2
    NfcTagType_Type3    = 1 << 2,       //!< TYPE3
    NfcTagType_Type4A   = 1 << 3,       //!< TYPE4A
    NfcTagType_Type4B   = 1 << 4,       //!< TYPE4B
    NfcTagType_Iso15693 = 1 << 5,       //!< ISO15693
    NfcTagType_Mifare   = 1 << 6,       //!< MIFARE
    NfcTagType_All      = 0xFFFFFFFF    //!< 全タイプ (デフォルト)
};

/**
 * @brief   NFC タグの Polling のマスク設定です。
 */
enum NfcPollingMask : uint32_t
{
    NfcPollingMask_All                = 0x00,       //!< デフォルト値です。全ての Type をポーリングします。
    NfcPollingMask_TechnologyA        = 0x01,       //!< Type A だけをポーリングするマスク設定です。
    NfcPollingMask_TechnologyB        = 0x02,       //!< Type B だけをポーリングするマスク設定です。
    NfcPollingMask_TechnologyF        = 0x04,       //!< Type F だけをポーリングするマスク設定です。
    NfcPollingMask_TechnologyIso15693 = 0x08        //!< ISO15693 だけをポーリングするマスク設定です。
};

/**
 * @brief   Type2 タグの種類を表すバージョン情報です。
 */
enum NfcType2TagVersion : uint8_t
{
    NfcType2TagVersion_Ntag215     = 0,     //!< NTAG215
    NfcType2TagVersion_Ntag210     = 1,     //!< NTAG210
    NfcType2TagVersion_Ntag212     = 2,     //!< NTAG212
    NfcType2TagVersion_Ntag213     = 3,     //!< NTAG213
    NfcType2TagVersion_Ntag216     = 4,     //!< NTAG216
    NfcType2TagVersion_Unsupported = 5      //!< サポートしていないタグ
};

/**
 * @brief   MIFARE タグアクセスに使用する鍵の指定方法です。
 */
enum MifareKeyValueFormat : uint8_t
{
    MifareKeyValueFormat_Id  = 0,   //!< ID 指定
    MifareKeyValueFormat_Raw = 1    //!< 生鍵指定
};

/**
 * @brief   MIFARE タグアクセスに使用する鍵の種類です。
 */
enum MifareKeyType : uint8_t
{
    MifareKeyType_A = 0x60,         //!< Key A
    MifareKeyType_B = 0x61          //!< Key B
};

/**
 * @brief   Tera MCU のファームウェアバージョン情報です。
 */
struct McuVersionDataForNfc
{
    uint16_t            major;          //!< Tera MCU のメジャーバージョン
    uint16_t            minor;          //!< Tera MCU のマイナーバージョン
    nn::xcd::DeviceType deviceType;     //!< デバイスの種類
    char                _reserve[27];   // 予約
};

/**
 * @brief   タグの ID (UID) を示す構造体です。
 */
struct NfcTagId
{
    char    uid[NfcUidLengthMax];  //!< タグの UID
    NN_PADDING2;
    int32_t length;                //!< UID の長さ
};

/**
 * @brief       NFC タグの基本情報です。
 */
struct NfcTagInfo
{
    NfcTagId       tagId;           //!< タグのユニーク ID (UID)
    NfcProtocol    protocol;        //!< 通信プロトコル
    NfcTagType     tagType;         //!< タグタイプ
};

/**
 * @brief       NFC のエラー情報です。
 */
struct NfcErrorInfo
{
    NfcResultCode resultCode;       //!< リザルトコード

    // エラーに応じた付加情報を追加する可能性あり
};

/**
 * @brief       NTAG の FAST_READ コマンド用のアドレス情報
 */
struct NtagReadAddress
{
    uint8_t  startPage;         //!< FAST_READ コマンドのスタートページアドレス
    uint8_t  endPage;           //!< FAST_READ コマンドのエンドページアドレス
    NN_PADDING2;
};

/**
 * @brief       NTAG シリーズのタグデータ情報です。
 */
struct NfcNtagData
{
    NfcTagInfo          tagInfo;                            //!< NFC タグの基本情報

    uint8_t             type2TagVersion;                    //!< Type2 タグのバージョン。 @ref NfcType2TagVersion の値を参照
    NN_PADDING3;
    char                signature[Type2TagSignatureSize];   //!< Type2 タグの認証情報
    int32_t             blockCount;                         //!< 読み込んだブロック数
    struct
    {
        NtagReadAddress address;                            //!< 読み込んだデータのアドレス情報
        char            data[NtagReadBlockSizeMax];         //!< データ本体
    } readDataBlocks[NtagReadBlockCountMax];                //!< 読み込んだデータ
};

/**
 * @brief       パススルーモードでの受信データです。
 */
struct NfcPassThruData
{
    NfcTagInfo tagInfo;                                //!< NFC タグの基本情報
    char       responseData[NfcPassThruDataSizeMax];   //!< 受信データ
    uint32_t   responseSize;                           //!< 受信サイズ
};

/**
 * @brief       MIFARE タグに対する処理の応答です。
 */
struct NfcMifareData
{
    int32_t     dataCount;      //!< 受信したデータの数。読み込みに対する応答以外の場合は 0
    struct
    {
        uint8_t blockAddress;                       //!< ブロックアドレス
        char    data[MifareBlockBytes];             //!< データ本体
    } readDataBlocks[MifareReadBlockCountMax];      //!< 読み込んだデータ。dataCount 分の要素が有効なデータ
};

/**
 * @brief       NFC で受信する可能性あるデータを含んだ構造体です。
 *
 * @details     共用体内部には、reason の値に応じた付加情報が格納されます。
 *              特に記述がない reason に対しては、付加情報はありません (共用体内部は all 0 になります)。
 */
struct NfcInfo
{
    NfcEventReason reason;     //!< イベントの Reason (共用体の内部フォーマットを特定)

    union
    {
        NfcErrorInfo    errorInfo;                  //!< エラー情報。reason が @ref NfcEventReason_Error のときに使用します。
        NfcTagInfo      tagInfo;                    //!< NFC タグの基本情報。reason が @ref NfcEventReason_Detected のときに使用します。
        NfcNtagData     ntagData;                   //!< NTAG シリーズのタグデータ。reason が @ref NfcEventReason_ReadFinish のときに使用します。
        NfcPassThruData passThruData;               //!< パススルーモードの取得データ。reason が @ref NfcEventReason_PassThruResult のときに使用します。
        NfcMifareData   mifareData;                 //!< MIFARE タグに対する操作への応答。reason が @ref NfcEventReason_MifareResult のときに使用します。

        char _dummy[NfcReceiveDataSizeMax];         // 上限となるデータサイズを規定
    };
};

/**
 * @brief       タグ検出用のパラメータです。
 */
struct NfcDiscoveryParameter
{
    NfcPollingMask pollingMask;     //!< 検出するタグのマスク
    uint16_t activationTimeout;     //!< polling のタイムアウト時間設定 (0 : 無制限, 1 - 65535 ms)
    uint16_t discoveryPeriod;       //!< polling の周期 (0 - 65535 ms)
};

/**
 * @brief       NTAG Read アクセス用のパラメータです。
 */
struct NtagReadParameter
{
    uint16_t  timeoutMsec;                //!< コマンドのタイムアウト (ミリ秒単位で指定)
    bool      isPasswordRequired;         //!< パスワード認証を行うかどうかのフラグ
    NN_PADDING1;
    NfcTagId  tagId;                      //!< アクセスするタグの UID (All 0 の場合は指定なし）
    int32_t   blockCount;                 //!< Read コマンドで読み込むブロックの数
    NtagReadAddress addresses[NtagReadBlockCountMax];  //!< 各ブロックのデータ読み込みアドレスを指定。ブロック毎に 60 ページ分指定可能
};

/**
 * @brief       NTAG の WRITE コマンド用の指定データです。
 */
struct NtagWriteData
{
    bool    isActivationNeeded;                     //!< Activation を行うかどうかのフラグ
    NN_PADDING3;
    char    clearData[NtagActivationAreaSize];      //!< Clear 時に書き込むデータ
    char    activationData[NtagActivationAreaSize]; //!< Activation 時に書き込むデータ
    int32_t blockCount;                             //!< 書き込むブロック数（1 - 4）
    struct
    {
        uint8_t  startPageAddress;                  //!< n 回目の WRITE コマンドのスタートページアドレス
        uint8_t  dataSize;                          //!< n ブロック目の Write サイズ (4 - 240 byte : 4 の倍数)
        NN_PADDING2;
        char     data[NtagWriteBlockSizeMax];       //!< n ブロック目に書き込むデータ（dataSize 分が有効なデータ）
    } dataBlocks[NtagWriteBlockCountMax];
};

/**
 * @brief       NTAG Write アクセス用のパラメータです。
 */
struct NtagWriteParameter
{
    uint16_t        timeoutMsec;            //!< タイムアウト (ミリ秒単位で指定)
    bool            isPasswordRequired;     //!< パスワード認証を行うかどうかのフラグ
    uint8_t         type2TagVersion;        //!< Type2 タグのバージョン
    NfcTagId        tagId;                  //!< アクセスするタグの UID
    NtagWriteData   ntagWriteData;          //!< タグに書き込むデータ
};

/**
 * @brief       パススルーコマンド用のデータです。
 */
struct NfcPassThruParameter
{
    uint16_t  timeoutMsec;                       //!< タイムアウト (ミリ秒単位で指定)
    NN_PADDING2;
    NfcTagId  tagId;                             //!< アクセスするタグの UID (すべて 0 の場合は最初に見つけたタグが対象)
    uint32_t  sendDataSize;                      //!< パススルーコマンドのデータサイズ
    char      sendData[NfcPassThruDataSizeMax];  //!< 送信データ
};

/**
 * @brief       MIFARE 鍵の登録に使用する、暗号化された鍵の値です。
 */
struct MifareEncryptedKeyValue
{
    nn::Bit8    value[MifareEncryptedKeyLength];    //!< 暗号化された鍵の値
};

/**
 * @brief       MIFARE 鍵の値です。
 */
struct MifareKeyValue
{
    nn::Bit8    value[MifareKeyLength];         //!< 鍵の値
    NN_PADDING2;
};

/**
 * @brief       MIFARE 鍵の書き込みパラメータです。
 */
struct MifareKeyWriteParameter
{
    uint16_t                timeoutMsec;                //!< コマンドのタイムアウト (ミリ秒単位で指定)
    uint8_t                 keyCount;                   //!< 登録する鍵の数 (1 - 12)
    NN_PADDING1;
    MifareEncryptedKeyValue keys[MifareKeyCountMax];    //!< 登録する暗号化済み鍵の一覧。keyCount 以降の要素は無視される
};

/**
 * @brief       MIFARE 鍵の消去パラメータです。
 */
struct MifareKeyClearParameter
{
    uint16_t        timeoutMsec;                //!< コマンドのタイムアウト (ミリ秒単位で指定)
    NN_PADDING2;
};

/**
 * @brief       単一の MIFARE 鍵情報です。
 */
struct MifareKey
{
    MifareKeyType           type;       //!< 認証に使用する鍵の種類
    NN_PADDING3;
    union
    {
        uint8_t             id;         //!< 鍵 ID。keyFormat が @ref MifareKeyValueFormat_Id の場合に使用
        MifareKeyValue      raw;        //!< 生鍵の値。keyFormat が @ref MifareKeyValueFormat_Raw の場合に使用
        char                _dummy[8];  // 上限となるデータサイズを規定
    };
};

/**
 * @brief       MIFARE タグの各ブロックを読み込むためのパラメータです。
 */
struct MifareReadBlockParameter
{
    MifareKey   key;                //!< 認証に使用する MIFARE 鍵
    uint8_t     blockAddress;       //!< 読み込むブロックのアドレス
    NN_PADDING3;
};

/**
 * @brief       MIFARE Read アクセス用のパラメータです。
 */
struct MifareReadParameter
{
    uint16_t                 timeoutMsec;   //!< コマンドのタイムアウト (ミリ秒単位で指定)
    MifareKeyValueFormat     keyFormat;     //!< 認証に使用する鍵の指定方法
    NN_PADDING1;
    NfcTagId                 tagId;         //!< アクセスするタグの UID
    int32_t                  blockCount;    //!< 読み込むブロックの数 (1 - 16)
    MifareReadBlockParameter blocks[MifareReadBlockCountMax];   //!< 読み込むブロックの指定。blockCount 分の要素が有効
};

/**
 * @brief       MIFARE タグの各ブロックに書き込むためのパラメータです。
 */
struct MifareWriteBlockParameter
{
    MifareKey   key;                        //!< 認証に使用する MIFARE 鍵
    uint8_t     blockAddress;               //!< 読み込むブロックのアドレス
    NN_PADDING3;
    char        data[MifareBlockBytes];     //!< 書き込むデータ
};

/**
 * @brief       MIFARE Write アクセス用のパラメータです。
 */
struct MifareWriteParameter
{
    uint16_t                  timeoutMsec;  //!< コマンドのタイムアウト (ミリ秒単位で指定)
    MifareKeyValueFormat      keyFormat;    //!< 認証に使用する鍵の指定方法
    NN_PADDING1;
    NfcTagId                  tagId;        //!< アクセスするタグの UID
    int32_t                   blockCount;   //!< 書き込むブロックの数 (1 - 16)
    MifareWriteBlockParameter blocks[MifareWriteBlockCountMax];   //!< 書き込むブロックの指定。blockCount 分の要素が有効
};

}} // namespace nn::xcd
