﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       デバイスの列挙などに関する API の宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_BitTypes.h>
#include <nn/fs/fs_File.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/xcd/xcd_Device.h>
#include <nn/xcd/xcd_FirmwareTypes.h>

namespace nn { namespace xcd {

/**
 * @brief       Bluetooth IC のファームウェア更新を開始します。
 *
 * @param[in]   handle                   ファームウェア更新を行うデバイス
 * @param[in]   image                    ファームウェアイメージの情報
 * @param[in]   pSystemEvent             ファームウェア更新が終了した際に通知を受けるイベント
 *
 * @pre         - pSystemEvent != nullptr
 *
 * @return      実行結果を返します。
 * @retval      ResultSuccess            処理に成功しました。
 * @retval      ResultNotConneced        コントローラーが接続されていません。
 */
nn::Result StartBtFirmwareUpdate(
    DeviceHandle handle,
    const FirmwareImage& image,
    nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

/**
 * @brief       Bluetooth IC のファームウェア更新を中断します。
 *
 * @detail      実行中の Bluetooh IC ファームウェア更新を中断します。
 *              ファームウェア更新を中断すると、更新前のファームウェアが動作する状態になります。
 *              Bluetooth IC ファームウェア更新中でない場合は、何もせず ResultSuccess を返します。
 *
 * @param[in]   handle                   ファームウェア更新を行うデバイス
 *
 * @return      実行結果を返します。
 * @retval      ResultSuccess            処理に成功しました。
 * @retval      ResultNotConneced        コントローラーが接続されていません。
 */
nn::Result AbortBtFirmwareUpdate(DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       Bluetooth IC のファームウェアバージョンを取得します
 *
 * @param[out]  pOutValue                Bluetooth IC の ファームウェアバージョン
 * @param[in]   handle                   ファームウェア更新を行うデバイス
 *
 * @pre         - pOutValue != nullptr
 *
 * @return      実行結果を返します。
 * @retval      ResultSuccess            処理に成功しました。
 * @retval      ResultNotConneced        コントローラーが接続されていません。
 */
nn::Result GetBtFirmwareVersion(BtFirmwareVersion* pOutValue, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       Bluetooth 経由のみの Firmware 更新がサポートされているかどうかをチェックします。
 *
 * @param[in]   handle                   ファームウェア更新を行うデバイス
 *
 * @return      Bluetooth接続以外の時に Bluetooth 経由の Firmware 更新が必須であれば true を返します
 */
bool IsBtFirmwareUpdateThroughBtRequired(DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       Bluetoot IC のファームウェアの更新中の経過状況を取得します
 *
 * @param[out]  pOutStage                ファームウェアの更新中のステージ
 * @param[out]  pOutProgress             ステージにおけるファームウェアの経過 (100段階)
 *
 * @pre         - pSystemEvent != nullptr
　*
 * @return      実行結果を返します。
 * @retval      ResultSuccess                 　 処理に成功しました。
 * @retval      ResultNotConneced              　コントローラーが接続されていません。
 * @retval      ResultFirmwareUpdateNotStarted 　ファームウェア更新処理が開始していません。
 * @retval      ResultFirmwareUpdateVerifyError ファームウェアの更新後のベリファイ中にエラーが発生しました。
 */
nn::Result GetBtFirmwareUpdateProgress(DeviceHandle handle, FirmwareUpdateStage* pOutStage, int* pOutProgress) NN_NOEXCEPT;

/**
 * @brief       ファームウェア更新対象のデバイスを指定します。
 *
 * @detail      指定したデバイスがファームウェア更新中である旨を通知し、ファームウェア更新に影響する処理が動作しないようにします。
 *
 * @param[in]   handle                  ファームウェア更新を行うデバイス
 *
 * @return      実行結果を返します。
 * @retval      ResultSuccess                 　 処理に成功しました。
 * @retval      ResultNotConneced              　コントローラーが接続されていません。
 */
nn::Result SetFirmwareUpdatingDevice(DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       ファームウェア更新対象デバイスの指定を解除します。
 *
 * @detail      ファームウェア更新対象デバイスの指定を解除し、通常通りの動作に戻します。
 */
void UnsetFirmwareUpdatingDevice() NN_NOEXCEPT;

/**
 * @brief       Bluetooth IC のファームウェア更新失敗エミュレーション機能の有効状態を設定します。
 *
 * @detail      Bluetooth IC のファームウェア更新失敗エミュレーション機能の有効状態を設定します。
 *              有効にすると、Bluetooth IC のファームウェア更新が常に Verify で失敗するようになります。
 *
 * @param[in]   isEnabled                   有効状態
 */
void SetBtFirmwareUpdateFailureEmulationEnabled(bool isEnabled) NN_NOEXCEPT;

}} // namespace nn::xcd
