﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       データフォーマットに関係する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/xcd/xcd_Device.h>

namespace nn { namespace xcd {

/**
 * @brief       デバイスから定期的な受信するデータのフォーマットです。
 */
enum PeriodicDataFormat : uint8_t
{
    PeriodicDataFormat_Basic = 0,              //!< ボタン, アナログスティック, 6軸センサーを利用する際のデータフォーマットです。 Joy-Con (R)  /  Joy-Con (L)  / Switch Proコントローラー
    PeriodicDataFormat_MCU ,                   //!< NFCまたはIR Sensorを利用する際のデータフォーマットです。  Joy-Con (R)  / Switch Proコントローラー
    PeriodicDataFormat_Attachment ,            //!< 拡張デバイスを利用する際のデータフォーマットです。 Joy-Con (R)
    PeriodicDataFormat_Audio,                  //!< Audioを利用する際のデータフォーマットです。 Switch Proコントローラー
    PeriodicDataFormat_Generic,                //!< 汎用データフォーマットです。  Joy-Con (R)  /  Joy-Con (L)  / Switch Proコントローラー
    PeriodicDataFormat_McuUpdate,              //!< TeraMCU の更新に使用するデータフォーマットです。 Joy-Con (R)  / Switch Proコントローラー
    PeriodicDataFormat_BtUpdate,               //!< Bt の更新に使用するデータフォーマットです。  Joy-Con (R)  /  Joy-Con (L)  / Switch Proコントローラー
};

/**
 * @brief      デバイスから定期的な受け取るデータのフォーマットを取得します。
 *
 * @details    デバイスから定期的な受け取るデータのフォーマットを取得します。
 *             デバイスは通信間隔に応じて定期的にコントローラデータを送信します。
 *             デバイス毎にサポートされる通信フォーマットは異なります。
 *             総てのデバイスにおいて、接続直後は PeriodicDataFormat::DataFormatBasic で通信します。
 *             データフォーマットを指定するためには SetDataFormat() を利用します。
 *
 * @param[out] pOutValue                   現在使用しているデータフォーマット
 * @param[in]  handle                      データフォーマットを取得するデバイスへのハンドル
 *
 * @return     実行結果を返します。
 * @retval     ResultSuccess               処理に成功しました。
 * @retval     ResultNotConnected          デバイスが接続されていません。
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 *             - pOutValue!= nullptr
 */
Result GetDataFormat(PeriodicDataFormat* pOutValue, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief      デバイスから定期的な受け取るデータのフォーマットを指定します。
 *
 * @details    デバイスから定期的な受け取るデータのフォーマットを指定します。
 *             指定できるデータフォーマットは本体の通信モードによって異なります。
 *
 * @param[in]  handle                      データフォーマットを変更するデバイスへのハンドル
 * @param[in]  dataFormat                  設定するデータフォーマット
 *
 * @return     実行結果を返します。
 * @retval     ResultSuccess               処理に成功しました。
 * @retval     ResultNotConnected          デバイスが接続されていません。
 * @retval     ResultInvalidInterval       指定のデータフォーマットは現在の通信モードではサポートされていません。
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 */
Result SetDataFormat(PeriodicDataFormat dataFormat, DeviceHandle handle) NN_NOEXCEPT;

}} // namespace nn::xcd
