﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os/os_SharedMemoryTypes.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/mbuf/mbuf_Definitions.h>

#include "wlan_Common.h"
#include "wlan_MacAddress.h"
#include "wlan_Ssid.h"
#include "wlan_StateDefinition.h"
#include "wlan_Types.h"

namespace nn {
namespace wlan {


/*!
   @name       ソケットインターフェース
   @{
 */


//--------------------------------------------------------------------------
/**
 *    @brief        WLANライブラリのソケットインターフェースのサービスを開始します。@n
 *    この初期化を行っていないと、ソケットインターフェースに関する API は使用できません。本関数は必ず成功します。@n
 *    既に本関数を呼んでいた場合、何も行わず成功が返ります。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess    初期化に成功。
 */
nn::Result InitializeSocketManager() NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
 *    @brief        WLANライブラリのソケットインターフェースのサービスを終了します。本関数は必ず成功します。@n
 *    @ref InitializeSocketManger を呼んでいない状態で呼んだ場合、何もせず成功が返ります。(developビルドの場合アサートします)
 */
nn::Result FinalizeSocketManager() NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
 *    @brief        SharedMemory の登録を行います。初期化を行っていないと使用できません。
 */
nn::Result RegisterSharedMemory(
        nn::os::NativeHandle memHandle,
        uint32_t memSize,
        nn::os::NativeHandle txEventHandle,
        nn::os::NativeHandle rxEventHandle,
        nn::os::NativeHandle txCbufEventHandle,
        nn::os::NativeHandle rxCbufEventHandle) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        SharedMemory の使用を終了し登録を削除します。初期化を行っていないと使用できません。
 */
nn::Result UnregisterSharedMemory() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        SharedMemory の使用を開始します。初期化を行っていないと使用できません。
 */
nn::Result EnableSharedMemory() NN_NOEXCEPT;


namespace Socket {


//--------------------------------------------------------------------------
/**
 *    @brief        接続中の相手にデータフレームを送信します。コピー版。
 *
 *    @param[in] pInput 送信データへのポインタです。
 *    @param[in] size   送信データのサイズです。2KB未満を指定してください。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         成功。
 *    @retval       ResultInvalidArgument 不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultTxQueueIsFull   送信キューが一杯なため送信を諦めました。
 *    @retval       ResultNotAuthorized   接続先とのリンクが確立していないのでデータの送信が行えません。@n
 *                  接続は完了していても、@ref AddMatchingDataToRxEntry によるデータ列の登録がされていない場合は本Result値が返ります。
 *    @retval       ResultCommandFailure  送信に失敗しました。既に通信相手との接続が切れている場合に発生します。@n
 *                  wlanプロセスが切断を検知していればResultNotAuthorizedが返りますが、検知前に送信が行われた場合に発生します。
 */
nn::Result PutFrameRaw(const uint8_t pInput[], size_t size ) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        登録済みの受信エントリからデータフレームを取り出します。何も受信していなかった場合、受信するか接続が切れるまでブロックします。コピー版。
 *    受信エントリを登録している必要があります。@ref CreateRxEntry にて受信エントリを作成しておいて下さい。
 *
 *    @param[out] pOutBuf  受信バッファへのポインタです。
 *    @param[in]  size     受信バッファのサイズです。受信データは最大でも2KBを超えることはありません。
 *    @param[out] pOutSize 実際に受信したデータのサイズが返ります。
 *    @param[in]  rxId      受信エントリの ID
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess            成功。
 *    @retval       ResultInvalidArgument    不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultRxEntryNotFound    指定されたrxIdが見つかりません。
 *    @retval       ResultNotAuthorized      接続先とのリンクが確立していないのでデータの送信が行えません。
 *    @retval       ResultGetFrameCancelled  受信中止が指示されました。受信の中止は@ref CancelGetFrame を呼ぶか、@ref DeleteRxEntry または@n
 *                  CloseMaster/ClientMode の処理の中で自動的に呼ばれます。
 *    @retval       ResultRxEntryIsMuted     データパスが無効化されています。CloseMaster/ClientModeを呼ぶと、wlanプロセス内部でデータパスを一時的に無効化状態とします。@n
 *                  この状態で本関数を呼ぶと受信は行えずに本リザルトが返ります。OpenMaster/ClientModeを呼ぶとこの無効化状態は解除されます。
 *    @retval       ResultBufferTooShort     pOutBufのバッファサイズが小さく、受信データを格納出来ません。
 */
nn::Result GetFrameRaw(uint8_t pOutBuf[], size_t size, size_t* pOutSize, uint32_t rxId) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        @ref GetFrame または @ref GetFrameRaw で受信待ちの状態を解除します。
 *
 *    @param[in] rxId         受信エントリの ID
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess            成功。
 *    @retval       ResultRxEntryNotFound    指定されたrxIdが見つかりません。
 */
nn::Result CancelGetFrame(uint32_t rxId) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        受信エントリを作成します。
 *
 *    @param[out]   pRxId        作成された受信エントリの IDが返ります。
 *    @param[in]    pEthertypes  受信エントリに登録するEthertypeの配列です。
 *    @param[in]    count        Ethertype配列の要素数です。
 *    @param[in]    capacity     受信エントリに溜めておける受信フレーム数の上限値です。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess             成功。
 *    @retval       ResultInvalidArgument    不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultDuplicateRxEntry    指定した Ethertype についての受信エントリが既に存在しています。
 *    @retval       ResultRxEntryCountIsFull  作成可能な受信エントリ数が最大に達しています。
 */
nn::Result CreateRxEntry(uint32_t* pOutRxId, const uint16_t pEthertypes[], uint32_t count, uint32_t capacity) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        受信エントリを削除します。
 *
 *    @param[in]    rxId         受信エントリの ID
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess            成功。
 *    @retval       ResultRxEntryNotFound    指定されたrxIdが見つかりません。
 */
nn::Result DeleteRxEntry(uint32_t rxId) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        登録済み受信エントリに指定したEthertypeを追加します。
 *
 *    @param[in]    rxId        受信エントリの ID
 *    @param[in]    ethertype   追加するEthertype
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess          成功。
 *    @retval       ResultRxEntryNotFound  指定されたrxIdが見つかりません。
 *    @retval       ResultDuplicateRxEntry 受信エントリに指定したEthertype が既に登録されています。
 */
nn::Result AddEthertypeToRxEntry(uint32_t rxId, uint16_t ethertype) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        登録済み受信エントリから指定したEthertypeを削除し、もともと登録されていた受信エントリの ID を取得します。
 *
 *    @param[out]   pOutRxId    受信エントリの ID
 *    @param[in]    ethertype   削除するEthertype
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess            成功。
 *    @retval       ResultInvalidArgument    不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultRxEntryNotFound    指定されたrxIdが見つかりません。。または、受信エントリに指定したEthertype が登録されていません。
 */
nn::Result DeleteEthertypeFromRxEntry(uint32_t* pOutRxId, uint16_t ethertype) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        無線チップのMACアドレスを取得します。
 *
 *    @param[out]   mac 無線チップのMACアドレスが返ります。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultInvalidArgument 不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultSuccess         成功。
 */
nn::Result GetMacAddress(MacAddress* mac) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        @ref GetDeltaTimeBetweenSystemAndTsf を使用可能にします。(デバッグ用)@n
 *    TSFタイマー取得機能が有効化されます。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         成功。
 */
nn::Result EnableTsfTimerFunction() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        @ref GetDeltaTimeBetweenSystemAndTsf を使用不可能にします。(デバッグ用)
 *    TSFタイマー取得機能が無効化されます。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         成功。
 */
nn::Result DisableTsfTimerFunction() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        無線チップのTSFタイマー値とシステムチック値との差分を取得します。単位はマイクロ秒です。(デバッグ用)@n
 *    差分の計算は、「システムチック値(マイクロ秒換算) - TSFタイマー値(マイクロ秒)」で行っています。@n
 *    また、TSFタイマー値はリンクが確立している状態でないと正しい値となりません。@n
 *    未接続状態で本関数を呼んでも意味のある値は返りませんのでご注意ください。@n
 *    @ref EnableTsfTimerFunction で本機能を使用可能にしている必要があります。
 *
 *    @param[out]   pOutDeltaTime  無線チップのTSFタイマー値とシステムチック値との差分がマイクロ秒を単位として返ります。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         成功。
 *    @retval       ResultInvalidArgument 不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultNotEnabled      失敗。@ref EnableTsfTimerFunction で使用可能にする必要があります。
 */
nn::Result GetDeltaTimeBetweenSystemAndTsf(int64_t* pOutDeltaTime) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
 *    @brief        マルチキャストアドレスリストを設定します。
 *
 *    @param[in] info マルチキャストフィルタの情報です。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         成功。
 *    @retval       ResultInvalidArgument 不正な引数値です。（developビルドの場合、アサートします）
 */
nn::Result SetMulticastFilter(const MulticastFilterInfo& info) NN_NOEXCEPT;
}
}
}

