﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   MACアドレスを取り扱うクラスの宣言
 */

#pragma once

#include <cstring>
#include <nn/nn_Common.h>
#include <nn/util/util_FormatString.h>

namespace nn {
namespace wlan {

/*!
    @name       MACアドレス
    @{
 */

/*!
    :category   MACアドレス
    @brief MAC データクラス
 */
class MacAddress
{
public:
    static const size_t MacAddressSize = 6; //!< MAC アドレスのバイトサイズです。
    static const size_t OuiSize        = 3; //!< OUI のバイトサイズです。
    static const size_t MacStringSize  = sizeof("00:00:00:00:00:00");

private:
    uint8_t m_MacAddress[MacAddressSize];

public:

    /*!
       @brief     MacAddress クラスのコンストラクタです。MAC アドレスは0クリアされます。
     */
    MacAddress() NN_NOEXCEPT
    {
        std::memset( &m_MacAddress[0], 0, MacAddressSize);
    };

    /*!
       @brief     MacAddress クラスのコンストラクタです。MAC アドレスは与えられたバイト配列値で初期化されます。
       @param[in] mac    セットする MAC アドレスのバイト配列。
     */
    explicit MacAddress(const uint8_t macAddress[MacAddressSize]) NN_NOEXCEPT
    {
        this->Set(macAddress);
    }

    /*!
       @brief     MacAddress クラスのコンストラクタです。MAC アドレスは6 バイト分の引数値で初期化されます。
       @param[in] value1    セットする MAC アドレスの 1 バイト目。
       @param[in] value2    セットする MAC アドレスの 2 バイト目。
       @param[in] value3    セットする MAC アドレスの 3 バイト目。
       @param[in] value4    セットする MAC アドレスの 4 バイト目。
       @param[in] value5    セットする MAC アドレスの 5 バイト目。
       @param[in] value6    セットする MAC アドレスの 6 バイト目。
     */
    MacAddress(uint8_t value1, uint8_t value2, uint8_t value3, uint8_t value4, uint8_t value5, uint8_t value6) NN_NOEXCEPT
    {
        SetDirect(value1, value2, value3, value4, value5, value6);
    }

    /*!
       @brief     MacAddress クラスのデストラクタです。
     */
    ~MacAddress() NN_NOEXCEPT
    {
    }



    //-----------------------------------------------------
    // 入力
    //-----------------------------------------------------

    /*!
       @brief     バイト配列による MAC アドレスのデータセットを行います。
       @param[in] inputMacAddress    セットする MAC アドレスのバイト配列。
     */
    inline void Set(const uint8_t inputMacAddress[MacAddressSize]) NN_NOEXCEPT
    {
        std::memcpy(&m_MacAddress[0], &inputMacAddress[0], MacAddressSize);
    }

    //! 6Byteでの直接入力
    /*!
       @brief     6 バイト分の引数指定による MAC アドレスのデータセットを行います。
       @param[in] A    セットする MAC アドレスの 1 バイト目。
       @param[in] B    セットする MAC アドレスの 2 バイト目。
       @param[in] C    セットする MAC アドレスの 3 バイト目。
       @param[in] D    セットする MAC アドレスの 4 バイト目。
       @param[in] E    セットする MAC アドレスの 5 バイト目。
       @param[in] F    セットする MAC アドレスの 6 バイト目。
     */
    inline void SetDirect(uint8_t value1, uint8_t value2, uint8_t value3,
            uint8_t value4, uint8_t value5, uint8_t value6) NN_NOEXCEPT
    {
        m_MacAddress[0] = value1;
        m_MacAddress[1] = value2;
        m_MacAddress[2] = value3;
        m_MacAddress[3] = value4;
        m_MacAddress[4] = value5;
        m_MacAddress[5] = value6;
    }



    //-----------------------------------------------------
    // 出力
    //-----------------------------------------------------

    /*!
       @brief     指定バッファに MAC アドレス文字列をコピーします。各バイト値の間には':'のセパレータが入ります。
       @param[out] outputMacString    MAC アドレス文字列をコピーするバッファポインタ。MacStringSize Byteのバッファが必要です。
       @return    MAC アドレス文字列へのポインタ。outputMacString 自身の先頭アドレスが返ります。
     */
    inline char* GetString(char outputMacString[MacStringSize]) const NN_NOEXCEPT
    {
        nn::util::SNPrintf(outputMacString, MacStringSize,
                            "%02X:%02X:%02X:%02X:%02X:%02X",
                            m_MacAddress[0], m_MacAddress[1], m_MacAddress[2], m_MacAddress[3], m_MacAddress[4], m_MacAddress[5]);
        return outputMacString;
    }


    // MACアドレス用の配列を定義する
    typedef uint8_t MacAddressArray[MacAddressSize];

    /*!
       @brief     MAC アドレスデータ列の取得を行います。
                  注意：ここで取得したデータ列は、書き換えることが可能です。
       @return    MAC アドレスデータ列へのポインタ。const ではないので、MacAddress クラスに直接書き換えが可能です。
     */
    inline MacAddressArray& GetMacAddressData() NN_NOEXCEPT
    {
        return m_MacAddress;
    }

    /*!
       @brief     MAC アドレスデータ列の取得を行います。
       @return    MAC アドレスデータ列へのポインタ
     */
    inline const MacAddressArray& GetMacAddressData() const NN_NOEXCEPT
    {
        return m_MacAddress;
    }

    // OUI用の配列を定義する
    typedef uint8_t OuiArray[OuiSize];

    /*!
       @brief     OUI データ列の取得を行います。
                  注意：ここで取得したデータ列は、書き換えることが可能です。
       @return    OUI データ列へのポインタ。const ではないので、MacAddress クラスに直接書き換えが可能です。
     */
    inline OuiArray& GetOuiData() NN_NOEXCEPT
    {
        return *reinterpret_cast<uint8_t(*)[OuiSize]>(m_MacAddress);
    }

    /*!
       @brief     OUI データ列の取得を行います。

       @return    OUI データ列へのポインタ。
     */
    inline const OuiArray& GetOuiData() const NN_NOEXCEPT
    {
        return *reinterpret_cast<const uint8_t(*)[OuiSize]>(m_MacAddress);
    }



    //-----------------------------------------------------
    // 解析
    //-----------------------------------------------------
    /*!
       @brief     MAC アドレスが Unicast アドレスかどうかを判定します。
       @return    MAC アドレスが Unicast アドレスの場合、true を返し、そうでない場合に false を返します。
     */
    inline bool IsUnicastAddress() const NN_NOEXCEPT
    {
        return ((m_MacAddress[0] & 0x01) == 0x00) ? true : false;
    }

    /*!
       @brief     MAC アドレスが Multicast アドレスかどうかを判定します。
       @return    MAC アドレスが Multicast アドレスの場合、true を返し、そうでない場合に false を返します。
     */
    inline bool IsMulticastAddress() const NN_NOEXCEPT
    {
        return ((m_MacAddress[0] & 0x01) == 0x01) ? true : false;
    }

    /*!
       @brief     MAC アドレスが Global アドレスかどうかを判定します。
       @return    MAC アドレスが Global アドレスの場合、true を返し、そうでない場合に false を返します。
     */
    inline bool IsGlobalAddress() const NN_NOEXCEPT
    {
        return ((m_MacAddress[0] & 0x02) == 0x00) ? true : false;
    }

    /*!
       @brief     MAC アドレスが Local アドレスかどうかを判定します。
       @return    MAC アドレスが Local アドレスの場合、true を返し、そうでない場合に false を返します。
     */
    inline bool IsLocalAddress() const NN_NOEXCEPT
    {
        return ((m_MacAddress[0] & 0x02) == 0x02) ? true : false;
    }

    /*!
       @brief     MAC アドレスが Broadcast アドレスかどうかを判定します。
       @return    MAC アドレスが Broadcast アドレスの場合、true を返し、そうでない場合に false を返します。
     */
    inline bool IsBroadcastAddress() const NN_NOEXCEPT
    {
        return m_MacAddress[0] == 0xff && m_MacAddress[1] == 0xff
               && m_MacAddress[2] == 0xff && m_MacAddress[3] == 0xff
               && m_MacAddress[4] == 0xff && m_MacAddress[5] == 0xff;
    }


    // 比較演算子のオーバーロード
    /*!
       @brief     MAC アドレス 同士の比較をし、一致するかどうかを判定します。
       @param[in] macAddress    比較対象の MAC アドレスです。
       @return    MAC アドレス が完全一致した場合、true を返し、そうでない場合に false を返します。
     */
    bool operator == (const MacAddress& macAddress) const NN_NOEXCEPT
    {
        if( std::memcmp(&m_MacAddress[0], &macAddress.m_MacAddress[0], MacAddressSize) == 0 )
        {
            return true;
        }
        else
        {
            return false;
        }
    }

    /*!
       @brief     MAC アドレス 同士の比較をし、一致しないかどうかを判定します。
       @param[in] macAddress    比較対象の MAC アドレスです。
       @return    MAC アドレス が完全一致した場合、false を返し、そうでない場合に true を返します。
     */
    bool operator != (const MacAddress& macAddress) const NN_NOEXCEPT
    {
        return !( *this == macAddress );
    }

    /*!
       @brief     00:00:00:00:00:00 の MacAddress オブジェクトを生成します。
       @return    MacAddress クラスオブジェクトです。
     */
    static inline MacAddress CreateZeroMacAddress()
    {
        const uint8_t zeroMacAddress[MacAddress::MacAddressSize] = { 0, 0, 0, 0, 0, 0 };
        return MacAddress(zeroMacAddress);
    }

    /*!
       @brief     FF:FF:FF:FF:FF:FF の MacAddress オブジェクトを生成します。
       @return    MacAddress クラスオブジェクトです。
     */
    static inline MacAddress CreateBroadcastMacAddress()
    {
        const uint8_t broadcastMacAddress[MacAddress::MacAddressSize] = { 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF };
        return MacAddress(broadcastMacAddress);
    }
};


/*!
    @}
 */

} // end of namespace wlan
} // end of namespace nn

