﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief Queries related to framebuffers.
 */

#include <cstdint>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>

namespace nn { namespace vi {

    struct FramebufferInfo;
    class Layer;

    /**
     * @brief Query the latest frame number generated by the display subsystem.
     *
     * @param[out] pOutFrameNumber The frame number.
     * @param[in]  pLayer          Layer handle.
     *
     * @return                       Whether the query was successful.
     * @retval ResultSuccess         The frame number for the last queued framebuffer was
     *                               retrieved successfully.
     * @retval ResultNotReady        No framebuffers have been queued for use on the display.
     * @retval ResultNotSupported    This feature is not supported on this platform.
     * @retval ResultOperationFailed An unknown error occurred.
     *
     * @pre
     *       - VI library is initialized.
     *       - @a pOutFrameNumber != nullptr
     *       - @a pLayer != nullptr and @a pLayer is a valid handle.
     * @post The latest frame number generated by the display subsystem has been
     *       written to @a pOutFrameNumber.
     *
     * @details After calling a presentation API, @a pOutFrameNumber may be used to identify a
     *          particular framebuffer in ListFramebuffers.
     */
    nn::Result GetLatestFrameNumber(std::uint64_t* pOutFrameNumber, const Layer* pLayer) NN_NOEXCEPT;

    /**
     * @brief Query the status of framebuffers associated with a Layer.
     *
     * @param[out] pOutFramebuffers    Buffer to receive framebuffer status information.
     * @param[in]  framebufferCountMax The maximum number of FramebufferInfo objects that exist in the buffer.
     * @param[in]  pLayer              Layer handle.
     *
     * @return The number of valid FramebufferInfo objects written to @a pOutFramebuffers.
     *
     *         If @a framebufferCountMax == 0 or @a pOutFramebuffers == nullptr, then the number
     *         of listable FramebufferInfo objects is returned.
     *
     * @pre
     *       - VI library is initialized.
     *       - @a framebufferCountMax >= 0
     *       - @a pLayer != nullptr and @a pLayer is a valid handle.
     * @post The current status of all framebuffers is written to @a pOutFramebuffers and the number of
     *       entries written is returned.
     *
     *       If @a framebufferCountMax == 0 or @a pOutFramebuffers == nullptr, no data is written
     *       and the number of listable FramebufferInfo objects is returned.
     *
     * @details The information returned is not necessarily sorted in any particular way.
     *
     *          This function may not be supported on all platforms. In this case, preconditions will
     *          be verified, but the function will always return 0. Please check the Display documentation
     *          for platform-specific details.
     */
    int ListFramebuffers(FramebufferInfo* pOutFramebuffers, int framebufferCountMax, const Layer* pLayer) NN_NOEXCEPT;
}}

