﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   Display manipulation functions for the VI library.
 * @details The functions described in this file are for internal developers only
 *          and not to be released to third parties.
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/vi/vi_Types.h>

namespace nn { namespace vi {

    /**
     * @brief Query the minimum z-order value for a layer on a display.
     *
     * @param[in] pDisplay Display handle.
     *
     * @return The minimum z-order value that can be used for a layer.
     *
     * @pre  VI library is initialized. @n
     *       @a pDisplay != nullptr and @a pDisplay is a valid handle.
     * @post The minimum z-order value returned.
     *
     * @details System policy determines the valid range of z-order values for
     *          a given process.  This ensures content from all processes is
     *          layered appropriately.
     */
    int GetZOrderCountMin(const Display* pDisplay) NN_NOEXCEPT;

    /**
     * @brief Query the maximum z-order value for a layer on a display.
     *
     * @param[in] pDisplay Display handle.
     *
     * @return The maximum z-order value that can be used for a layer.
     *
     * @pre  VI library is initialized. @n
     *       @a pDisplay != nullptr and @a pDisplay is a valid handle.
     * @post The maximum z-order value returned.
     *
     * @details System policy determines the valid range of z-order values for
     *          a given process.  This ensures content from all processes is
     *          layered appropriately.
     */
    int GetZOrderCountMax(const Display* pDisplay) NN_NOEXCEPT;

    /**
     * @brief Retrieves the display's current resolution.
     *
     * @param[out] pOutWidth  The display's current width in pixels.
     * @param[out] pOutHeight The display's current height in pixels.
     * @param[in]  pDisplay   Display handle.
     *
     * @return                       Whether the current resolution was retrieved successfully.
     * @retval ResultSuccess         Successfully retrieved the display's resolution.
     * @retval ResultOperationFailed An unknown error occurred.
     *
     * @pre  VI library is initialized. @n
     *       @a pOutWidth != nullptr @n
     *       @a pOutHeight != nullptr @n
     *       @a pDisplay != nullptr and @a pDisplay is a valid handle.
     * @post @a pOutWidth and @a pOutHeight describe the display's current resolution.
     */
    Result GetDisplayResolution(int* pOutWidth, int* pOutHeight, const Display* pDisplay) NN_NOEXCEPT;

    /**
     * @brief Query the logical space used for positioning and sizing layers.
     *
     * @param[out] pOutWidth  Width of the logical space.
     * @param[out] pOutHeight Height of the logical space.
     * @param[in]  pDisplay   Display handle.
     *
     * @return                       Whether the dimensions were retrieved successfully.
     * @retval ResultSuccess         Successfully retrieved the dimensions.
     * @retval ResultNotSupported    This display does not support this feature.
     * @retval ResultOperationFailed An unknown error occurred.
     *
     * @pre
     *       - VI library is initialized.
     *       - @a pOutWidth != nullptr
     *       - @a pOutHeight != nullptr
     *       - @a pDisplay != nullptr and @a pDisplay is a valid handle.
     * @post The dimensions were retrieved and stored in @a pOutWidth and @a pOutHeight.
     *
     * @details When this feature is enabled on a display, SetLayerSize and SetLayerPosition
     *          are relative to these dimensions.  The dimensions are independent of the
     *          current resolution of the display.
     */
    Result GetDisplayLogicalResolution(int* pOutWidth, int* pOutHeight, const Display* pDisplay) NN_NOEXCEPT;

    /**
     * @brief Upscales content within a rectangle to be full screen.
     *
     * @param[in] pDisplay Display handle.
     * @param[in] x        x-coordinate of the top-left corner of a rectangle.
     * @param[in] y        y-coordinate of the top-left corner of a rectangle.
     * @param[in] width    Width of the rectangle.
     * @param[in] height   Height of the rectangle.
     *
     * @return                         Whether the magnification was set successfully.
     * @retval ResultSuccess           Successfully set the display magnification.
     * @retval ResultNotSupported      @a pDisplay does not support this feature.
     * @retval ResultInvalidDimensions The rectangle is malformed.
     * @retval ResultOperationFailed   An unknown error occurred.
     *
     * @pre
     *       - VI library is initialized.
     *       - @a pDisplay != nullptr and @a pDisplay is a valid handle.
     * @post The display is upscaling all content within the provided rectangle to fullscreen.
     *
     * @details The corners for the rectangle are defined as follows:
     *            - Top left:     (@a x, @a y)
     *            - Top right:    (@a x + @a width, @a y)
     *            - Bottom left:  (@a x, @a y + @a height)
     *            - Bottom right: (@a x + @a width, @a y + @a height)
     *          All coordinates are specified in logical space as defined by GetDisplayLogicalResolution,
     *          which is independent of the display resolution.
     *
     *          Constraints for the rectangle are defined as follows:
     *            - 0 <= @a x < Logical resolution width
     *            - 0 <= @a y < Logical resolution height
     *            - @a width > 0
     *            - @a height > 0
     *            - @a x + @a width <= Logical resolution width
     *            - @a y + @a height <= Logical resolution height
     *          Failing any of these constraints will return ResultInvalidDimensions.
     *
     *          Magnification of very small dimensions (i.e. 1x1) and adjusting x/y coordinates may result
     *          in content dropping due to rounding errors.  This will be affected by the size of the
     *          buffers submitted to the display.
     */
    Result SetDisplayMagnification(Display* pDisplay, int x, int y, int width, int height) NN_NOEXCEPT;

}}
