﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdarg>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include <nn/vi/vi_CropRegion.h>
#include <nn/vi/native/vi_TNativeWindow.h>
#include <nn/vi/sf/vi_DisplayService.sfdl.h>

#include <nn/vi/fbshare/vi_SharedBufferConfig.h>
#include <nn/vi/fbshare/vi_SharedTextureIndexExchangeTable.h>
#include <nn/vi/fbshare/vi_SharedNativeWindowBuffer.h>
#include <nn/vi/fbshare/vi_SharedLayerTextureIndexList.h>

namespace nn{ namespace vi{ namespace fbshare{

    class SharedNativeWindow
        : public native::TNativeWindow<SharedNativeWindow>
    {
    public:
        static const int BufferTextureCountMax = SharedBufferTextureCountMax;
        static const int WindowTextureCountMax = NativeWindowTextureCountMax;
    public:
        typedef nn::visrv::sf::ISystemDisplayService ServiceType;
    public:
        SharedNativeWindow() NN_NOEXCEPT;
        ~SharedNativeWindow() NN_NOEXCEPT;

        nn::Result Initialize(
            nn::sf::SharedPointer<ServiceType> pService,
            nn::vi::fbshare::SharedBufferHandle hBuffer,
            nn::vi::fbshare::SharedLayerHandle hLayer
        ) NN_NOEXCEPT;

        void Finalize() NN_NOEXCEPT;

        // インデックスの相互変換テーブルを構築する
        //
        // SharedBuffer に登録できるバッファの枚数は BufferTextureCountMax(=16) なのに対して、
        // NVNwindow に登録できるバッファの枚数は WindowTextureCountMax(=4) しかない。
        // このため、 NVNwindow におけるバッファのインデックスと SharedBuffer におけるバッファのインデックスの対応表を用意する必要がある。
        // SharedBuffer の全バッファのうち、使用中の SharedLayer にアタッチされているバッファのインデックスのリストを設定する。
        //
        // SharedLayer にアタッチされているバッファが変更された場合、
        // PreAcquireTexture() の実行時に変更が検出され、 nvnWindowAcquireTexture() を呼び出す前にテーブルの再構築などを行う必要がある。
        void ResetExchangeTable(int* pBufferTextureIndexList, int count) NN_NOEXCEPT;

        // 先行して AcquireTexture を呼び出す。
        //
        // SharedNativeWindow::PreAcquireTexture()
        // nvnWindowAcquireTexture()
        //   -> SharedNativeWindow::DequeueBufferImpl()
        //
        // の順に呼ぶことになる。
        //
        // - Acquire に成功しなければ nn::vi::ResultNotReady を返す。具体的には以下の場合。
        //   - Acquirable イベントがタイムアウト時間内にシグナルしなかった
        //   - Acquirable イベントはシグナルしたが nn::vi::ResultPrivateSharedLayerBufferChanged が返ってきた
        // - Acquirable イベントがシグナルした直後に Acquire 不可能な状態になった場合、
        //   nn::vi::AcquireSharedFrameBuffer() が nn::vi::ResultPrivateSharedLayerNotAcquirable を返す。
        //   この場合も nn::vi::ResultNotReady を返す。
        // - Acquire したバッファのインデックスを内部に記憶する
        // - もし Acquire したバッファのインデックスが ExchangeTable に登録されていなければ、 nn::vi::ResultPrivateSharedLayerBufferChanged を返す。
        //   （ SharedLayer にアタッチされているバッファが変更された場合これが発生する。
        //      サーバ側でデタッチとアタッチが発生しても。結果的に前回と同じバッファがアタッチされていればそのまま成功になる）
        //   この場合、ExchangeTable の再構築と NVNwindow の作り直しが必要。
        //   作り直す際に使用する TextureIndexList の値が *pOutIndexList に代入される。
        //   （作り直す必要がない場合でも値は代入されるが、無視して構わない）
        nn::Result PreAcquireTexture(SharedLayerTextureIndexList* pOutIndexList, nn::TimeSpan timeout) NN_NOEXCEPT;

        // DequeueBuffer(AcquireTexture) で取得したバッファを返却する。
        // PreAcquireTexture() の時点ではバッファをキャンセルできない。一旦 DequeueBuffer する必要がある。
        void CancelTexture(int iWin) NN_NOEXCEPT;

        void SetSwapInterval(int value) NN_NOEXCEPT;
        void SetCropRegion(const CropRegion& crop) NN_NOEXCEPT;


        int SetSwapIntervalImpl(int interval) NN_NOEXCEPT;
        int DequeueBufferImpl(ANativeWindowBuffer** pOutBuffer, int* pOutFenceFd) NN_NOEXCEPT;
        int CancelBufferImpl(ANativeWindowBuffer* pBuffer, int fenceFd) NN_NOEXCEPT;
        int QueueBufferImpl(ANativeWindowBuffer* pBuffer, int fenceFd) NN_NOEXCEPT;

        int QueryFormatImpl(int* pOutValue) const NN_NOEXCEPT;
        int QueryQueuesToWindowComposerImpl(int* pOutValue) const NN_NOEXCEPT;
        int QueryConcreteTypeImpl(int* pOutValue) const NN_NOEXCEPT;
        int QueryDefaultWidthImpl(int* pOutValue) const NN_NOEXCEPT;
        int QueryDefaultHeightImpl(int* pOutValue) const NN_NOEXCEPT;
        int QueryTransformHintImpl(int* pOutValue) const NN_NOEXCEPT;
        int QueryWidthImpl(int* pOutValue) const NN_NOEXCEPT;
        int QueryHeightImpl(int* pOutValue) const NN_NOEXCEPT;
        int QueryMinUndequeuedBuffersImpl(int* pOutValue) const NN_NOEXCEPT;
        int QueryConsumerRunningBehindImpl(int* pOutValue) const NN_NOEXCEPT;
        int QueryConsumerUsageBits(int* pOutValue) const NN_NOEXCEPT;
        int QueryDefault(int* pOutValue, int what) const NN_NOEXCEPT;

        int PerformSetUsageImpl(uint32_t usage) NN_NOEXCEPT;
        int PerformSetCropImpl(const android_native_rect_t* pRect) NN_NOEXCEPT;
        int PerformSetBufferCountImpl(size_t bufferCount) NN_NOEXCEPT;
        int PerformSetBuffersGeometryImpl(int w, int h, int f) NN_NOEXCEPT;
        int PerformSetBuffersTransformImpl(int transform) NN_NOEXCEPT;
        int PerformSetBuffersFlagsImpl(uint32_t flags) NN_NOEXCEPT;
        int PerformSetBuffersTimestampImpl(int64_t timestamp) NN_NOEXCEPT;
        int PerformSetBuffersDimensionsImpl(int w, int h) NN_NOEXCEPT;
        int PerformSetBuffersUserDimensionsImpl(int w, int h) NN_NOEXCEPT;
        int PerformSetBuffersFormatImpl(int f) NN_NOEXCEPT;
        int PerformLockImpl(ANativeWindow_Buffer* pOutBuffer, ARect* pInOutDirtyBounds) NN_NOEXCEPT;
        int PerformUnlockAndPostImpl() NN_NOEXCEPT;
        int PerformSetScalingModeImpl(int mode) NN_NOEXCEPT;
        int PerformConnectImpl(int api) NN_NOEXCEPT;
        int PerformDisconnectImpl(int api) NN_NOEXCEPT;
        int PerformSetSidebandStreamImpl(native_handle_t* pHandle) NN_NOEXCEPT;
        int PerformSetPreallocatedBufferImpl(size_t slot, const native_handle_t* pHandle, int width, int height, int format, int usage, int stride) NN_NOEXCEPT;
        int PerformDefault(int operation, va_list args) NN_NOEXCEPT;

    private:
        nn::sf::SharedPointer<ServiceType> m_pService;
        nn::vi::fbshare::SharedBufferHandle m_hBuffer;
        nn::vi::fbshare::SharedLayerHandle  m_hLayer;

        // SharedLayer から Acquire 可能になるまで待つためのイベント
        nn::os::SystemEventType m_AcquirableEvent;
        // バッファインデックスの相互変換テーブル
        SharedTextureIndexExchangeTable m_ExchangeTable;

        // PreAcquireTexture で取得されたバッファのインデックス（SharedBuffer 内のインデックス）
        // DequeueBufferImpl() 内で ExchangeTable を参照して NVNwindow 用のインデックスに変換される。
        // （このため、 PreAcquireTexture と nvnWindowAcquireTexture の間で NVNwindow の作り直しが可能）
        int                        m_PreAcquiredBufferTextureIndex;
        nn::vi::native::NativeSync m_PreAcquiredBufferSync;

        // NVNwindow とやり取りする ANativeWindow の派生オブジェクト。
        // NVNwindow の初期化時に値が設定される。
        // NVNwindow の破棄時にクリアされる。
        SharedNativeWindowBuffer m_WindowBufferList[WindowTextureCountMax];

        int m_Format;
        uint32_t m_Usage;
        int m_Transform;
        int m_SwapInterval;
        nn::vi::CropRegion m_CropRegion;
    };

}}}
