﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/applet/applet_Apis.h>

#include <nn/vi/vi_Types.h>
#include <nn/vi/detail/vi_IndirectLayerImpl.h>
#include <nn/vi/vi_Result.h>

#include <nn/vi/sf/vi_DisplayService.sfdl.h>

namespace nn{ namespace vi{
    class IndirectLayer;
}}

namespace nn{ namespace vi{ namespace detail{

    class IndirectLayerTable;

    class IndirectLayerImpl
    {
    public:
        template<typename ServiceProxy>
        static nn::Result OpenIndirectLayer(
            IndirectLayer** pOutLayer,
            IndirectProducerHandleType handle,
            const nn::sf::SharedPointer<ServiceProxy>& pProxy,
            IndirectLayerTable* pLayerTable
            ) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pOutLayer);
            NN_SDK_REQUIRES_NOT_NULL(pProxy);
            bool isSuccess = false;

            // bind layer on server
            char nativeWindowDataBuffer[256];
            int64_t nativeWindowDataSize = 0;
            auto aruid = nn::applet::GetAppletResourceUserId();
            NN_RESULT_DO(pProxy->OpenIndirectLayer(
                &nativeWindowDataSize,
                nn::sf::OutBuffer(nativeWindowDataBuffer, sizeof(nativeWindowDataBuffer)),
                handle,
                aruid
            ));
            NN_UTIL_SCOPE_EXIT {
                if(!isSuccess)
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS(pProxy->CloseIndirectLayer(handle));
                }
            };

            // register on client
            NN_RESULT_DO(RegisterIndirectLayerImpl(pOutLayer, handle, nativeWindowDataBuffer, nativeWindowDataSize, pLayerTable));
            NN_UTIL_SCOPE_EXIT {
                if(!isSuccess)
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS(UnregisterIndirectLayerImpl(*pOutLayer, pLayerTable));
                }
            };

            isSuccess = true;
            NN_RESULT_SUCCESS;
        }

        template<typename ServiceProxy>
        static nn::Result CloseIndirectLayer(
            IndirectLayer* pLayer,
            const nn::sf::SharedPointer<ServiceProxy>& pProxy,
            IndirectLayerTable* pLayerTable
            ) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pLayer);
            NN_SDK_REQUIRES_NOT_NULL(pProxy);
            NN_SDK_REQUIRES_NOT_NULL(pLayerTable);

            IndirectProducerHandleType handle = 0;
            NN_RESULT_DO(GetLayerHandle(&handle, pLayer, pLayerTable));

            // unregister on client
            NN_RESULT_DO(UnregisterIndirectLayerImpl(pLayer, pLayerTable));

            // close layer on server
            NN_ABORT_UNLESS_RESULT_SUCCESS(pProxy->CloseIndirectLayer(handle));

            NN_RESULT_SUCCESS;
        }

        template<typename ServiceProxy>
        static nn::Result FlipIndirectLayer(
            IndirectLayer* pLayer,
            const nn::sf::SharedPointer<ServiceProxy>& pProxy,
            IndirectLayerTable* pLayerTable
            ) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pLayer);
            NN_SDK_REQUIRES_NOT_NULL(pProxy);
            NN_SDK_REQUIRES_NOT_NULL(pLayerTable);

            IndirectProducerHandleType handle = 0;
            NN_RESULT_DO(GetLayerHandle(&handle, pLayer, pLayerTable));

            NN_RESULT_DO(pProxy->FlipIndirectLayer(handle));

            NN_RESULT_SUCCESS;
        }

        static nn::Result GetNativeWindow(
            NativeWindowHandle* pOutNativeWindow,
            IndirectLayer* pLayer,
            IndirectLayerTable* pLayerTable
            ) NN_NOEXCEPT;

    private:
        static nn::Result GetLayerHandle(
            IndirectProducerHandleType* pOutHandle,
            IndirectLayer* pLayer,
            IndirectLayerTable* pLayerTable
            ) NN_NOEXCEPT;

        static nn::Result RegisterIndirectLayerImpl(
            IndirectLayer** pOutLayer,
            IndirectProducerHandleType handle,
            void* pNativeWindowBuffer,
            size_t nativeWindowBufferSize,
            IndirectLayerTable* pLayerTable
            ) NN_NOEXCEPT;

        static nn::Result UnregisterIndirectLayerImpl(
            IndirectLayer* pLayer,
            IndirectLayerTable* pLayerTable
            ) NN_NOEXCEPT;


    };

}}}
