﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file    usb_Types.h
 * @brief   Core public data types
 */

#include <nn/TargetConfigs/build_Compiler.h>
#include <nn/nn_Common.h>
#include <nn/usb/usb_Limits.h>

#if defined(NN_BUILD_CONFIG_COMPILER_GCC) || \
    defined(NN_BUILD_CONFIG_COMPILER_CLANG) || \
    defined(NN_BUILD_CONFIG_COMPILER_GHS) || \
    defined(NN_BUILD_CONFIG_COMPILER_OASIS_CAFE_CLANG)

    #define NN_USB_PACKED_ALIGNED_STRUCT_ATTRIBUTE(alignment)           \
        __attribute__((__packed__,__aligned__(alignment)))
    #define NN_USB_PACKED_STRUCT_ATTRIBUTE __attribute__((__packed__))
    #define NN_USB_PACKED_STRUCT_BEGIN()
    #define NN_USB_PACKED_STRUCT_END()
#else // VC

    #define NN_USB_PACKED_ALIGNED_STRUCT_ATTRIBUTE(alignment)
    #define NN_USB_PACKED_STRUCT_ATTRIBUTE
    #define NN_USB_PACKED_STRUCT_BEGIN() __pragma(pack(push, 1))
    #define NN_USB_PACKED_STRUCT_END()   __pragma(pack(pop))

#endif

#define NN_USB_DMA_ALIGN NN_ALIGNAS(::nn::usb::HwLimitDmaBufferAlignmentSize)

#define NN_USB_IS_DMA_ALIGNED(address)                                  \
    ((((uint64_t)(address)) &                                           \
      (::nn::usb::HwLimitDmaBufferAlignmentSize - 1)) == 0)

#define BmRequestType(type, recip, dir)                                 \
    (nn::usb::UsbCtrlXferReqTypeMask_Type##type   |                     \
     nn::usb::UsbCtrlXferReqTypeMask_Recip##recip |                     \
     nn::usb::UsbCtrlXferReqTypeMask_Dir##dir     )


namespace nn { namespace usb {

//----------------------------------------------------------------------------
// USB spec defined data structures and constants
//----------------------------------------------------------------------------

NN_USB_PACKED_STRUCT_BEGIN();

/**
 * @brief USB Class Codes
 *
 * @details
 *   USB-IF defined 1.0 Class Codes as listed in
 *   http://www.usb.org/developers/defined_class
 */
enum UsbClass
{
    UsbClass_PerInterface         = 0x00,
    UsbClass_Audio                = 0x01,
    UsbClass_Comm                 = 0x02,
    UsbClass_Hid                  = 0x03,
    UsbClass_Physical             = 0x05,
    UsbClass_StillImage           = 0x06,
    UsbClass_Printer              = 0x07,
    UsbClass_MassStorage          = 0x08,
    UsbClass_Hub                  = 0x09,
    UsbClass_CdcData              = 0x0a,
    UsbClass_SmartCard            = 0x0b,
    UsbClass_ContentSecurity      = 0x0d,
    UsbClass_Video                = 0x0e,
    UsbClass_Healthcare           = 0x0f,
    UsbClass_AudioVideo           = 0x10,
    UsbClass_Billboard            = 0x11,
    UsbClass_Diagnostic           = 0xdc,
    UsbClass_WirelessController   = 0xe0,
    UsbClass_Misc                 = 0xef,
    UsbClass_AppSpecific          = 0xfe,
    UsbClass_VendorSpecific       = 0xff,
};

/**
 * @brief USB Descriptor Types
 *
 * @details
 *   USB descriptor types as defined in USB 2.0 Spec Table 9-5 and USB 3.1
 *   Spec Tabld 9-6
 */
enum UsbDescriptorType
{
    UsbDescriptorType_Device                  = 1,
    UsbDescriptorType_Config                  = 2,
    UsbDescriptorType_String                  = 3,
    UsbDescriptorType_Interface               = 4,
    UsbDescriptorType_Endpoint                = 5,
    UsbDescriptorType_DeviceQualifier         = 6,
    UsbDescriptorType_OtherSpeedConfig        = 7,
    UsbDescriptorType_InterfacePower          = 8,
    UsbDescriptorType_Otg                     = 9,
    UsbDescriptorType_Debug                   = 10,
    UsbDescriptorType_InterfaceAssociation    = 11,
    UsbDescriptorType_Bos                     = 15,
    UsbDescriptorType_DeviceCapability        = 16,
    UsbDescriptorType_Hid                     = 33,
    UsbDescriptorType_Report                  = 34,
    UsbDescriptorType_Physical                = 35,
    UsbDescriptorType_Hub                     = 41,
    UsbDescriptorType_EndpointCompanion       = 48, // 3.x only
    UsbDescriptorType_IsocEndpointCompanion   = 49,
};

/**
 * @brief Common Descriptor Header
 *
 * @details
 *   All standard USB descriptors have these 2 fields at the beginning.
 */
struct UsbDescriptorHeader
{
    uint8_t  bLength;
    uint8_t  bDescriptorType;
} NN_USB_PACKED_STRUCT_ATTRIBUTE;

/**
 * @brief Standard Device Descriptor
 *
 * @details
 *   Standard Device Descriptor as defined in USB 2.0 Spec, Table 9-8
 */
struct UsbDeviceDescriptor
{
    uint8_t  bLength;
    uint8_t  bDescriptorType;
    uint16_t bcdUSB;
    uint8_t  bDeviceClass;
    uint8_t  bDeviceSubClass;
    uint8_t  bDeviceProtocol;
    uint8_t  bMaxPacketSize0;
    uint16_t idVendor;
    uint16_t idProduct;
    uint16_t bcdDevice;
    uint8_t  iManufacturer;
    uint8_t  iProduct;
    uint8_t  iSerialNumber;
    uint8_t  bNumConfigurations;
} NN_USB_PACKED_STRUCT_ATTRIBUTE;

/**
 * @brief Standard Configuration Descriptor
 *
 * @details
 *   Standard Configuration Descriptor as defined in USB 2.0 Spec, Table 9-10
 */
struct UsbConfigDescriptor
{
    uint8_t  bLength;
    uint8_t  bDescriptorType;
    uint16_t wTotalLength;
    uint8_t  bNumInterfaces;
    uint8_t  bConfigurationValue;
    uint8_t  iConfiguration;
    uint8_t  bmAttributes;
    uint8_t  bMaxPower;
} NN_USB_PACKED_STRUCT_ATTRIBUTE;

/**
 * @brief Standard Interface Descriptor
 *
 * @details
 *   Standard Interface Descriptor as defined in USB 2.0 Spec, Table 9-12
 */
struct UsbInterfaceDescriptor
{
    uint8_t  bLength;
    uint8_t  bDescriptorType;
    uint8_t  bInterfaceNumber;
    uint8_t  bAlternateSetting;
    uint8_t  bNumEndpoints;
    uint8_t  bInterfaceClass;
    uint8_t  bInterfaceSubClass;
    uint8_t  bInterfaceProtocol;
    uint8_t  iInterface;
} NN_USB_PACKED_STRUCT_ATTRIBUTE;

/**
 * @brief Standard Endpoint Descriptor
 *
 * @details
 *   Standard Endpoint Descriptor as defined in USB 2.0 Spec, Table 9-13
 */
struct UsbEndpointDescriptor
{
    uint8_t  bLength;
    uint8_t  bDescriptorType;
    uint8_t  bEndpointAddress;
    uint8_t  bmAttributes;
    uint16_t wMaxPacketSize;
    uint8_t  bInterval;
} NN_USB_PACKED_STRUCT_ATTRIBUTE;

/**
 * @brief Endpoint Companion descriptor
 *
 * @details
 *  Applies only to superspeed 3.x
 */
struct UsbEndpointCompanionDescriptor
{
    uint8_t  bLength;
    uint8_t  bDescriptorType;
    uint8_t  bMaxBurst;
    uint8_t  bmAttributes;
    uint16_t wBytesPerInterval;
} NN_USB_PACKED_STRUCT_ATTRIBUTE;

/**
 * @brief UNICODE String Descriptor
 *
 * @details
 *   UNICODE String Descriptor as defined in USB 2.0 Spec, Table 9-16
 */
struct UsbStringDescriptor
{
    uint8_t  bLength;
    uint8_t  bDescriptorType;
    uint16_t wData[DsLimitMaxNameSize]; /* UTF-16LE encoded */
} NN_USB_PACKED_STRUCT_ATTRIBUTE;

/**
 * @brief Standard Interface Association Descriptor
 *
 * @details
 *   Standard Interface Association Descriptor as defined in USB 3.1 Spec,
 *   Table 9-22
 */
struct UsbInterfaceAssociationDescriptor
{
    uint8_t  bLength;
    uint8_t  bDescriptorType;
    uint8_t  bFirstInterface;
    uint8_t  bInterfaceCount;
    uint8_t  bFunctionClass;
    uint8_t  bFunctionSubClass;
    uint8_t  bFunctionProtocol;
    uint8_t  iFunction;
} NN_USB_PACKED_STRUCT_ATTRIBUTE;

/**
 * @brief Standard Configuration Descriptor bmAttributes field bitmap
 */
enum UsbConfigDescriptorAttributeMask
{
    UsbConfigDescriptorAttributeMask_One       = (1 << 7),    // must be set
    UsbConfigDescriptorAttributeMask_SelfPower = (1 << 6),    // self powered
    UsbConfigDescriptorAttributeMask_Wakeup    = (1 << 5),    // can wakeup
    UsbConfigDescriptorAttributeMask_Battery   = (1 << 4),    // battery powered
};

/**
 * @brief Standard Endpoint Descriptor bEndpointAddress field encoding
 */
enum UsbEndpointAddressMask
{
    UsbEndpointAddressMask_EndpointNumber      = (15 << 0),
    UsbEndpointAddressMask_Dir                 = (1  << 7),
    UsbEndpointAddressMask_DirHostToDevice     = (0  << 7),
    UsbEndpointAddressMask_DirDeviceToHost     = (1  << 7),
};

/**
 * @brief Standard Endpoint Descriptor bmAttributes field bitmap
 */
enum UsbEndpointAttributeMask
{
    UsbEndpointAttributeMask_XferType          = (3 << 0),
    UsbEndpointAttributeMask_XferTypeControl   = (0 << 0),
    UsbEndpointAttributeMask_XferTypeIsoc      = (1 << 0),
    UsbEndpointAttributeMask_XferTypeBulk      = (2 << 0),
    UsbEndpointAttributeMask_XferTypeInt       = (3 << 0),
};

/**
 * @brief Standard Endpoint Descriptor wMaxPacketSize field encoding
 */
enum UsbEndpointMaxPacketSize
{
    UsbEndpointMaxPacketSize_PayloadShift      = 0,
    UsbEndpointMaxPacketSize_PayloadMask       = (0x7ff << UsbEndpointMaxPacketSize_PayloadShift),
    UsbEndpointMaxPacketSize_TxnPerUframeShift = 11,
    UsbEndpointMaxPacketSize_TxnPerUframeMask  = (3 << UsbEndpointMaxPacketSize_TxnPerUframeShift),
};

/**
 * @brief USB Device Requests
 *
 * @details
 *   Format of Setup Data as defined in USB 2.0 Spec, Table 9-2
 */
struct UsbCtrlRequest
{
    uint8_t  bmRequestType;
    uint8_t  bRequest;
    uint16_t wValue;
    uint16_t wIndex;
    uint16_t wLength;
} NN_USB_PACKED_STRUCT_ATTRIBUTE;

/**
 * @brief Standard Feature Selectors
 *
 * @details
 *   Standard Feature Selectors as defined in USB 2.0 Spec, Table 9-6
 */
enum UsbFeatureSelector
{
    UsbFeatureSelector_RemoteWakeup    = 1,
    UsbFeatureSelector_EndpointHalt    = 0,
    UsbFeatureSelector_TestMode        = 2,
};

/**
 * @brief Setup Data bmRequestType field bitmap
 */
enum UsbCtrlXferReqTypeMask
{
    UsbCtrlXferReqTypeMask_DirHostToDevice = (0  << 7),
    UsbCtrlXferReqTypeMask_DirDeviceToHost = (1  << 7),
    UsbCtrlXferReqTypeMask_Type            = (3  << 5),
    UsbCtrlXferReqTypeMask_TypeStandard    = (0  << 5),
    UsbCtrlXferReqTypeMask_TypeClass       = (1  << 5),
    UsbCtrlXferReqTypeMask_TypeVendor      = (2  << 5),
    UsbCtrlXferReqTypeMask_TypeReserved    = (3  << 5),
    UsbCtrlXferReqTypeMask_Recip           = (31 << 0),
    UsbCtrlXferReqTypeMask_RecipDevice     = (0  << 0),
    UsbCtrlXferReqTypeMask_RecipInterface  = (1  << 0),
    UsbCtrlXferReqTypeMask_RecipEndpoint   = (2  << 0),
    UsbCtrlXferReqTypeMask_RecipOther      = (3  << 0),
};

/**
 * @brief Standard Request Codes
 *
 * @details
 *   Standard Request Codes as defined in USB 2.0 Spec, Table 9-4
 */
enum UsbCtrlXferReq
{
    UsbCtrlXferReq_GetStatus        = 0x00,
    UsbCtrlXferReq_ClearFeature     = 0x01,
    UsbCtrlXferReq_SetFeature       = 0x03,
    UsbCtrlXferReq_SetAddress       = 0x05,
    UsbCtrlXferReq_GetDescriptor    = 0x06,
    UsbCtrlXferReq_SetDescriptor    = 0x07,
    UsbCtrlXferReq_GetConfiguration = 0x08,
    UsbCtrlXferReq_SetConfiguration = 0x09,
    UsbCtrlXferReq_GetInterface     = 0x0A,
    UsbCtrlXferReq_SetInterface     = 0x0B,
    UsbCtrlXferReq_SetHubDepth      = 0x0C,
    UsbCtrlXferReq_SetSel           = 0x30,
    UsbCtrlXferReq_SetIsocDelay     = 0x31,
};

/**
 * @brief UsbDevice States
 *
 * @details
 *    USB device states as defined in USB 2.0 Spec, Table 9-1
 */
enum UsbState
{
    UsbState_Detached,
    UsbState_Attached,
    UsbState_Powered,
    UsbState_Default,
    UsbState_Address,
    UsbState_Configured,
    UsbState_Suspended,
};

//----------------------------------------------------------------------------
// Types and constants used by USB APIs
//----------------------------------------------------------------------------

/**
 * @brief Size of the USB descriptors
 */
enum UsbDescriptorSize
{
    UsbDescriptorSize_Device               = sizeof(UsbDeviceDescriptor),
    UsbDescriptorSize_Config               = sizeof(UsbConfigDescriptor),
    UsbDescriptorSize_Interface            = sizeof(UsbInterfaceDescriptor),
    UsbDescriptorSize_Endpoint             = sizeof(UsbEndpointDescriptor),
    UsbDescriptorSize_EndpointCompanion    = sizeof(UsbEndpointCompanionDescriptor),
    UsbDescriptorSize_InterfaceAssociation = sizeof(UsbInterfaceAssociationDescriptor),
};

/**
 * @brief USB Complex ID
 *
 * @details
 *   The ID allocated to each USB complex in the system
 *
 *   - ComplexId_Stub: Implementation stub for unsupported platforms
 *   - ComplexId_Tegra124: TK1
 *   - ComplexId_Tegra21x: TX1
 *   - ComplexId_Count: Total number of complexes supported
 */
enum ComplexId
{
    ComplexId_Stub,

    ComplexId_Tegra124,
    ComplexId_Tegra21x,

    ComplexId_Count
};

/**
 * @brief Capability bitmasks for USB Controller
 *
 * @details
 *   - UsbCapability_Device: It's a device controller
 *   - UsbCapability_Host: It's a host controller
 *   - UsbCapability_LowSpeed: The controller is LowSpeed capable
 *   - UsbCapability_FullSpeed: The controller is FullSpeed capable
 *   - UsbCapability_HighSpeed: The controller is HighSpeed capable
 *   - UsbCapability_SuperSpeed: The controller is SuperSpeed capable
 *   - UsbCapability_SuperSpeedPlus: The controller is SuperSpeedPlus capable
 *   - UsbCapability_Invalid: Any genuine controller shouldn't have this capability
 */
enum UsbCapability {
    // role
    UsbCapability_Device         = 1 << 0,
    UsbCapability_Host           = 1 << 1,

    // speed
    UsbCapability_LowSpeed       = 1 << 2,
    UsbCapability_FullSpeed      = 1 << 3,
    UsbCapability_HighSpeed      = 1 << 4,
    UsbCapability_SuperSpeed     = 1 << 5,
    UsbCapability_SuperSpeedPlus = 1 << 6,

    UsbCapability_Invalid        = 1 << 31
};

/**
 * @brief USB endpoint directions
 */
enum UsbEndpointDirection
{
    UsbEndpointDirection_Invalid,
    UsbEndpointDirection_ToDevice, // "out" from host stack perspective
    UsbEndpointDirection_ToHost,   // "in" from host stack perspective
    UsbEndpointDirection_Control,  // control endpoint type designation
};

/**
 * @brief USB endpoint types
 */
enum UsbEndpointType
{
    UsbEndpointType_Invalid,
    UsbEndpointType_Control,
    UsbEndpointType_Isoc,
    UsbEndpointType_Bulk,
    UsbEndpointType_Int,
};

/**
 * @brief USB speed
 */
enum UsbDeviceSpeed
{
    UsbDeviceSpeed_Invalid = 0,
    UsbDeviceSpeed_Low,           /* USB 1.0 */
    UsbDeviceSpeed_Full,          /* USB 1.1 */
    UsbDeviceSpeed_High,          /* USB 2.0 */
    UsbDeviceSpeed_Super,         /* USB 3.0 */
    UsbDeviceSpeed_SuperPlus,     /* USB 3.1 */
};

/**
 * @brief Extract endpoint number from the endpoint descriptor
 */
static inline uint8_t UsbGetEndpointNumber(const UsbEndpointDescriptor *pDescriptor) NN_NOEXCEPT
{
    return pDescriptor->bEndpointAddress & UsbEndpointAddressMask_EndpointNumber;
}

/**
 * @brief Check if the direction of given endpoint is device to host
 */
static inline bool UsbEndpointIsDeviceToHost(const UsbEndpointDescriptor *pDescriptor) NN_NOEXCEPT
{
    return (pDescriptor->bEndpointAddress & UsbEndpointAddressMask_Dir) == UsbEndpointAddressMask_DirDeviceToHost;
}

/**
 * @brief Check if the direction of given endpoint is host to device
 */
static inline bool UsbEndpointIsHostToDevice(const UsbEndpointDescriptor *pDescriptor) NN_NOEXCEPT
{
    return (pDescriptor->bEndpointAddress & UsbEndpointAddressMask_Dir) == UsbEndpointAddressMask_DirHostToDevice;
}

/**
 * @brief Calculate the endpoint address from endpoint number and direction
 */
static inline uint8_t UsbGetEndpointAddress(uint8_t endpointNumber, UsbEndpointDirection direction) NN_NOEXCEPT
{
    uint8_t bEndpointAddress = (uint8_t)(endpointNumber & UsbEndpointAddressMask_EndpointNumber);
    if (direction == UsbEndpointDirection_ToHost)
    {
        bEndpointAddress |= UsbEndpointAddressMask_DirDeviceToHost;
    }
    else
    {
        bEndpointAddress |= UsbEndpointAddressMask_DirHostToDevice;
    }
    return bEndpointAddress;
}

/**
 * @brief Check if the given endpoint is a bulk endpoint
 */
static inline bool UsbEndpointIsBulk(const UsbEndpointDescriptor *pDescriptor) NN_NOEXCEPT
{
    return (pDescriptor->bmAttributes & UsbEndpointAttributeMask_XferType) == UsbEndpointAttributeMask_XferTypeBulk;
}

/**
 * @brief Check if the given endpoint is a control endpoint
 */
static inline bool UsbEndpointIsControl(const UsbEndpointDescriptor *pDescriptor) NN_NOEXCEPT
{
    return (pDescriptor->bmAttributes & UsbEndpointAttributeMask_XferType) == UsbEndpointAttributeMask_XferTypeControl;
}

/**
 * @brief Check if the given endpoint is an interrupt endpoint
 */
static inline bool UsbEndpointIsInt(const UsbEndpointDescriptor *pDescriptor) NN_NOEXCEPT
{
    return (pDescriptor->bmAttributes & UsbEndpointAttributeMask_XferType) == UsbEndpointAttributeMask_XferTypeInt;
}

/**
 * @brief Check if the given endpoint is an isochronous endpoint
 */
static inline bool UsbEndpointIsIsoc(const UsbEndpointDescriptor *pDescriptor) NN_NOEXCEPT
{
    return (pDescriptor->bmAttributes & UsbEndpointAttributeMask_XferType) == UsbEndpointAttributeMask_XferTypeIsoc;
}

/**
 * @brief Get the direction of given endpoint
 */
static inline UsbEndpointDirection UsbGetEndpointDirection(const UsbEndpointDescriptor *pDescriptor) NN_NOEXCEPT
{
    if (UsbEndpointIsDeviceToHost(pDescriptor))
    {
        return UsbEndpointDirection_ToHost;
    }
    else
    {
        return UsbEndpointDirection_ToDevice;
    }
}

/**
 * @brief Get the type of given endpoint
 */
static inline UsbEndpointType UsbGetEndpointType(const UsbEndpointDescriptor *pDescriptor) NN_NOEXCEPT
{
    UsbEndpointType endpointType = UsbEndpointType_Invalid;

    switch (pDescriptor->bmAttributes & UsbEndpointAttributeMask_XferType)
    {
    case UsbEndpointAttributeMask_XferTypeControl:
        endpointType = UsbEndpointType_Control;
        break;

    case UsbEndpointAttributeMask_XferTypeIsoc:
        endpointType = UsbEndpointType_Isoc;
        break;

    case UsbEndpointAttributeMask_XferTypeBulk:
        endpointType = UsbEndpointType_Bulk;
        break;

    case UsbEndpointAttributeMask_XferTypeInt:
        endpointType = UsbEndpointType_Int;
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }
    return endpointType;
}

/**
 * @brief Check if the given endpoint descriptor is valid
 */
static inline bool UsbEndpointIsValid(const UsbEndpointDescriptor *pDescriptor) NN_NOEXCEPT
{
    return
        pDescriptor &&
        pDescriptor->bLength >= UsbDescriptorSize_Endpoint &&
        pDescriptor->bEndpointAddress != 0;
}

/**
 * @brief Mark the given endpoint descriptor as invalid
 */
static inline void UsbMarkEndpointInvalid(UsbEndpointDescriptor *pDescriptor) NN_NOEXCEPT
{
    pDescriptor->bLength = 0;
    pDescriptor->bEndpointAddress = 0;
}

NN_USB_PACKED_STRUCT_END();

} // end of namespace usb
} // end of namespace nn

