﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/util/util_BitPack.h>

namespace nn {
namespace usb {
namespace pd {

/**
 * @brief SVID
 */
const uint16_t VdmStanderdId = 0xFF00;
const uint16_t VdmUnassignedId = 0x0000;
const uint16_t VdmNintendoId = 0x057E;

/**
 * @brief VDM ヘッダコマンド
 */
enum VdmHeaderCommand
{
    VdmHeaderCommand_None                                   = 0,
    VdmHeaderCommand_DiscoverIdentity                       = 1,        // Valid with SOP/SOP'
    VdmHeaderCommand_DiscoverSvIds                          = 2,        // Valid with SOP*
    VdmHeaderCommand_DiscoverModes                          = 3,        // Valid with SOP*
    VdmHeaderCommand_EnterMode                              = 4,        // Valid with SOP*
    VdmHeaderCommand_ExitMode                               = 5,        // Valid with SOP*
    VdmHeaderCommand_Attention                              = 6,        // Valid with SOP

    VdmHeaderCommand_SvIdCommandMin                         = 16,       // Valid with SOP*
    VdmHeaderCommand_SvIdCommandMax                         = 31        // Valid with SOP*
};


/**
 * @brief VDM タイプ
 */
enum VdmType
{
    VdmType_UnstructuredVdm                                 = 0,
    VdmType_StructuredVdm                                   = 1
};

/**
 * @brief VDM バージョン
 */
enum VdmVersion
{
    VdmVersion_1_0                                          = 0
};

/**
 * @brief VDO ポジション
 */
enum VdmObjectPosition
{
    VdmObjectPosition_Reserved                              = 0,
    VdmObjectPosition_1                                     = 1,
    VdmObjectPosition_2                                     = 2,
    VdmObjectPosition_3                                     = 3,
    VdmObjectPosition_4                                     = 4,
    VdmObjectPosition_5                                     = 5,
    VdmObjectPosition_6                                     = 6,

    VdmObjectPosition_ExitAllMode                           = 7
};

/**
 * @brief VDM ユニットのビット定義
 */
typedef nn::util::BitPack32 VdmUnit;

/**
 * @brief VDM ヘッダのビット定義
 */
struct VdmHeader : VdmUnit
{
    typedef nn::util::BitPack32::Field< 16, 16, Bit32 > SvId;           // USB Standard or Vendor ID
    typedef nn::util::BitPack32::Field< 15,  1, bool  > VdmType;        // 1 = Structured VDM
    typedef nn::util::BitPack32::Field< 13,  2, Bit32 > VdmVersion;
    typedef nn::util::BitPack32::Field<  8,  3, Bit32 > ObjectPosition; // For the Enter Mode and Exit Mode Commands
    typedef nn::util::BitPack32::Field<  6,  2, Bit32 > CommandType;
    typedef nn::util::BitPack32::Field<  0,  5, Bit32 > HeaderCommand;
};

/**
 * @brief VDO のビット定義
 */
typedef VdmUnit Vdo;

/**
 * @brief Vendor ID のビット定義
 */
typedef Bit16 VdmVendorId;

/**
 * @brief Product ID のビット定義
 */
typedef Bit16 VdmProductId;

/**
 * @brief ID ヘッダのビット定義
 */
struct VdmIdHeader : VdmUnit
{
    typedef nn::util::BitPack32::Field< 31,  1, bool  > UsbHostDataCapable;
    typedef nn::util::BitPack32::Field< 32,  1, bool  > UsbDeviceDataCapable;
    typedef nn::util::BitPack32::Field< 27,  3, Bit8  > ProductType;
    typedef nn::util::BitPack32::Field< 26,  1, bool  > ModalOperationSupported;
    typedef nn::util::BitPack32::Field<  0, 16, Bit16 > VendorId;
};

} // pd
} // usb
} // nn
