﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/time/detail/time_CommonDetail.h>

#include <nn/timesrv/detail/core/timesrv_EphemeralNetworkSystemClockCore.h>
#include <nn/timesrv/detail/core/timesrv_SharedMemoryManager.h>
#include <nn/timesrv/detail/core/timesrv_StandardSteadyClockCore.h>
#include <nn/timesrv/detail/core/timesrv_StandardLocalSystemClockCore.h>
#include <nn/timesrv/detail/core/timesrv_StandardUserSystemClockCore.h>
#include <nn/timesrv/detail/core/timesrv_StandardNetworkSystemClockCore.h>
#include <nn/timesrv/detail/core/timesrv_TickBasedSteadyClockCore.h>
#include <nn/timesrv/detail/core/timesrv_TimeZoneServiceCore.h>
#include <nn/timesrv/detail/core/timesrv_WorkerThread.h>
#include <nn/timesrv/detail/service/timesrv_HipcServerManager.h>

#include <nn/util/util_Optional.h>


namespace nn
{

namespace time
{
namespace detail
{
namespace service
{
    class ServiceProviderClient;
}
}
}

namespace timesrv
{
namespace detail
{


namespace core
{
    class StaticServiceServer;
}

namespace service
{

/**
 * @brief   PosixTime をFSへ通知するクラス
 * @details
 *  StandardUserSystemClockCore のセットアップが未完了の状態で時刻取得をしてしまわないように、
 *  Activate() 実行後に Notify() 内部での通知処理が有効になります。
 *  StandardUserSystemClockCore のセットアップ完了後に Activate() を行ってください。
 */
class PosixTimeNotifierToFs
{
public:
    PosixTimeNotifierToFs(
        core::StandardUserSystemClockCore* pStandardUserSystemClockCore,
        core::TimeZoneServiceCore* pTimeZoneServiceCore) NN_NOEXCEPT;
    void Activate() NN_NOEXCEPT;
    void Notify() NN_NOEXCEPT;

private:
    bool m_IsAvailable;
    core::StandardUserSystemClockCore* m_pStandardUserSystemClockCore;
    core::TimeZoneServiceCore* m_pTimeZoneServiceCore;

    nn::Result NotifyImpl() NN_NOEXCEPT;
};

//!< LocalSystemClock の SystemClockContext 永続化用コールバッククラス
class LocalSystemClockContextWriter : public core::SystemClockContextUpdateCallback
{
public:
    LocalSystemClockContextWriter(
        PosixTimeNotifierToFs* pPosixTimeNotifierToFs,
        core::WorkerThread* pWorkerThread,
        core::SharedMemoryManager* pSharedMemoryManager) NN_NOEXCEPT;
    virtual nn::Result Set(const nn::time::SystemClockContext& systemClockContext) NN_NOEXCEPT NN_OVERRIDE;
private:
    PosixTimeNotifierToFs* m_pPosixTimeNotifierToFs;
    core::WorkerThread* m_pWorkerThread;
    core::SharedMemoryManager* m_pSharedMemoryManager;
    nn::util::optional<nn::time::SystemClockContext> m_PreviousSystemClockContext;
};

//!< NetworkSystemClock の SystemClockContext 永続化用コールバッククラス
class NetworkSystemClockContextWriter : public core::SystemClockContextUpdateCallback
{
public:
    NetworkSystemClockContextWriter(
        PosixTimeNotifierToFs* pPosixTimeNotifierToFs,
        core::WorkerThread* pWorkerThread,
        core::SharedMemoryManager* pSharedMemoryManager) NN_NOEXCEPT;
    virtual nn::Result Set(const nn::time::SystemClockContext& systemClockContext) NN_NOEXCEPT NN_OVERRIDE;
private:
    PosixTimeNotifierToFs* m_pPosixTimeNotifierToFs;
    core::WorkerThread* m_pWorkerThread;
    core::SharedMemoryManager* m_pSharedMemoryManager;
    nn::util::optional<nn::time::SystemClockContext> m_PreviousSystemClockContext;
};

class ServiceProvider
{
    friend class nn::time::detail::service::ServiceProviderClient;
    friend class nn::timesrv::detail::service::StaticServiceServer;

private:
    class GeneralArranger
    {
    public:
        GeneralArranger() NN_NOEXCEPT;
        ~GeneralArranger() NN_NOEXCEPT;
    };

private:
    // ServiceProvider のメンバになるクラスは、コンストラクタの引数でポインタを与えられても
    // コンストラクタ内では、変数への格納以外、ポインタに対して操作をおこなってはいけない
    GeneralArranger m_GeneralArranger;

    core::StandardSteadyClockCore m_StandardSteadyClockCore;
    core::TickBasedSteadyClockCore m_TickBasedSteadyClockCore;

    core::StandardLocalSystemClockCore m_StandardLocalSystemClockCore;
    core::StandardNetworkSystemClockCore m_StandardNetworkSystemClockCore;
    core::StandardUserSystemClockCore m_StandardUserSystemClockCore;
    core::TimeZoneServiceCore m_TimeZoneServiceCore;
    core::EphemeralNetworkSystemClockCore m_EphemeralNetworkSystemClockCore;
    core::SharedMemoryManager m_SharedMemoryManager;

    core::WorkerThread m_WorkerThread;
    PosixTimeNotifierToFs m_PosixTimeNotifierToFs;
    LocalSystemClockContextWriter m_LocalSystemClockContextWriter;
    NetworkSystemClockContextWriter m_NetworkSystemClockContextWriter;

    HipcServerManager m_HipcServerManager;

#if NN_DETAIL_TIME_CONFIG_SERVICE_FRAMEWORK_MODEL == NN_DETAIL_TIME_CONFIG_SERVICE_FRAMEWORK_MODEL_HIPC
    void LoopService() NN_NOEXCEPT;
#endif

public:
    ServiceProvider() NN_NOEXCEPT;
    ~ServiceProvider() NN_NOEXCEPT;

    nn::Result Start() NN_NOEXCEPT;
    nn::Result RequestStopServer() NN_NOEXCEPT;

    void SetupStandardSteadyClockCore() NN_NOEXCEPT;
    void SetupStandardLocalSystemClockCore() NN_NOEXCEPT;
    void SetupStandardNetworkSystemClockCore() NN_NOEXCEPT;
    void SetupStandardUserSystemClockCore() NN_NOEXCEPT;
    void SetupTimeZoneServiceCore() NN_NOEXCEPT;
    void SetupEphemeralNetworkSystemClockCore() NN_NOEXCEPT;
};

}
}
}
}
