﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/time/detail/time_CommonDetail.h>

#include <nn/time.h>
#include <nn/time/time_TimeZoneRule.h>
#include <nn/time/time_TimeZoneRuleVersion.h>
#include <nn/timesrv/detail/service/timesrv_ITimeZoneService.sfdl.h> // nn::time::sf::CalendarAdditionalInfo

#include <nn/os/os_SdkMutex.h>

namespace nn
{
namespace timesrv
{

struct TimeZoneRuleInner;

namespace detail
{

namespace service
{
    class PosixTimeNotifierToFs;
}

namespace core
{

/**
 * @brief   タイムゾーン関連処理のコア実装
 * @details
 *  メンバ関数はスレッドセーフです。
 *  (FSプロセスへの時差情報通知で、IPCスレッドとは異なるスレッドからも叩かれるため)
 *
 *  本クラスは設定永続化の処理やファイル操作をいっさい行わず、
 *  インスタンス生成時に、クラス内の設定値は無効な値を保持します。@n
 *  例えば GetDeviceLocationName() が有効な値を返すためには、
 *  事前に SetDeviceLocationName() で有効な値をセットしておく必要があります。
 */
class TimeZoneServiceCore
{
public:
    explicit TimeZoneServiceCore(nn::timesrv::detail::service::PosixTimeNotifierToFs* pPosixTimeNotifierToFs) NN_NOEXCEPT;

    // GetDeviceLocationName 利用の前に SetDeviceLocationName されること前提
    void GetDeviceLocationName(nn::time::LocationName* pOutValue) const NN_NOEXCEPT;
    Result SetDeviceLocationName(const nn::time::LocationName& value) NN_NOEXCEPT;

    // GetTotalLocationNameCount 利用の前に SetTotalLocationNameCount されること前提
    void SetTotalLocationNameCount(int32_t count) NN_NOEXCEPT;
    int32_t GetTotalLocationNameCount() const NN_NOEXCEPT;

    // GetDeviceLocationUpdatedTime 利用の前に SetDeviceLocationUpdatedTime されること前提
    void SetDeviceLocationUpdatedTime(const nn::time::SteadyClockTimePoint& value);
    nn::time::SteadyClockTimePoint GetDeviceLocationUpdatedTime() const;

    Result LoadTimeZoneRule(
        nn::time::TimeZoneRule* pOut, const nn::time::LocationName& locationName) NN_NOEXCEPT;

    Result LoadLocationNameList(
        int32_t* pOutCount,
        nn::time::LocationName* pOutLocationNameList,
        size_t locationNameListCount,
        int32_t offset) NN_NOEXCEPT;

    // GetTimeZoneRuleVersion 利用の前に SetTimeZoneRuleVersion されること前提
    void SetTimeZoneRuleVersion(const nn::time::TimeZoneRuleVersion& version) NN_NOEXCEPT;
    void GetTimeZoneRuleVersion(nn::time::TimeZoneRuleVersion* pOut) const NN_NOEXCEPT;

    Result ToCalendarTime(
        nn::time::CalendarTime* pOutCalendarTime,
        nn::time::sf::CalendarAdditionalInfo* pOutCalendarAdditionalInfo,
        const nn::time::PosixTime& posixTime,
        const nn::time::TimeZoneRule& rule) NN_NOEXCEPT;

    Result ToCalendarTimeWithMyRule(
        nn::time::CalendarTime* pOutCalendarTime,
        nn::time::sf::CalendarAdditionalInfo* pOutCalendarAdditionalInfo,
        const nn::time::PosixTime& posixTime) NN_NOEXCEPT;

    Result ToPosixTime(
        int32_t* pOutCount,
        nn::time::PosixTime* pOutPosixTimeList,
        size_t outPosixTimeListCount,
        const nn::time::CalendarTime& calendarTime,
        const nn::time::TimeZoneRule& rule) NN_NOEXCEPT;

    Result ToPosixTimeWithMyRule(
        int32_t* pOutCount,
        nn::time::PosixTime* pOutPosixTimeList,
        size_t outPosixTimeListCount,
        const nn::time::CalendarTime& calendarTime) NN_NOEXCEPT;

private:
    mutable nn::os::SdkRecursiveMutex m_Lock;
    nn::time::LocationName m_OwnLocationName;
    nn::time::TimeZoneRule m_OwnTimeZoneRule;
    nn::time::SteadyClockTimePoint m_LocationUpdatedTime;
    int32_t m_LocationNameCount;
    nn::time::TimeZoneRuleVersion m_TimeZoneRuleVersion;
    nn::timesrv::detail::service::PosixTimeNotifierToFs* m_pPosixTimeNotifierToFs;

    const nn::timesrv::TimeZoneRuleInner& GetRuleInnerRef(const nn::time::TimeZoneRule& rule) const NN_NOEXCEPT
    {
        auto p = reinterpret_cast<const nn::timesrv::TimeZoneRuleInner*>(&rule);
        return *p;
    }

    nn::timesrv::TimeZoneRuleInner* GetRuleInner(nn::time::TimeZoneRule* pRule) const NN_NOEXCEPT
    {
        return reinterpret_cast<nn::timesrv::TimeZoneRuleInner*>(pRule);
    }
};

}
}
}
}
