﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
*   @file
*   @brief  メニュー向け API の宣言
*/

#pragma once

#include <nn/time/time_LocationName.h>
#include <nn/time/time_TimeZoneRuleVersion.h>


namespace nn
{
namespace time
{

//! @name メニュー向け API
//! @{

/**
 * @brief   本体メニュー向けの TIME ライブラリの初期化を行います。
 *
 * @return  処理の結果が返ります。事前条件を満たしていない場合、以下に列挙されていない失敗が返ることがあります。
 * @retval  ResultSuccess                       成功しました。
 *
 * @details
 *  本関数を使って TIME ライブラリを初期化した場合にも、破棄には nn::time::Finalize を利用してください。
 *  通常権限向けの初期化を行う @ref Initialize 関数と重ねて呼ぶことはできません。
 *
 *  ライブラリがすでに初期化された状態でも重ねて呼ぶことができますが、その場合、ライブラリの利用を完全に終了するにはこの関数を呼んだのと同じ回数だけ
 *  @ref nn::time::Finalize 関数を呼ぶ必要があります。
 *
 *  この関数を成功させるには特権が必要です。適切な権限がない場合、列挙されていない Result が返ることがあります。
 */
Result InitializeForMenu() NN_NOEXCEPT;

/**
 * @brief   ライブラリが対応している地域名のリストを取得します。
 * @param[out]  pOutCount                   格納された地域名の個数
 * @param[out]  pOutLocationNameList        地域名の格納先バッファ
 * @param[in]   count                       pOutLocationNameList の要素数
 * @param[in]   offset                      読み込みを開始する先頭からのオフセット数
 *
 * @pre
 *  - TIME ライブラリが初期化されている
 *  - pOutCount != nullptr
 *  - pOutLocationNameList != nullptr
 *  - 1 <= count
 *  - 0 <= offset
 *
 * @post
 *  - 0 <= *pOutCount
 *
 * @details
 *  地域名を最大 count 個まで pOutLocationNameList に格納し、
 *  格納した数を pOutCount に入れます。
 *
 *  本関数は最大数 msec ブロックする可能性があります。
 *
 * @see GetTotalLocationNameCount()
 */
void LoadLocationNameList(
    int* pOutCount,
    LocationName* pOutLocationNameList,
    int count,
    int offset
    ) NN_NOEXCEPT;

/**
 * @brief   ライブラリが対応している地域名の総数を返します。
 *
 * @pre
 *  - TIME ライブラリが初期化されている
 *
 * @details
 *  @ref LoadLocationNameList 関数で取得できる地域名　( @ref LocationName ) の総数を返します。
 */
int GetTotalLocationNameCount() NN_NOEXCEPT;

/**
 * @brief   デバイスの地域名を設定します。
 * @param[in]   value   地域名
 *
 * @pre
 *  - TIME ライブラリが nn::time::InitializeForMenu 関数で初期化されている
 *  - ライブラリが対応している地域名を指定している
 *
 * @details
 *  この関数を成功させるには、特権が必要です。
 *
 *  本関数による地域名の設定は、即座に以下の API の挙動に反映されます。
 *  - @ref GetDeviceLocationName 関数@n
 *    設定した地域名が返ります。
 *  - @ref TimeZoneRule を指定しない @ref ToCalendarTime 関数, @ref ToPosixTime 関数@n
 *    設定した地域名のタイムゾーンルールが内部で利用されます。
 */
void SetDeviceLocationName(const LocationName& value) NN_NOEXCEPT;

/**
 * @brief   タイムゾーンルールのバージョンを取得します。
 * @param[out]  pOut    バージョンの格納先バッファ
 *
 * @pre
 *  - TIME ライブラリが初期化されている
 *  - pOut != nullptr
 * @post
 *  - pOut->data は NULL 終端している
 *
 * @details
 *  システムが管理しているタイムゾーンルールのバージョンを取得します。
 */
void GetTimeZoneRuleVersion(TimeZoneRuleVersion* pOut) NN_NOEXCEPT;

/**
 * @brief   ユーザー時計の初期値となる西暦を取得します。
 * @return  西暦
 *
 * @pre
 *  - TIME ライブラリが初期化されている
 *
 * @details
 *
 *  ユーザー時計( nn::time::StandardUserSystemClock )の初期値となる西暦を取得します。
 *  ここで言う初期値とは、デバイスの初回起動時や、RTCリセット発生後などにユーザー時計がリセットされる日時を指します。
 */
int GetStandardUserSystemClockInitialYear() NN_NOEXCEPT;

//! @}
}
}

#include <nn/time.h>
#include <nn/time/time_StandardLocalSystemClockPrivilegeApi.h>
#include <nn/time/time_StandardUserSystemClockPrivilegeApi.h>
#include <nn/time/time_ApiForEnsureNetworkClockAvailability.h>
