﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/ahid/ahid.h>
#include <nn/stdfu/stdfu_DfuUpdateStatus.h>


/**
 * @file
 * @brief   STDFU API
 */

namespace nn {
namespace stdfu{

enum UpdateParameters
{
    UpdateParameters_IdFromBinary = 0xffff,
};

class StDfu
{

    NN_DISALLOW_COPY(StDfu);
    NN_DISALLOW_MOVE(StDfu);

public:

    StDfu() NN_NOEXCEPT {};
    ~StDfu() NN_NOEXCEPT {};

    /**
     * @brief Initialize stdfu library
     *
     * @retresult
     *   @handleresult{nn::ResultSuccess, Initialize succeeded}
     * @endretresult
     *
     * @details
     *  This method does nothing, it is here only for consistency in API style/
     */
    Result Initialize() NN_NOEXCEPT;

    /**
     * @brief Finalize stdfu library
     *
     * @retresult
     *   @handleresult{nn::ResultSuccess, Finalize succeeded}
     * @endretresult
     *
     * @details
     *  This method does nothing, it is here only for consistency in API style/
     */
    Result Finalize() NN_NOEXCEPT;

    /**
    * @brief Update DfuDevice firmware
    *
    * @param[in]  pBinary
    *   Pointer to DFU binary.
    *
    * @param[in]  totalBytes
    *   Size of DFU binary in bytes.
    *
    * @param[in]  vid
    *   Vendor ID of device to update, use UpdateParameters::UpdateParameters_IdFromBinary unless ID is other than intended by data from binary.
    *
    * @param[in]  pid
    *   Product ID of device to update, use UpdateParameters::UpdateParameters_IdFromBinary unless ID is other than intended by data from binary.
    *
    * @param[in]  bcd
    *   Device BCD of device to update, use UpdateParameters::UpdateParameters_IdFromBinary unless ID is other than intended by data from binary.
    *
    * @retresult
    *   @handleresult{nn::ResultSuccess, Operation succeeded}
    *   @handleresult{nn::stdfu::ResultInvalidDfuPrefixSignature, DFUPrefix signature is invalid, DFU binary is corrupt}
    *   @handleresult{nn::stdfu::ResultInvalidDfuPrefixVersion, DFUPrefix version, DFU binary is corrupt}
    *   @handleresult{nn::stdfu::ResultInvalidDfuPrefixImageSize, DFUPrefix image size is invalid, DFU binary is corrupt}
    *   @handleresult{nn::stdfu::ResultInvalidDfuPrefixTargets, DFUPrefix number of targets is invalid, DFU binary is corrupt}
    *   @handleresult{nn::stdfu::ResultInvalidDfuSuffixSignature, DFUSuffix signature is invalid, DFU binary is corrupt}
    *   @handleresult{nn::stdfu::ResultInvalidDfuSuffixCrc, DFUSuffix CRC is invalid, DFU binary is corrupt}
    *   @handleresult{nn::stdfu::ResultInvalidDfuImageSignature, DFU Image signature is invalid, DFU binary is corrupt}
    *   @handleresult{nn::stdfu::ResultDfuImageNotFound, DFU file does not contain image, DFU binary is corrupt}
    *   @handleresult{nn::stdfu::ResultDfuElementNotFound, DFU file does not contain element, DFU binary is corrupt}
    *   @handleresult{nn::stdfu::ResultInvalidDfuState, Unexpected DFU state encountered during update, update is not successful}
    *   @handleresult{nn::stdfu::ResultInvalidDfuStatus, Unexpected DFU status encountered during update, update is not successful}
    *   @handleresult{nn::stdfu::ResultDfuStatusTimeout, Unexpected DFU timeout during update, update is not successful}
    *   @handleresult{nn::usb::ResultNotInitialized, This object is not initialized}
    *   @handleresult{nn::usb::ResultTransferDescriptorStall, Control transfers complete with stall if device is logically unable to complete the request}
    *   @handleresult{nn::usb::ResultInterfaceInvalid, Device in which this interface is contained has disconnected}
    * @endretresult
    *
    * @pre
    *
    * @post
    *   If successful, firmware is updated to the target device.
    *
    * @details
    *   Update target dfu device firmware with DFU binary.
    */
    Result UpdateDfuDevice(void *pBinary, uint32_t bytes, uint16_t vid, uint16_t pid, uint16_t bcd) NN_NOEXCEPT;

    /**
     * @brief Update AhidDevice firmware
     *
     * @param[in]  pBinary
     *   Pointer to DFU binary.
     *
     * @param[in]  totalBytes
     *   Size of DFU binary in bytes.
     *
     * @param[in]  vid
     *   Vendor ID of device to update, use UpdateParameters::UpdateParameters_IdFromBinary unless ID is other than intended by data from binary.
     *
     * @param[in]  pid
     *   Product ID of device to update, use UpdateParameters::UpdateParameters_IdFromBinary unless ID is other than intended by data from binary.
     *
     * @param[in]  bcd
     *   Device BCD of device to update, use UpdateParameters::UpdateParameters_IdFromBinary unless ID is other than intended by data from binary.
     *
     * @retresult
     *   @handleresult{nn::ResultSuccess, Operation succeeded}
     *   @handleresult{nn::stdfu::ResultInvalidDfuPrefixSignature, DFUPrefix signature is invalid, DFU binary is corrupt}
     *   @handleresult{nn::stdfu::ResultInvalidDfuPrefixVersion, DFUPrefix version, DFU binary is corrupt}
     *   @handleresult{nn::stdfu::ResultInvalidDfuPrefixImageSize, DFUPrefix image size is invalid, DFU binary is corrupt}
     *   @handleresult{nn::stdfu::ResultInvalidDfuPrefixTargets, DFUPrefix number of targets is invalid, DFU binary is corrupt}
     *   @handleresult{nn::stdfu::ResultInvalidDfuSuffixSignature, DFUSuffix signature is invalid, DFU binary is corrupt}
     *   @handleresult{nn::stdfu::ResultInvalidDfuSuffixCrc, DFUSuffix CRC is invalid, DFU binary is corrupt}
     *   @handleresult{nn::stdfu::ResultInvalidDfuImageSignature, DFU Image signature is invalid, DFU binary is corrupt}
     *   @handleresult{nn::stdfu::ResultDfuImageNotFound, DFU file does not contain image, DFU binary is corrupt}
     *   @handleresult{nn::stdfu::ResultDfuElementNotFound, DFU file does not contain element, DFU binary is corrupt}
     *   @handleresult{nn::stdfu::ResultInvalidDfuState, Unexpected DFU state encountered during update, update is not successful}
     *   @handleresult{nn::stdfu::ResultInvalidDfuStatus, Unexpected DFU status encountered during update, update is not successful}
     *   @handleresult{nn::stdfu::ResultDfuStatusTimeout, Unexpected DFU timeout during update, update is not successful}
     *   @handleresult{nn::usb::ResultNotInitialized, This object is not initialized}
     *   @handleresult{nn::usb::ResultTransferDescriptorStall, Control transfers complete with stall if device is logically unable to complete the request}
     *   @handleresult{nn::usb::ResultInterfaceInvalid, Device in which this interface is contained has disconnected}
     * @endretresult
     *
     * @pre
     *   Target AHID interface IS NOT initialized.
     *
     * @post
     *   If successful, firmware is updated to the target device.
     *
     * @details
     *   Update target device firmware with DFU binary.
     */
    Result Update(void *pBinary, uint32_t totalBytes, uint16_t vid, uint16_t pid, uint16_t bcd) NN_NOEXCEPT;

    /**
     * @brief Update AhidDevice firmware
     *
     * @param[in]  pBinary
     *   Pointer to DFU binary.
     *
     * @param[in]  totalBytes
     *   Size of DFU binary in bytes.
     *
     * @param[in]  pAhid
     *   Pointer to caller initialized nn::ahid::Ahid.
     *
     * @retresult
     *   @handleresult{nn::ResultSuccess, Operation succeeded}
     *   @handleresult{nn::stdfu::ResultInvalidDfuPrefixSignature, DFUPrefix signature is invalid, DFU binary is corrupt}
     *   @handleresult{nn::stdfu::ResultInvalidDfuPrefixVersion, DFUPrefix version, DFU binary is corrupt}
     *   @handleresult{nn::stdfu::ResultInvalidDfuPrefixImageSize, DFUPrefix image size is invalid, DFU binary is corrupt}
     *   @handleresult{nn::stdfu::ResultInvalidDfuPrefixTargets, DFUPrefix number of targets is invalid, DFU binary is corrupt}
     *   @handleresult{nn::stdfu::ResultInvalidDfuSuffixSignature, DFUSuffix signature is invalid, DFU binary is corrupt}
     *   @handleresult{nn::stdfu::ResultInvalidDfuSuffixCrc, DFUSuffix CRC is invalid, DFU binary is corrupt}
     *   @handleresult{nn::stdfu::ResultInvalidDfuImageSignature, DFU Image signature is invalid, DFU binary is corrupt}
     *   @handleresult{nn::stdfu::ResultDfuImageNotFound, DFU file does not contain image, DFU binary is corrupt}
     *   @handleresult{nn::stdfu::ResultDfuElementNotFound, DFU file does not contain element, DFU binary is corrupt}
     *   @handleresult{nn::stdfu::ResultInvalidDfuState, Unexpected DFU state encountered during update, update is not successful}
     *   @handleresult{nn::stdfu::ResultInvalidDfuStatus, Unexpected DFU status encountered during update, update is not successful}
     *   @handleresult{nn::stdfu::ResultDfuStatusTimeout, Unexpected DFU timeout during update, update is not successful}
     *   @handleresult{nn::usb::ResultNotInitialized, This object is not initialized}
     *   @handleresult{nn::usb::ResultTransferDescriptorStall, Control transfers complete with stall if device is logically unable to complete the request}
     *   @handleresult{nn::usb::ResultInterfaceInvalid, Device in which this interface is contained has disconnected}
     * @endretresult
     *
     * @pre
     *   Target AHID interface IS initialized.
     *
     * @post
     *   If successful, firmware is updated to the target device.
     *
     * @details
     *   Update target device firmware with DFU binary.
     */
    Result Update(void *pBinary, uint32_t bytes, nn::ahid::Ahid *pAhid) NN_NOEXCEPT;

    /**
     * @brief Get status of firmware update
     *
     * @param[in]  pUpdateStatus
     *   Pointer to UpdateStatus.
     *     *
     * @retresult
     *   @handleresult{nn::ResultSuccess, Operation succeeded}
     * @endretresult
     *
     * @pre
     *   StDfu is initialized.
     *
     * @post
     *   Firmware update status is written to pUpdateStatus.
     *
     * @details
     *   Get status of firmware update. This should be called from another thread during Update().
     */
    Result GetUpdateStatus(UpdateStatus *pUpdateStatus) NN_NOEXCEPT;

private:

    DfuUpdateStatus m_DfuUpdateStatus;
};

} // end of namespace stdfu
} // end of namespace nn
