﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/srepo/detail/service/srepo_Common.h>
#include <nn/os/os_SystemEvent.h>

namespace nn { namespace srepo { namespace detail { namespace service { namespace core {

/*!
    @brief      レポートデータのバッファリングモジュールです。
*/
class ReportBuffer
{
private:
    NN_DISALLOW_COPY(ReportBuffer);
    NN_DISALLOW_MOVE(ReportBuffer);

private:
    /*!
        @brief      コンストラクタです。
    */
    ReportBuffer() NN_NOEXCEPT;

public:
    /*!
        @brief      インスタンスを取得します。

        @return     インスタンス。
    */
    static ReportBuffer& GetInstance() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(ReportBuffer, s_Instance);
        return s_Instance;
    }

public:
    /*!
        @brief      バッファにレポートデータが追加されたときに、シグナルされるシステムイベントの ReadableHandle を取得します。

        @return     処理結果。
    */
    nn::os::NativeHandle GetPushEventReadableHandle() NN_NOEXCEPT;

    /*!
        @brief      レポートデータを追加します。

        @param[in]  category    分類。
        @param[in]  uid         ユーザーアカウント。
        @param[in]  eventId     イベント ID。
        @param[in]  appId       アプリケーション ID。
        @param[in]  data        データ。
        @param[in]  dataSize    データのサイズ。

        @return     処理結果。

        @pre
            - eventId != nullptr
            - appId != nn::ApplicationId::GetInvalidId()
            - data != nullptr
            - dataSize > 0

        @details
                    バッファに空きが無い場合は、即座に返ります。
    */
    nn::Result Push(ReportCategory category, const nn::account::Uid& uid, const char* eventId, const nn::ApplicationId& appId, const void* data, size_t dataSize) NN_NOEXCEPT;

    /*!
        @brief      バッファに保存されたレポートデータを 1 個取り出します。

        @param[out] outCategory             取り出した分類。
        @param[out] outUid                  取り出したユーザーアカウント。
        @param[out] outEventIdBuffer        取り出したイベント ID。
        @param[out] outAppId                取り出したアプリケーション ID。
        @param[out] outDataSize             取り出したデータのサイズ。
        @param[out] outDataBuffer           取り出したデータ。
        @param[in]  outEventIdBufferSize    @a outEventIdBuffer のサイズ。
        @param[in]  outDataBufferSize       @a outDataBuffer のサイズ。

        @return     処理結果。

        @pre
            - outCategory != nullptr
            - outUid != nullptr
            - outEventIdBuffer != nullptr
            - outAppId != nullptr
            - outDataSize != nullptr
            - outDataBuffer != nullptr
            - outEventIdBufferSize > 0
            - outDataBufferSize > 0

        @details
                    取り出したレポートはバッファから削除されます。
    */
    nn::Result Pop(ReportCategory* outCategory, nn::account::Uid* outUid, char* outEventIdBuffer, nn::ApplicationId* outAppId, size_t* outDataSize, void* outDataBuffer, size_t outEventIdBufferSize, size_t outDataBufferSize) NN_NOEXCEPT;

private:
    /*!
        @brief      バッファサイズです。
    */
    static const size_t BufferSize = 128 * 1024;

    /*!
        @brief      バッファです。
    */
    struct Buffer
    {
        Bit8 buffer[BufferSize]; //!< バッファ。
        size_t filledSize;       //!< 書き込んだバイト数。

        /*!
            @brief      書き込み位置のポインタを取得します。

            @return     書き込み位置のポインタ。
        */
        inline Bit8* GetWritePointer() NN_NOEXCEPT
        {
            return &(buffer[filledSize]);
        }

        /*!
            @brief      データの残りサイズを取得します。

            @return     データの残りサイズ。
        */
        inline size_t GetRemainSize() const NN_NOEXCEPT
        {
            return BufferSize - filledSize;
        }
    };

private:
    //
    nn::os::SdkMutex m_Mutex;
    //
    Buffer m_Buffer;
    //
    nn::os::SystemEvent m_PushEvent;
};

}}}}}
