﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>
#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/spsm/detail/spsm_Log.h>

namespace nn { namespace spsm { namespace server {

    template<typename T, int MessageIdMin, int MessageIdMax>
    class UniqueMessageQueue
    {
        NN_DISALLOW_COPY(UniqueMessageQueue);
        NN_DISALLOW_MOVE(UniqueMessageQueue);

    public:
        UniqueMessageQueue() NN_NOEXCEPT
        {
            NN_STATIC_ASSERT(0 <= MessageIdMin);
            NN_STATIC_ASSERT(0 < MessageIdMax - MessageIdMin);
            for ( auto&& e : m_MessageBuffer )
            {
                e.isQueued = false;
            }
            ResetMessageNodeCount();
        }

        void Enqueue(const T& message) NN_NOEXCEPT
        {
            std::lock_guard<decltype(m_QueueLock)> lock(m_QueueLock);

            // 対応するメッセージバッファに上書き
            PushMessage(message);
        }

        T Dequeue() NN_NOEXCEPT
        {
            T* pPopped = nullptr;
            while ( NN_STATIC_CONDITION(true) )
            {
                // 受信待ち中は m_QueueLock をロックしない
                m_NonEmptyInternalEvent.Wait();
                std::lock_guard<decltype(m_QueueLock)> lock(m_QueueLock);

                // データを取り出してノードを解放
                pPopped = PopMessage();
                if ( !pPopped )
                {
                    // 複数スレッドが同時待ちするユースケースや ResetQueue とかぶさった時などでここで nullptr になりうる
                    continue;
                }
                return *pPopped;
            }
        }

        bool TryDequeue(T* pOutMessage) NN_NOEXCEPT
        {
            NN_SDK_ASSERT_NOT_NULL(pOutMessage);
            T* pPopped = nullptr;
            while ( NN_STATIC_CONDITION(true) )
            {
                // 受信トライ中は m_QueueLock をロックしない
                if ( !m_NonEmptyInternalEvent.TryWait() )
                {
                    return false;
                }
                std::lock_guard<decltype(m_QueueLock)> lock(m_QueueLock);

                // データを取り出してノードを解放
                pPopped = PopMessage();
                if ( !pPopped )
                {
                    // 複数スレッドが同時待ちするユースケースや ResetQueue とかぶさった時などでここで nullptr になりうる
                    continue;
                }
                *pOutMessage = *pPopped;
                return true;
            }
        }

        void Cancel(const T& message) NN_NOEXCEPT
        {
            std::lock_guard<decltype(m_QueueLock)> lock(m_QueueLock);

            auto& messageNode = GetMessageBuffer(message);
            if ( messageNode.isQueued )
            {
                //NN_DETAIL_SPSM_INFO("Canceled %d\n", static_cast<int>(message));
                messageNode.isQueued = false;
                DecMessageNodeCount();
            }
        }

        void ResetQueue() NN_NOEXCEPT
        {
            std::lock_guard<decltype(m_QueueLock)> lock(m_QueueLock);
            for ( auto&& e : m_MessageBuffer )
            {
                e.isQueued = false;
            }
            ResetMessageNodeCount();
        }

        os::SystemEvent* GetNonEmptyEvent() NN_NOEXCEPT
        {
            return &m_NonEmptyEvent;
        }

    private:
        struct MessageNode
        {
            T body;
            bool isQueued;
        };

        MessageNode& GetMessageBuffer(const T& message) NN_NOEXCEPT
        {
            // T が int に暗黙キャスト可能な型であることをダックタイピング的に要求
            NN_ABORT_UNLESS_RANGE(static_cast<int>(message), MessageIdMin, MessageIdMax);
            return m_MessageBuffer[message - MessageIdMin];
        }

        void PushMessage(const T& message) NN_NOEXCEPT
        {
            auto& messageNode = GetMessageBuffer(message);

            // すでに入っているメッセージは上書きする
            messageNode.body = message;

            if ( !messageNode.isQueued )
            {
                messageNode.isQueued = true;
                IncMessageNodeCount();
            }
        }

        T* PopMessage() NN_NOEXCEPT
        {
            // T を int にキャスト可能で、それがソート用 index として使用される
            for ( auto&& messageNode : m_MessageBuffer )
            {
                if ( messageNode.isQueued )
                {
                    messageNode.isQueued = false;
                    DecMessageNodeCount();
                    return &messageNode.body;
                }
            }
            return nullptr;
        }

        void IncMessageNodeCount() NN_NOEXCEPT
        {
            ++m_QueuedMessageCount;
            // メッセージが一つ以上あればシグナル
            if ( m_QueuedMessageCount > 0 )
            {
                // NN_DETAIL_SPSM_INFO_V3("%s: Signaling\n", __FUNCTION__);
                m_NonEmptyInternalEvent.Signal();
                m_NonEmptyEvent.Signal();
            }
        }

        void DecMessageNodeCount() NN_NOEXCEPT
        {
            --m_QueuedMessageCount;
            if ( m_QueuedMessageCount == 0 )
            {
                // NN_DETAIL_SPSM_INFO_V3("%s: Clearing\n", __FUNCTION__);
                m_NonEmptyInternalEvent.Clear();
                m_NonEmptyEvent.Clear();
            }
        }

        void ResetMessageNodeCount() NN_NOEXCEPT
        {
            m_QueuedMessageCount = 0;
            // NN_DETAIL_SPSM_INFO_V3("%s: Clearing\n", __FUNCTION__);
            m_NonEmptyInternalEvent.Clear();
            m_NonEmptyEvent.Clear();
        }

    private:
        MessageNode m_MessageBuffer[MessageIdMax - MessageIdMin];
        os::Mutex m_QueueLock{ false };
        int m_QueuedMessageCount{ 0 };

        os::LightEvent m_NonEmptyInternalEvent{ os::EventClearMode_ManualClear };
        os::SystemEvent m_NonEmptyEvent{ os::EventClearMode_ManualClear, true };
    };

}}}
