﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       システムアプリケーションの設定に関する API の宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/settings/system/settings_Common.h>
#include <nn/time/time_SteadyClockTimePoint.h>
#include <nn/util/util_BitFlagSet.h>

namespace nn { namespace settings { namespace system {

/**
 * @brief       初回起動設定の制御フラグ定義です。
 */
struct InitialLaunchFlag
{
    typedef ::nn::util::BitFlagSet<32, InitialLaunchFlag>::Flag<0>
            IsCompleted;    //!< 初回起動が完了しているか否か
    typedef ::nn::util::BitFlagSet<32, InitialLaunchFlag>::Flag<8>
            IsUserAddable;  //!< ユーザ追加が可能なシーンまで進んだか否か
    typedef ::nn::util::BitFlagSet<32, InitialLaunchFlag>::Flag<16>
            HasTimeStamp;   //!< 完了時刻が設定されているか否か
};

/**
 * @brief       初回起動設定の制御フラグ集合を扱う型です。
 */
typedef ::nn::util::BitFlagSet<32, InitialLaunchFlag> InitialLaunchFlagSet;

/**
 * @brief       初回起動設定を表す構造体です。
 */
struct InitialLaunchSettings
{
    InitialLaunchFlagSet flags;                 //!< 制御フラグ
    NN_PADDING4;
    ::nn::time::SteadyClockTimePoint timeStamp; //!< 完了時刻
};

/**
 * @brief       本体の識別名を表す構造体です。
 */
struct DeviceNickName
{
    char string[128];   //!< 本体の識別名
};

/**
 * @brief       アプレット起動の制御フラグ定義です。
 */
struct AppletLaunchFlag
{
    typedef ::nn::util::BitFlagSet<32, AppletLaunchFlag>::Flag<0>
            Gift;       //!< gift
    typedef ::nn::util::BitFlagSet<32, AppletLaunchFlag>::Flag<1>
            Migration;  //!< migration
};

/**
 * @brief       アプレット起動の制御フラグ集合を扱う型です。
 */
typedef ::nn::util::BitFlagSet<32, AppletLaunchFlag> AppletLaunchFlagSet;

/**
 * @brief       携帯モード時誤操作防止画面の有無を取得します。
 *
 * @return      携帯モード時誤操作防止画面の有無を表す値です。
 */
bool GetLockScreenFlag() NN_NOEXCEPT;

/**
 * @brief       携帯モード時誤操作防止画面の有無を設定します。
 *
 * @param[in]   value                       携帯モード時誤操作防止画面の有無を表す値
 */
void SetLockScreenFlag(bool value) NN_NOEXCEPT;

/**
 * @brief       カラーセット ID を返します。
 *
 * @return      カラーセット ID です。
 *              カラーセット ID が設定されていない場合は 0 を返します。
 */
int32_t GetColorSetId() NN_NOEXCEPT;

/**
 * @brief       カラーセット ID を設定します。
 *
 * @param[in]   value                       カラーセット ID
 */
void SetColorSetId(int32_t value) NN_NOEXCEPT;

/**
 * @brief       バッテリー残量のパーセンテージ表示を行うか否か表す値を返します。
 *
 * @return      バッテリー残量のパーセンテージ表示を行うか否か表す値です。
 */
bool GetBatteryPercentageFlag() NN_NOEXCEPT;

/**
 * @brief       バッテリー残量のパーセンテージ表示を行うか否か表す値を設定します。
 *
 * @param[in]   value                       バッテリー残量のパーセンテージ表示を行うか否か表す値
 */
void SetBatteryPercentageFlag(bool value) NN_NOEXCEPT;

/**
 * @brief       初回起動設定を取得します。
 *
 * @param[out]  pOutValue                   初回起動設定を読み出すバッファ
 *
 * @pre
 *              - pOutValue != nullptr
 */
void GetInitialLaunchSettings(InitialLaunchSettings* pOutValue) NN_NOEXCEPT;

/**
 * @brief       初回起動設定を変更します。
 *
 * @param[in]   value                       初回起動設定
 */
void SetInitialLaunchSettings(const InitialLaunchSettings& value) NN_NOEXCEPT;

/**
 * @brief       本体の識別名を取得します。
 *
 * @param[out]  pOutValue                   本体の識別名を読み出すバッファ
 *
 * @pre
 *              - pOutValue != nullptr
 */
void GetDeviceNickName(DeviceNickName* pOutValue) NN_NOEXCEPT;

/**
 * @brief       本体の識別名を変更します。
 *
 * @param[in]   value                       本体の識別名
 */
void SetDeviceNickName(const DeviceNickName& value) NN_NOEXCEPT;

/**
 * @brief       アプレット起動の制御フラグを取得します。
 *
 * @param[out]  pOutValue                   アプレット起動の制御フラグを読み出すバッファ
 *
 * @pre
 *              - pOutValue != nullptr
 */
void GetAppletLaunchFlags(AppletLaunchFlagSet* pOutValue) NN_NOEXCEPT;

/**
 * @brief       アプレット起動の制御フラグを変更します。
 *
 * @param[in]   value                       アプレット起動の制御フラグ
 */
void SetAppletLaunchFlags(const AppletLaunchFlagSet& value) NN_NOEXCEPT;

/**
 * @brief       Quest メニューを起動するか否かを表す値を返します。
 *
 * @return      Quest メニューを起動するか否かを表す値です。
 */
bool GetQuestFlag() NN_NOEXCEPT;

/**
 * @brief       Quest メニューを起動するか否かを表す値を設定します。
 *
 * @param[in]   value                       Quest メニューを起動するか否かを表す値
 */
void SetQuestFlag(bool value) NN_NOEXCEPT;

/**
 * @brief       無線認証画像のファイルサイズを返します。
 *
 * @return      無線認証画像のファイルサイズです。
 *              ファイルが存在しない場合は 0 を返します。
 */
size_t GetWirelessCertificationFileSize() NN_NOEXCEPT;

/**
 * @brief       無線認証画像を取得します。
 *
 * @details     GetWirelessCertificationFileSize() で得られるファイルサイズ
 *              以上のバッファを渡すことでファイル全体を取得することができます。
 *              ファイルサイズより小さいバッファが指定された場合は
 *              バッファサイズ分だけをファイル先頭から取得します。
 *              ファイルサイズより大きいバッファが指定された場合は
 *              余った領域に対しては何も行いません。
 *
 * @param[out]  outBuffer                   無線認証画像の読み出し先
 * @param[in]   size                        outBuffer のバイトサイズ
 *
 * @return      取得した無線認証画像のサイズです。
 *
 * @pre
 *              - outBuffer != nullptr
 */
size_t GetWirelessCertificationFile(void* outBuffer, size_t size) NN_NOEXCEPT;

}}} // nn::settings::system
