﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       スリープ設定に関する API の宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/settings/system/settings_Common.h>
#include <nn/util/util_BitFlagSet.h>

namespace nn { namespace settings { namespace system {

/**
 * @brief       スリープ設定の制御フラグ定義です。
 */
struct SleepFlag
{
    typedef ::nn::util::BitFlagSet<32, SleepFlag>::Flag<0>
            SleepsWhilePlayingMedia; //!< メディア再生時に自動スリープするか否か
};

/**
 * @brief       スリープ設定の制御フラグ集合を扱う型です。
 */
typedef ::nn::util::BitFlagSet<32, SleepFlag> SleepFlagSet;

/**
 * @brief       携帯モード時の自動スリープまでの時間を表す列挙型です。
 */
enum HandheldSleepPlan
{
    HandheldSleepPlan_1Min,     //!< 1 分
    HandheldSleepPlan_3Min,     //!< 3 分
    HandheldSleepPlan_5Min,     //!< 5 分
    HandheldSleepPlan_10Min,    //!< 10 分
    HandheldSleepPlan_30Min,    //!< 30 分
    HandheldSleepPlan_Never,    //!< しない
};

/**
 * @brief       据置モード時の自動スリープまでの時間を表す列挙型です。
 */
enum ConsoleSleepPlan
{
    ConsoleSleepPlan_1Hour,    //!< 1 時間
    ConsoleSleepPlan_2Hour,    //!< 2 時間
    ConsoleSleepPlan_3Hour,    //!< 3 時間
    ConsoleSleepPlan_6Hour,    //!< 6 時間
    ConsoleSleepPlan_12Hour,   //!< 12 時間
    ConsoleSleepPlan_Never,    //!< しない
};

/**
 * @brief       スリープ設定を表す構造体です。
 */
struct SleepSettings
{
    SleepFlagSet flags;         //!< 制御フラグ
    int32_t handheldSleepPlan;  //!< 携帯モード時の自動スリープまでの時間
    int32_t consoleSleepPlan;   //!< 据置モード時の自動スリープまでの時間
};

//! @name スリープ設定関連 API
//! @{

/**
 * @brief       スリープ設定を取得します。
 *
 * @details     スリープ設定として指定されている値を取得します。
 *
 * @param[out]  pOutValue                   スリープ設定を読み出すバッファ
 *
 * @pre
 *              - pOutValue != nullptr
 */
void GetSleepSettings(SleepSettings* pOutValue) NN_NOEXCEPT;

/**
 * @brief       スリープ設定を変更します。
 *
 * @details     スリープ設定を指定された値に変更します。
 *
 * @param[in]   value                       スリープ設定
 */
void SetSleepSettings(const SleepSettings& value) NN_NOEXCEPT;

//! @}

}}} // nn::settings::system
