﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       お知らせに関する設定 API の宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/account/account_Config.h>
#include <nn/account/account_Types.h>
#include <nn/util/util_BitFlagSet.h>

namespace nn { namespace settings { namespace system {

/**
 * @brief       通知設定の制御フラグ定義です。
 */
struct NotificationFlag
{
    typedef ::nn::util::BitFlagSet<32, NotificationFlag>::Flag<0>
            Ringtone;           //!< 通知と同時に音を出すか否か
    typedef ::nn::util::BitFlagSet<32, NotificationFlag>::Flag<1>
            DownloadCompletion; //!< ダウンロード完了を通知するか否か
    typedef ::nn::util::BitFlagSet<32, NotificationFlag>::Flag<8>
            EnablesNews;        //!< スリープ中にサプライズニュースを表示するか否か
    typedef ::nn::util::BitFlagSet<32, NotificationFlag>::Flag<9>
            IncomingLamp;       //!< スリープ中にサプライズニュースを受信した際に通知ランプを点灯するか否か
};

/**
 * @brief       通知設定の制御フラグ集合を扱う型です。
 */
typedef ::nn::util::BitFlagSet<32, NotificationFlag> NotificationFlagSet;

/**
 * @brief       通知設定の時刻を表す構造体です。
 */
struct NotificationTime
{
    int32_t hour;   //!< 時
    int32_t minute; //!< 分
};

/**
 * @brief       通知設定の音量を表す構造体です。
 */
enum NotificationVolume
{
    NotificationVolume_Mute,    //!< なし
    NotificationVolume_Low,     //!< 小さい
    NotificationVolume_High,    //!< 大きい
};

/**
 * @brief       通知設定を表す構造体です。
 */
struct NotificationSettings
{
    NotificationFlagSet flags;  //!< 通知設定の制御フラグ
    int32_t volume;             //!< サプライズニュース通知の音量 ( NotificationVolume 型の値を設定 )
    NotificationTime headTime;  //!< サプライズニュース表示を開始する時刻
    NotificationTime tailTime;  //!< サプライズニュース表示を終了する時刻
};

const int AccountNotificationSettingsCountMax =
    ::nn::account::UserCountMax;    //!< アカウント毎の通知設定の最大数

/**
 * @brief       フレンドプレゼンスのオーバーレイ通知の表示範囲を表す列挙体です。
 */
enum FriendPresenceOverlayPermission
{
    FriendPresenceOverlayPermission_NotConfirmed,    //!< 未設定
    FriendPresenceOverlayPermission_NoDisplay,       //!< 表示しない
    FriendPresenceOverlayPermission_FavoriteFriends, //!< お気に入りのフレンドだけ
    FriendPresenceOverlayPermission_Friends,         //!< すべてのフレンド
};

/**
 * @brief       アカウント毎の通知設定の制御フラグ定義です。
 */
struct AccountNotificationFlag
{
    typedef ::nn::util::BitFlagSet<32, AccountNotificationFlag>::Flag<0>
            FriendOnline;       //!< フレンドのオンライン通知を行うか否か
    typedef ::nn::util::BitFlagSet<32, AccountNotificationFlag>::Flag<1>
            FriendRequest;      //!< フレンドリクエストの通知を行うか否か
    typedef ::nn::util::BitFlagSet<32, AccountNotificationFlag>::Flag<8>
            CoralInvitation;    //!< Coral のルーム招待の通知を行うか否か
};

/**
 * @brief       アカウント毎の通知設定の制御フラグ集合を扱う型です。
 */
typedef ::nn::util::BitFlagSet<32, AccountNotificationFlag>
        AccountNotificationFlagSet;

/**
 * @brief       アカウント毎の通知設定を表す構造体です。
 */
struct AccountNotificationSettings
{
    ::nn::account::Uid uid;                 //!< アカウントの識別子
    AccountNotificationFlagSet flags;       //!< アカウント毎の通知設定の制御フラグ
    int8_t friendPresenceOverlayPermission; //!< フレンドプレゼンスのオーバーレイ通知の表示範囲 ( FriendPresenceOverlayPermission 型の値を設定 )
    ::nn::Bit8 _reserved[3];
};

//! @name お知らせ設定関連 API
//! @{

/**
 * @brief       アプリケーション自動配信が有効か否かを表す値を返します。
 *
 * @return      アプリケーション自動配信の有効か否かを表す値です。
 */
bool GetAutomaticApplicationDownloadFlag() NN_NOEXCEPT;

/**
 * @brief       アプリケーション自動配信が有効か否かを表す値を設定します。
 *
 * @param[in]   value                       アプリケーション自動配信が有効か否かを表す値
 */
void SetAutomaticApplicationDownloadFlag(bool value) NN_NOEXCEPT;

/**
 * @brief       通知設定を取得します。
 *
 * @param[out]  pOutValue                   通知設定を読み出すバッファ
 *
 * @pre
 *              - pOutValue != nullptr
 */
void GetNotificationSettings(NotificationSettings* pOutValue) NN_NOEXCEPT;

/**
 * @brief       通知設定を変更します。
 *
 * @param[in]   value                       通知設定
 */
void SetNotificationSettings(const NotificationSettings& value) NN_NOEXCEPT;

/**
 * @brief       アカウント毎の通知設定を取得します。
 *
 * @details     アカウント毎の通知設定として指定されている値を取得可能な数だけ読み出します。
 *              取得可能な通知設定の数より大きなバッファ（配列）が指定された場合、
 *              余った領域に対しては何も行いません。
 *              読み出し可能な通知設定の最大数は AccountNotificationSettingsCountMax 個です。
 *
 * @param[out]  outValues                   通知設定を読み出すバッファ（配列）
 * @param[in]   count                       outValues の数
 *
 * @return      読み出した通知設定の数を返します。
 *
 * @pre
 *              - outValues != nullptr
 */
int GetAccountNotificationSettings(
    AccountNotificationSettings outValues[], int count) NN_NOEXCEPT;

/**
 * @brief       アカウント毎の通知設定を変更します。
 *
 * @details     通知設定を指定された値に変更します。
 *              通知設定の変更によって、以前に設定されていた値は全て消去されます。
 *              指定可能な通知設定の最大数は AccountNotificationSettingsCountMax 個です。
 *
 * @param[in]   values                      通知設定（配列）
 * @param[in]   count                       values の数
 *
 * @pre
 *              - values != nullptr
 *              - 0 <= count && count <= AccountNotificationSettingsCountMax
 */
void SetAccountNotificationSettings(
    const AccountNotificationSettings values[], int count) NN_NOEXCEPT;

//! @}

}}} // nn::settings::system
