﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       デバイス証明書の生産時設定に関する API の宣言
 */

#pragma once

#include <nn/nn_BitTypes.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/settings/factory/settings_Result.h>

namespace nn { namespace settings { namespace factory {

/**
 * @brief       デバイス証明書 (ECC-P256 版) を表す構造体です。
 */
struct EccP256DeviceCertificate
{
    ::nn::Bit8 data[384];   //!< デバイス証明書
};

/**
 * @brief       デバイス証明書 (ECC-B233 版) を表す構造体です。
 */
struct EccB233DeviceCertificate
{
    ::nn::Bit8 data[384];   //!< デバイス証明書
};

/**
 * @brief       デバイス証明書 (RSA-2048 版) を表す構造体です。
 */
struct Rsa2048DeviceCertificate
{
    ::nn::Bit8 data[576];   //!< デバイス証明書
};

/**
 * @brief       デバイス登録用のデバイス証明書 (ECC-B233 版) を取得します。
 *
 * @param[out]  pOutValue                    デバイス証明書を読みだすバッファ
 *
 * @retresult
 * @handleresult{nn::settings::factory::ResultCalibrationDataFileSystemCorrupted}
 * @handleresult{nn::settings::factory::ResultCalibrationDataCrcError}
 * @endretresult
 *
 * @pre
 *              - pOutValue != nullptr
 */
::nn::Result GetEciDeviceCertificate(
    EccB233DeviceCertificate* pOutValue) NN_NOEXCEPT;

/**
 * @brief       デバイス登録用のデバイス証明書 (RSA-2048 版) を取得します。
 *
 * @param[out]  pOutValue                    デバイス証明書を読みだすバッファ
 *
 * @retresult
 * @handleresult{nn::settings::factory::ResultCalibrationDataFileSystemCorrupted}
 * @handleresult{nn::settings::factory::ResultCalibrationDataCrcError}
 * @endretresult
 *
 * @pre
 *              - pOutValue != nullptr
 */
::nn::Result GetEciDeviceCertificate(
    Rsa2048DeviceCertificate* pOutValue) NN_NOEXCEPT;

/**
 * @brief       eTicket 発行用のデバイス証明書を取得します。
 *
 * @param[out]  pOutValue                    デバイス証明書を読みだすバッファ
 *
 * @retresult
 * @handleresult{nn::settings::factory::ResultCalibrationDataFileSystemCorrupted}
 * @handleresult{nn::settings::factory::ResultCalibrationDataCrcError}
 * @endretresult
 *
 * @pre
 *              - pOutValue != nullptr
 */
::nn::Result GetEticketDeviceCertificate(
    Rsa2048DeviceCertificate* pOutValue) NN_NOEXCEPT;

}}} // nn::settings::factory
