﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief PSM ライブラリのシステムプロセス用の API 宣言。
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/psm/psm_SystemProcessTypes.h>

namespace nn { namespace psm {

/**
 * @brief      コントローラへの給電の使用権を取得します。
 * @pre
 *  - PSM ライブラリが初期化済みの状態である
 * @details    コントローラへの給電の使用権を取得します。
 *             本関数を呼び出さなくてもコントローラと同じ電源系統の回路を
 *             動作させるためコントローラにも給電がおこなわれることがありますが、
 *             スリープ時などに他の回路の動作停止とともに給電されなくなります。
 */
void AcquireControllerPowerSupply() NN_NOEXCEPT;

/**
 * @brief      コントローラへの給電の使用権を解放します。
 * @pre
 *  - PSM ライブラリが初期化済みの状態である
 * @details    コントローラへの給電の使用権を解放します。
 *             本関数を呼び出した後でもコントローラと同じ電源系統の回路を
 *             動作させるためにコントローラへの給電が停止しないことがあります。
 */
void ReleaseControllerPowerSupply() NN_NOEXCEPT;

/**
 * @brief      セッションをオープンします。
 * @pre
 *  - PSM ライブラリが初期化済みの状態である
 * @post
 *  - 指定したセッションがオープン状態である
 * @details    セッションをオープンします。
 */
void OpenSession(Session* pOutSession) NN_NOEXCEPT;

/**
 * @brief      セッションをクローズします。
 * @pre
 *  - PSM ライブラリが初期化済みの状態である
 *  - 指定したセッションがオープン状態である
 * @post
 *  - 指定したセッションがクローズ状態である
 * @details    セッションをクローズします。
 */
void CloseSession(Session* pSession) NN_NOEXCEPT;

/**
 * @brief      本体に接続されている充電器の変化とイベントを紐付けます。
 * @param[in]  pEvent   本体に接続されている充電器や電池電圧レベルの状態の変化したときシグナル化するイベント
 * @param[in]  pSession セッション
 * @pre
 *  - PSM ライブラリが初期化済みの状態である
 *  - 指定したセッションがオープン状態である
 * @details    本体に接続されている充電器の変化とイベントを紐付けます。
 */
void BindStateChangeEvent(
    nn::os::SystemEventType* pEvent,
    Session* pSession) NN_NOEXCEPT;

/**
 * @brief      本体に接続されている充電器の変化とイベントの紐付けを解除します。
 * @param[in]  pSession セッション
 * @pre
 *  - PSM ライブラリが初期化済みの状態である
 *  - 指定したセッションがオープン状態である
 * @details    本体に接続されている充電器の変化とイベントの紐付けを解除します。
 */
void UnbindStateChangeEvent(Session* pSession) NN_NOEXCEPT;

/**
 * @brief      充電器の変化によるイベントのシグナルを有効化するかどうかを設定します。
 * @param[in]  pSession  セッション
 * @param[in]  isEnabled 充電器の変化によるイベントのシグナルを有効化するかどうか
 * @pre
 *  - PSM ライブラリが初期化済みの状態である
 *  - 指定したセッションがオープン状態である
 * @details    充電器の変化によるイベントのシグナルを有効化するかどうかを設定します。
 */
void SetChargerTypeChangeEventEnabled(Session* pSession, bool isEnabled) NN_NOEXCEPT;

/**
 * @brief      給電能力の変化によるイベントのシグナルを有効化するかどうかを設定します。
 * @param[in]  pSession  セッション
 * @param[in]  isEnabled 給電能力の変化によるイベントのシグナルを有効化するかどうか
 * @pre
 *  - PSM ライブラリが初期化済みの状態である
 *  - 指定したセッションがオープン状態である
 * @details    給電能力の変化によるイベントのシグナルを有効化するかどうかを設定します。
 */
void SetPowerSupplyChangeEventEnabled(Session* pSession, bool isEnabled) NN_NOEXCEPT;

/**
 * @brief      電池電圧の状態の変化によるイベントのシグナルを有効化するかどうかを設定します。
 * @param[in]  pSession  セッション
 * @param[in]  isEnabled 電池電圧の状態の変化によるイベントのシグナルを有効化するかどうか
 * @pre
 *  - PSM ライブラリが初期化済みの状態である
 *  - 指定したセッションがオープン状態である
 * @details    電池電圧の状態の変化によるイベントのシグナルを有効化するかどうかを設定します。
 */
void SetBatteryVoltageStateChangeEventEnabled(Session* pSession, bool isEnabled) NN_NOEXCEPT;

/**
 * @brief      給電エミュレーションを有効にします。
 * @pre
 *  - PSM ライブラリが初期化済みの状態である
 * @details    給電エミュレーションを有効にします。
 *             給電エミュレーションが有効な期間は給電能力が十分な充電器が接続されているように振舞います。
 */
void EnableEnoughPowerChargeEmulation() NN_NOEXCEPT;

/**
 * @brief      給電エミュレーションを無効にします。
 * @pre
 *  - PSM ライブラリが初期化済みの状態である
 * @details    給電エミュレーションを無効にします。
 */
void DisableEnoughPowerChargeEmulation() NN_NOEXCEPT;

/**
 * @brief      高速充電を有効にします。
 * @pre
 *  - PSM ライブラリが初期化済みの状態である
 * @details    高速充電を有効にします。
 */
void EnableFastBatteryCharging() NN_NOEXCEPT;

/**
 * @brief      高速充電を無効にします。
 * @pre
 *  - PSM ライブラリが初期化済みの状態である
 * @details    高速充電を無効にします。
 */
void DisableFastBatteryCharging() NN_NOEXCEPT;

/**
 * @brief      電池電圧の状態を取得します。
 * @return     @ref nn::psm::BatteryVoltageState 型で表される電池電圧の状態。
 * @pre
 *  - PSM ライブラリが初期化済の状態である
 * @details    電池電圧の状態を @ref nn::psm::BatteryVoltageState 型で取得します。
 */
BatteryVoltageState GetBatteryVoltageState() NN_NOEXCEPT;

/**
 * @brief      未加工の電池残量を取得します。
 * @return     電池残量 [%]
 * @pre
 *  - PSM ライブラリが初期化済の状態である
 * @details    未加工の電池残量を 0 % ～ 100 % までの値で取得します。
 */
double GetRawBatteryChargePercentage() NN_NOEXCEPT;

/**
 * @brief      十分な給電がされているかどうかを取得します。
 * @return     十分な給電がされているかどうか
 * @pre
 *  - PSM ライブラリが初期化済の状態である
 * @details    供給電力が十分な充電器が接続されており、VDD50 が適切に切り替わった状態であるかどうかを取得します。
 */
bool IsEnoughPowerSupplied() NN_NOEXCEPT;

}} // namespace nn::psm
