﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @file
    @brief PSC types
*/

#pragma once

#include <nn/util/util_BitFlagSet.h>

namespace nn  {
namespace psc {


/**
    @brief      PM States.
    @details    PM control service may dispatch a state change notification to PmModule,
                requesting it to prepare for system transition into one of the following
                states. Modules are required to support at least
                PmState_FullAwake, PmState_MinimumAwake and PmState_SleepReady.
                For other states PmModule may choose not to perform any
                handling, but it should still reply using Acknowledge()
                with nn::ResultSuccess and requested state.
*/
enum PmState
{
    /**
        @brief All services and drivers are running. Application / applets can run.
    */
    PmState_FullAwake,
    /**
        @brief   System runs at lower power in order to verify if it can transition into FullAwake state.
        @details Some modules may choose to be in SleepReady state to satisfy this request.
    */
    PmState_MinimumAwake,
    /**
        @brief   All default services and drivers are ready for system to enter sleep state.
    */
    PmState_SleepReady,
    /**
        @brief   Low level services providing functionality to default services are ready to enter sleep state.
    */
    PmState_EssentialServicesSleepReady,
    /**
        @brief   Low level services providing functionality to default services and drivers are running.
    */
    PmState_EssentialServicesAwake,
    /**
        @brief   All default services and drivers are ready for system to be powered off or rebooted.
    */
    PmState_ShutdownReady,

    PmState_Unknown,
};

/**
    @brief      PM flag bit definitions.
*/
struct PmFlag
{
    typedef ::nn::util::BitFlagSet<32, PmFlag>::Flag<0> FlagBit0;

    // FIXME: This flag is made to "workaround" the vcc issue (NSBG-6205, SIGLO-41037). Any other module shall not refer this flag.
    /**
        @brief  Hardware sleep will be skipped after entering PmState_EssentialServicesSleepReady.
    */
    typedef ::nn::util::BitFlagSet<32, PmFlag>::Flag<1> SkippingHardwareSleepAfterEssentialServicesSleepReady;
};

/**
    @brief      PM flags
    @details    PM control service may provide additional information to
                PmModule through PmFlagSet field.
*/
typedef ::nn::util::BitFlagSet<32, PmFlag> PmFlagSet;

/**
    @brief      Maximum number of dependency levels between modules.
*/
const uint32_t MaximumDependencyLevels = 20;

/**
    @brief      Maximum number of PM modules per process.
*/
const uint32_t MaximumModulesPerProcess = 50;

}}

