﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @file
    @brief  Public header defining PSC API exposed to power managed modules.
*/

#pragma once

#include <nn/os/os_SystemEvent.h>
#include <nn/psc.h>
#include <nn/psc/sfdl/psc.sfdl.h>

#include <nn/psc/server/psc_Server.h>
#include <nn/psc/psc_PmModule.h>
#include <nn/psc/psc_Result.h>

#include <nn/result/result_HandlingUtility.h>

namespace nn     {
namespace psc    {

/**
    @brief      Power managed module.

    @details    This class declares a power managed module or service.
                Once declared and initialized, this class can be used to receive notifications
                about PM state changes in the system. When such notification is received, it is
                module's responsibility to i) obtain information about change through @ref GetRequest(),
                ii) transition into requested state, and iii) notify PM control service once transition is
                completed using @ref Acknowledge().

                Notifications will be dispatched to all registered modules
                in accordance with the dependency information provided to PSC
                in @ref Initialize call. It is guaranteed that all dependencies are satisfied
                before notification is sent out to each module.

                Sample code:
                @code
                #include <nn/psc.h>
                void EthPmHandler()
                {
                    nn::Result           result;
                    nn::psc::PmModule    ethModule;
                    nn::psc::PmModuleId* ethDependencies[] = {PmModuleId_Usb, PmModuleId_Socket};
                    nn::psc::PmFlagSet   flags;
                    nn::psc::PmState     state;

                    result = ethModule.Initialize(
                                PmModuleId_Eth,
                                ethDependecies,
                                sizeof(ethDependencies) / sizeof(ethDependencies[0]),
                                nn::os::EventClearMode_ManualClear);

                    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

                    while (1)
                    {
                        ethModule.GetEventPointer().Wait();
                        ethModule.GetEventPointer().Clear();

                        result = ethModule.GetRequest(&state, &flags);
                        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

                        // PM modules are expected to handle at least
                        // PmState_FullAwake
                        // PmState_MinimumAwake
                        // PmState_SleepReady

                        switch (state)
                        {
                        case nn::psc::PmState_FullAwake:
                            result = HandleFullAwake(flags);
                            break;
                        case nn::psc::PmState_MinimumAwake:
                            // If not supported, handle as SleepReady.
                            // Note that it is possible that PmState_SleepReady may
                            // be signaled later. If needed, add logic to avoid
                            // calling HandleSleep twice.
                        case nn::psc::PmState_SleepReady:
                            result = HandleSleep(flags);
                            break;
                        default:
                            // no-op, but still return ResultSuccess()
                            // for unhandled request types - new types may be
                            // added in the future.
                            break;
                        }

                        result = ethModule.Acknowledge(state, result);
                        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                    }

                    ethModule.Finalize();
                }
                @endcode
*/
class PmModule
{
    NN_DISALLOW_COPY(PmModule);
    NN_DISALLOW_MOVE(PmModule);

private:
    nn::os::SystemEvent m_SystemEvent;
    bool                m_Initialized;
    PmModuleId          m_Id;

    // Once this header is public, we will not be able to change the size
    // of this class. Reserve pointer storage for any future expansion.
    uintptr_t           m_Reserved;

    PmState             m_State;
    PmFlagSet           m_Flags;

public:

    PmModuleId Id()
    NN_NOEXCEPT
    {
        return m_Id;
    }

    /**
        @brief      Obtain requested state and flags from PM control service.

        @param[out] pStateOut               Pointer to @ref PmState variable.
        @param[out] pFlagsOut               Pointer to @ref PmFlagSet variable.
        @return     nn::ResultSuccess       Operation successfull.

        @pre        @ref Initialize() must be called.
        @details    Requested state and flags are returned in pStateOut and pFlagsOut.
    */
    nn::Result GetRequest(PmState* pStateOut, PmFlagSet* pFlagsOut)
    NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_Initialized, nn::psc::ResultNotInitialized());

        *pStateOut = m_State;
        *pFlagsOut = m_Flags;

        NN_RESULT_SUCCESS;
    }

    /**
        @brief      Acknowledge completion of requested transition.

        @param[in]  state                   Copy of the @ref PmState obtained with @ref GetRequest().
        @param[in]  result                  Result code from operation.
        @return     nn::ResultSuccess       Acknowledgment is confirmed by PSC.

        @pre        @ref Initialize() must be called.
        @details    Once module completes transition into requested state, it has to notify PSC
                    of completion by calling @ref Acknowledge() with the copy of requested state (obtained
                    with @ref GetRequest()) and the result code. Currently, modules are not allowed to fail
                    and should return non-failure result code.
    */
    nn::Result Acknowledge(PmState state, nn::Result result)
    NN_NOEXCEPT
    {
        // We may decide to use result and state in the future,
        // e.g. to notify of a failure, pass module's state after failure.
        // Modyfing public API at that time will not be possible.
        NN_UNUSED(result);
        NN_UNUSED(state);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        NN_RESULT_THROW_UNLESS(m_Initialized, nn::psc::ResultNotInitialized());

        NN_RESULT_SUCCESS;
    }

    /**
        @brief      Obtain handle to system event associated with this module.
        @return     Pointer to system event.
        @pre        @ref Initialize() must be called.
        @details    Clients wishing to implement event handling logic may obtain
                    pointer to the system event associated with this module.
                    Please note that it is recommended to use nn::os::MultiWaitType
                    object to handle multiple events using a single thread.
    */
    nn::os::SystemEvent* GetEventPointer()
    NN_NOEXCEPT
    {
        return &m_SystemEvent;
    }

    /**
        @brief      Dispatch the arbitrary event
        @pre        @ref Initialize() must be called.
        @details    To test event handling on each module, this function dispatches the arbitrary event.
    */
    void Dispatch(PmState state, PmFlagSet flags)
    NN_NOEXCEPT
    {
        m_State = state;
        m_Flags = flags;
        m_SystemEvent.Signal();
    }

    /**
        @brief      Initialize power managed module
        @param[in]  moduleId                PmModuleId for this module.
        @param[in]  pDependencyList         List of PmModuleId's for all dependencies.
        @param[in]  dependencyCount         Number of id's in the dependency list.
        @param[in]  eventClearMode          Clear mode for notification event.

        @return     nn::ResultSuccess                       Operation successfull.
        @return     nn::psc::ResultAlreadyInitialized       Module has already been initialized.
        @return     nn::psc::ResultOutOfMemory              Out of memory condition was detected.
        @return     nn::psc::ResultSelfDependency           Id of the module is listed in the dependency list.
        @return     nn::psc::ResultCircularDependency       Initializing this module will result in a circular dependency.
        @return     nn::psc::ResultMaxDependencyLevels      Initializing this module will exceed nn::psc::MaximumDependencyLevels.
        @return     nn::sf::ResultMemoryAllocationFailed    Current process has already registered nn::psc::MaximumModulesPerProcess.

        @pre        Client should be provisioned to access "psc:m" port.
        @details    Initializes power managed module
    */
    nn::Result Initialize(const PmModuleId  moduleId, const PmModuleId* pDependencyList, uint32_t dependencyCount, nn::os::EventClearMode eventClearMode)
    NN_NOEXCEPT
    {
        NN_UNUSED(pDependencyList);
        NN_UNUSED(dependencyCount);

        NN_RESULT_THROW_UNLESS(!m_Initialized, nn::psc::ResultAlreadyInitialized());

        nn::os::CreateSystemEvent(m_SystemEvent.GetBase(), eventClearMode, false);

        m_Id = moduleId;
        m_Initialized = true;

        NN_RESULT_SUCCESS;
    }

    /**
        @brief      Finalizes power managed module.
        @result     nn::ResultSuccess
        @result     nn::psc::ResultNotInitialized       Module is not initialized.
        @result     nn::psc::ResultBreaksDependency     There are other modules dependent on this module.

        @pre        @ref Initialize() must be called.
        @details    This call will only succeed if there are no other modules in the system
                    with dependency on this module.

                    If successfull, dependency information for this module will be removed from PSC.
    */
    nn::Result Finalize()
    NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_Initialized, nn::psc::ResultNotInitialized());

        nn::os::DestroySystemEvent(m_SystemEvent.GetBase());
        m_Initialized   = false;

        NN_RESULT_SUCCESS;
    }

    PmModule()
    NN_NOEXCEPT
    : m_Initialized(false),
      m_Id(PmModuleId_Reserved0),
      m_Reserved(0),
      m_State(PmState_FullAwake)
    {
        m_Flags.Reset();
    }

    ~PmModule()
    NN_NOEXCEPT
    {
    }
};

}}
