﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief Nintendo NX CPU Profiler Control API
 * @details Nintendo NX CPU Profiler API
 */

#pragma once


#include <nn/nn_Common.h>
#include <nn/nn_Result.h>


namespace nn { namespace profiler {


/**
 *  @brief Defines the set of cores that data can be collected on.
 *
 *  @details
 *  These enumerations are used as a bit-wise OR and passed into SetProfileSettings().
 *  These flags correspond to the _Core_ buttons on the _Sampled Profile_ tab in the GUI.
 *  Core settings are ignored for Instrumented Profiles.
 *
 *  @sa SetProfileSettings, Flags
 */
enum Cores
{
    Cores_0     = (1 << 0), //!< Core 0.
    Cores_1     = (1 << 1), //!< Core 1.
    Cores_2     = (1 << 2), //!< Core 2.
    Cores_All   = (Cores_0 | Cores_1 | Cores_2), //!< All three cores.
};


/**
 *  @brief Flags used to control how the profiler works.
 *
 *  @details
 *  These flags are used as a bit-wise OR and are passed into SetProfilerSettings().
 *
 *  @sa SetProfileSettings, Cores
 */
enum Flags
{
    /**
     *  @brief Only Program counters are recorded.
     *
     *  @details
     *  This flag corresponds to have the _Leaf Only_ button toggled in the GUI.
     *
     *  When this flag is set, only Program Counters will be recorded in the profile data.
     *  This flag is mutually exclusive with @ref Flags_Callstack.
     *  If both are set, @ref Flags_Callstack takes priority.
     *  Other flags can be used in conjunction with this flag.
     *
     *  This flag is useful because it will show where the sample occurred.
     *  There is less overhead with this when compared to @ref Flags_Callstack.
     *  Because less data is recorded, longer profiles can also be taken.
     */
    Flags_Simple                        = 0,

    /**
     *  @brief Callstacks will be recorded.
     *
     *  @details
     *  This flag corresponds to have the _Callstack_ button toggled in the GUI.
     *
     *  When this flag is set, full callstack data will be recorded in the profile data.
     *  This flag is mutually exclusive with @ref Flags_Simple.
     *  If both are set, @ref Flags_Callstack takes priority.
     *  Other flags can be used in conjunction with this flag.
     *
     *  This flag is useful because it will show the complete callstack where the sample occurred.
     *  There is far more overhead with this when compared to @ref Flags_Simple.
     *  Because more data is recorded, profile time is reduced.
     */
    Flags_Callstack                     = (1 << 0),

    /**
     *  @brief When a sample is taken, also record performance counter data.
     *
     *  @details
     *  This flag corresponds to selecting a performance counter group in the GUI.
     *
     *  When this flag is set, performance counter data is recorded in the profile data.
     *
     *  When using this option, a @ref PerformanceCounterGroup "performance counter group"
     *  should be supplied in SetProfileSettings().
     */
    Flags_PerformanceCounters           = (1 << 2),

    /**
     *  @brief Take a sample when a performance counter value overflows.
     *
     *  @details
     *  This flag corresponds to selecting one of the Sample by Performance Counter
     *  sampling strategies.
     *  When this flag is set, a sample will occur every time a specified
     *  @ref PerformanceCounterGroup "performance counter group" overflows.
     *  Only the PerformanceCounterGroup values starting with 'SampleByPerf' can be specified.
     *
     *  When using this option, corresponding 'Sample by Perf' settings need
     *  to be supplied to SetProfileSettings().
     *
     *  @note
     *  This option is not currently supported.
     */
    Flags_SampleByPerformanceCounter    = (1 << 3),

    /**
     *  @brief Take an Instrumented profile, rather than a Sampled Profile.
     *
     *  @details
     *  This flag corresponds to the _Instrumented Profile_ tab in the GUI.
     *
     *  The default behavior in the profiler is to act as a sampling profiler.
     *  Using this flag changes this behavior to instead take a sample each time
     *  a function entered and exited.
     *
     *  @ref Cores "Core selection" and @ref SampleRate "sample rate"
     *  are ignored when this flag is set.
     *
     *  @note
     *  This option is not currently supported.
     */
    Flags_Instrumented                  = (1 << 4),


    /**
    *  @brief Take a profile using out-of-process profiling, rather than in-process profiling.
    *
    *  @details
    *  This flag corresponds to the _In_ and _Out_ buttons in the GUI on the _Sampled Profile_ tab.
    *
    *  The default behavior of the profiler is to use the in-process profiler.
    *  Setting this flag allows for the use the out-of-process profiler instead.
    *  Please see the GUI and/or manual for information on the two available profiling modes.
    */
    Flags_OutOfProcessProfiling         = (1 << 8),
};


/**
 *  @brief The performance counters that can be used in profiling.
 *
 *  @details
 *  A more detailed explanation of each group can be found in the manual.
 *  Look in the chapter entitled 'Performance Counter Groups'.
 */
enum PerformanceCounterGroup
{
    /** @brief No performance counter data is recorded. */
    PerformanceCounterGroup_Disabled = 0,

    /** @brief Shows a variety of performance counters on the instructions executed. */
    PerformanceCounterGroup_InstructionsExecuted = 1,

    /** @brief Shows how instructions interact with the L1 instruction cache. */
    PerformanceCounterGroup_L1ICache = 2,

    /** @brief Shows how data interact with the L1 data cache. */
    PerformanceCounterGroup_L1DCache = 3,

    /** @brief Shows how instructions and data interact with the unified L2 cache. */
    PerformanceCounterGroup_L2Cache = 4,

    /** @brief Tracks instruction usage related to the use of atomic values. */
    PerformanceCounterGroup_Atomics = 5,

    /** @brief Tracks branch instruction usage. */
    PerformanceCounterGroup_Branches = 6,

    /** @brief Tracks system calls made by the application. */
    PerformanceCounterGroup_SystemCalls = 7,

    /** @brief Tracks reads and writes where data is not properly aligned. */
    PerformanceCounterGroup_UnalignedDataAccess = 8,
};



/**
 *  @brief The different sampling rates supported by the profiler.
 *
 *  @details
 *  The list below contains the sampling rates currently supported
 *  by the profiler when taking Sampled Profile by Time.
 *  Rates are set as the number of times per 60Hz frame a sample should occur.
 */
enum SampleRate
{
    SampleRate_ByTimeMin = 0,
    SampleRate_ByTime1x = SampleRate_ByTimeMin, /*!< Sample 1 time every 60Hz. */
    SampleRate_ByTime10x,       /*!< Sample 10 times every 60Hz. */
    SampleRate_ByTime25x,       /*!< Sample 25 times every 60Hz. */
    SampleRate_ByTime50x,       /*!< Sample 50 times every 60Hz. */
    SampleRate_ByTime75x,       /*!< Sample 75 times every 60Hz. */
    SampleRate_ByTime100x,      /*!< Sample 100 times every 60Hz. */
    SampleRate_ByTime250x,      /*!< Sample 250 times every 60Hz. */
    SampleRate_ByTime500x,      /*!< Sample 500 times every 60Hz. */
    SampleRate_ByTime750x,      /*!< Sample 750 times every 60Hz. */
    SampleRate_ByTime1000x,     /*!< Sample 1000 times every 60Hz. */
    SampleRate_ByTime2000x,     /*!< Sample 2000 times every 60Hz. */
    SampleRate_ByTime4000x,     /*!< Sample 4000 times every 60Hz. */
    SampleRate_ByTimeMax,

    SampleRate_Max = SampleRate_ByTimeMax,
};


//! @name Runtime Control API
//! @{


/**
*  @brief Sets profile settings for runtime based control.
*
*  @warning This function is not currently implemented.
*  It will always return nn::os::ResultNotImplemented.
*
*  @param [in] affinityMask Which cores should be used to profile.
*      Bitwise OR the values from the @ref Cores enum.
*  @param [in] flags Specifies which flags should be set for profiling.
*      Bitwise OR the values form the @ref Flags enum.
*  @param [in] performanceCounterGroup
*      Specifies which performance counter group to use while profiling.
*      Setting a value here is only respected if the flag @ref Flags_PerformanceCounters is set.
*      Please see the manual for a more detailed explanation of the performance counter groups.
*  @param [in] sampleRate The requested sample rate.
*
*  @retresult
*      @handleresult{nn::ResultSuccess, Settings were set successfully.}
*      @handleresult{nn::profiler::ResultNotInitialized,
*          Setting cannot be set as the profiler status.}
*      @handleresult{nn::profiler::ResultInvalidArgument,
*          One of the arguments passed in contained an invalid value.}
*  @endretresult
*
*  @details
*  This function sets the settings that will be used by StartProfiling()
*  in the case of a manual startup.
*
*  The parameters @p affinityMask and @p sampleRate are ignored when taking an
*  Instrumented profile.
*
*  If @ref Flags_SampleByPerformanceCounter is set, @p perfCounters and @p sampleRate
*  need to be set to their "by performance counter" values.
*  For @p perfCounters you will need to use the counters of the form
*  @ref PerformanceCounterGroup "PROFILERPerfCounterGroup_SampleByPerf_*".
*  For @p sampleRate you will need to use the rates of the form
*  @ref SampleRate "PROFILERSampleRate_ByPerf_*".
*
*  When the profiler is @ref nn::profiler::Initialize "first initialized" the settings are set to:
*  @code
*  nn::profiler::SetProfileSettings(
*      nn::profiler::Cores_All,
*      nn::profiler::Flags_Callstack,
*      nn::profiler::PerformanceCounterGroup_Disabled,
*      nn::profiler::SampleRate_ByTime100x);
*  @endcode
*/
nn::Result SetProfileSettings(
    uint32_t affinityMask,
    uint32_t flags,
    PerformanceCounterGroup performanceCounterGroup,
    SampleRate sampleRate) NN_NOEXCEPT;


/**
*  @brief Starts profiling.
*
*  @warning This function is not currently implemented.
*  It will always return nn::os::ResultNotImplemented.
*
*  @retresult
*      @handleresult{nn::ResultSuccess, Profiling has started.}
*      @handleresult{nn::profiler::ResultNotInitialized, Profiler status is currently
*          @ref ProfilerStatus_Offline.}
*      @handleresult{nn::profiler::ResultAlreadyDone, Profiler status is currently
*          @ref ProfilerStatus_Profiling or @ref ProfilerStatus_Transferring.}
*  @endretresult
*
*  @details
*  This function starts profiling from the application.
*  The settings used for profiling will be whatever were the last ones set.
*  Settings can be set from either the GUI or by calling SetProfileSettings().
*  Profiling will continue until the internal buffer is full, StopProfiling() is called,
*  or the stop button is pressed in the GUI.
*
*  If StartProfiling() is called before any settings have been explicitly set,
*  the profiler will use a default profiling setup.
*  This default is the same as the default settings set when starting the
*  profiler GUI for the first time.
*  The default settings can be set manually with the following call:
*  @code
*  nn::profiler::SetProfileSettings(
*      nn::profiler::Cores_All,
*      nn::profiler::Flags_Callstack,
*      nn::profiler::PerformanceCounterGroup_Disabled,
*      nn::profiler::SampleRate_ByTime100x);
*  @endcode
*
*  @note
*  For profile data to be transferred to the PC, the GUI will need to be Synced.
*
*  @warning Do not call from a Callback.
*/
nn::Result StartProfiling() NN_NOEXCEPT;


/**
*  @brief Stop profiling.
*
*  @warning This function is not currently implemented.
*  It will always return nn::os::ResultNotImplemented.
*
*  @retresult
*      @handleresult{nn::ResultSuccess, Profiling has been stopped.}
*      @handleresult{nn::profiler::ResultNotInitialized, Profiler status is currently
*          @ref ProfilerStatus_Offline.}
*      @handleresult{nn::profiler::ResultAlreadyDone, Profiler status is currently
*          @ref ProfilerStatus_Active or @ref ProfilerStatus_Transferring.}
*  @endretresult
*
*  @details
*  This function causes profiling to stop and the captured profile data to be
*  sent back to the PC.
*  Profiling can also be stopped if the internal buffers completely fill or if
*  the _Stop_ button in the GUI is pressed.
*
*  @warning Do not call from a Callback.
*/
nn::Result StopProfiling() NN_NOEXCEPT;


//! @}


}} // namespace nn::profiler
