﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_ApplicationId.h>
#include <nn/prepo/prepo_Types.h>
#include <nn/account/account_Types.h>

namespace nn { namespace prepo {

/*!
    @brief      プレイレポートを扱うクラスです。

    @details
                本クラスは、アプリケーション ID を自ら宣言する以外は、 @ref PlayReport と同じです。

                システムアプリ／システムプロセスは通常の @ref PlayReport クラスは利用できないため、本クラスを利用してください。
*/
class SystemPlayReport
{
private:
    NN_DISALLOW_COPY(SystemPlayReport);
    NN_DISALLOW_MOVE(SystemPlayReport);

public:
    /*!
        @brief      レポートデータの記録バッファの最小サイズです。

        @details
                    @ref SystemPlayReport::SetBuffer には、最小サイズ以上のバッファを与える必要があります。
    */
    static const size_t BufferSizeMin = ReportBufferSizeMin;

    /*!
        @brief      レポートデータの記録バッファの最大サイズです。

        @details
                    仕様上、最大サイズ以上のバッファは利用しません。
    */
    static const size_t BufferSizeMax = ReportBufferSizeMax;

public:
    /*!
        @brief      コンストラクタです。

        @details
                    イベント ID には空文字列が指定されます。@n
                    @ref SystemPlayReport::Save を呼び出す前に、 @ref SystemPlayReport::SetEventId を呼び出す必要があります。

        @see
            - SystemPlayReport::SetEventId
    */
    SystemPlayReport() NN_NOEXCEPT;

    /*!
        @brief      コンストラクタです。

        @param[in]  eventId イベント ID。

        @pre
            - eventId != nullptr
            - eventId が有効なイベント ID である。

        @details
                    不正なイベント ID を指定した場合、ASSERT します。@n
                    イベント ID のフォーマットに関しては、 @ref prepo 名前空間の詳解を参照してください。
    */
    explicit SystemPlayReport(const char* eventId) NN_NOEXCEPT;

    /*!
        @brief      バッファを設定します。

        @param[in]  buffer  バッファ。
        @param[in]  size    バッファサイズ。

        @pre
            - buffer != nullptr
            - size >= @ref SystemPlayReport::BufferSizeMin

        @details
                    本関数で設定したバッファにレポートデータを記録します。@n
                    本オブジェクトを破棄するまで、バッファを開放してはいけません。

                    本関数で設定するバッファサイズの最適値は、以下の計算で求めることができます。

                    - @ref SystemPlayReport::BufferSizeMin + @ref KeyValueSizeMax × キーバリューの数
    */
    void SetBuffer(void* buffer, size_t size) NN_NOEXCEPT;

    /*!
        @brief      イベント ID を設定します。

        @param[in]  eventId イベント ID。

        @return     処理結果。

        @pre
            - eventId != nullptr

        @details
                    イベント ID がすでに設定されていた場合、上書きします。

        @see
            - @ref prepo 名前空間の詳解。
    */
    nn::Result SetEventId(const char* eventId) NN_NOEXCEPT;

    /*!
        @brief      アプリケーション ID を設定します。

        @param[in]  applicationId イベント ID。

        @return     処理結果。

        @pre
            - applicationId != ApplicationId::GetInvalidId()

        @details
                    アプリケーション ID がすでに設定されていた場合、上書きします。

        @see
            - @ref prepo 名前空間の詳解。
    */
    nn::Result SetApplicationId(const nn::ApplicationId& applicationId) NN_NOEXCEPT;

    /*!
        @brief      レポートデータをクリアします。

        @details
                    追加されたキーバリューをすべて消去します。@n
                    イベント ID はクリアされません。
    */
    void Clear() NN_NOEXCEPT;

    /*!
        @brief      キーバリュー（符号付き整数値）を追加します。

        @param[in]  key     キー。
        @param[in]  value   バリュー。

        @return     処理結果。

        @pre
            - key != nullptr

        @see
            - @ref prepo 名前空間の詳解。
    */
    nn::Result Add(const char* key, int64_t value) NN_NOEXCEPT;

    /*!
        @brief      キーバリュー（任意の 64 ビット長の ID）を追加します。

        @param[in]  key     キー。
        @param[in]  value   バリュー。

        @return     処理結果。

        @pre
            - key != nullptr

        @see
            - @ref prepo 名前空間の詳解。
    */
    nn::Result Add(const char* key, const Any64BitId& value) NN_NOEXCEPT;

    /*!
        @brief      キーバリュー（倍精度浮動小数点数）を追加します。

        @param[in]  key     キー。
        @param[in]  value   バリュー。

        @return     処理結果。

        @pre
            - key != nullptr

        @see
            - @ref prepo 名前空間の詳解。
    */
    nn::Result Add(const char* key, double value) NN_NOEXCEPT;

    /*!
        @brief      キーバリュー（文字列値）を追加します。

        @param[in]  key     キー。
        @param[in]  value   バリュー。

        @return     処理結果。

        @pre
            - key != nullptr
            - value != nullptr

        @details
                    文字列値は、妥当な UTF-8 形式にする必要があります。@n
                    文字列値の最大長は、NULL 終端を除いて @ref StringValueLengthMax バイトです。

        @see
            - @ref prepo 名前空間の詳解。
    */
    nn::Result Add(const char* key, const char* value) NN_NOEXCEPT;

    /*!
        @brief      レポートデータを保存します。（ユーザーアカウント未指定）

        @return     処理結果。

        @pre
            - 有効なイベント ID が設定されている。

        @details
                    本関数を使用した場合、レポートデータにユーザアカウントが関連付けられないため、ユーザーの属性（年齢や性別等）を使用したクロス集計を行うことはできません。

                    ユーザアカウントは関連付けられませんが、データの送信に同意しているユーザーアカウントが 1 人もいない場合、レポートデータは保存されません。@n
                    本関数の呼び出し後、バッファに記録されているレポートデータはクリアされます。
    */
    nn::Result Save() NN_NOEXCEPT;

    /*!
        @brief      レポートデータを保存します。

        @param[in]  uid ユーザーアカウント。

        @return     処理結果。

        @pre
            - 有効なイベント ID が設定されている。
            - 指定したユーザーアカウントが Open 状態である。

        @details
                    指定したユーザーアカウントがデータの送信に同意していない場合、レポートデータは保存されません。@n
                    本関数の呼び出し後、バッファに記録されているレポートデータはクリアされます。
    */
    nn::Result Save(const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      追加されたキーバリューの数を取得します。

        @return     追加されたキーバリューの数。
    */
    int GetCount() const NN_NOEXCEPT;

private:
    //
    char m_EventId[EventIdLengthMax + 1];
    nn::ApplicationId m_ApplicationId;
    //
    Bit8* m_Buffer;
    //
    size_t m_Size;
    size_t m_Position;
};

}}
