﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/prepo/detail/service/prepo_Common.h>
#include <nn/prepo/detail/service/core/prepo_UserManager.h>

namespace nn { namespace prepo { namespace detail { namespace service { namespace core {

/*!
    @brief      ユーザー情報の送信に、ユーザーが同意しているか調べるクラスです。
*/
class UserAgreementChecker
{
private:
    NN_DISALLOW_COPY(UserAgreementChecker);
    NN_DISALLOW_MOVE(UserAgreementChecker);

private:
    /*!
        @brief      コンストラクタです。
    */
    UserAgreementChecker() NN_NOEXCEPT
    {
    }

public:
    /*!
        @brief      インスタンスを取得します。

        @return     インスタンス。
    */
    static UserAgreementChecker& GetInstance() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(UserAgreementChecker, s_Instance);
        return s_Instance;
    }

public:
    /*!
        @brief      本クラスが有効化されているかどうか調べます。

        @param[out] outIsEnabled    有効化されているかどうか。

        @return     処理結果。

        @details
                    本クラスが無効化されている場合は、ユーザー同意のチェックが省略されて、
                    @ref Check が必ず成功するようになります。
    */
    nn::Result IsEnabled(bool* outIsEnabled) NN_NOEXCEPT;

    /*!
        @brief      本クラスを有効化または無効化します。

        @param[in]  isEnabled       有効化されているかどうか。

        @return     処理結果。

        @details
                    本クラスが無効化されている場合は、ユーザー同意のチェックが省略されて、
                    @ref Check が必ず成功するようになります。
    */
    nn::Result SetEnabled(bool isEnabled) NN_NOEXCEPT;

    /*!
        @brief      指定したユーザーが「ユーザー情報を送信」に同意しているか調べて、
                    同意している場合は成功を返します。

        @param[in]  nsaId   ネットワークサービスアカウント。

        @return     処理結果。
    */
    nn::Result IsUserAgreedWithAnalystics(const nn::account::NetworkServiceAccountId& nsaId) NN_NOEXCEPT;

    /*!
        @brief      いずれかのユーザーが「ユーザー情報を送信」に同意しているか調べて、
                    同意している場合は成功を返します。

        @return     処理結果。
    */
    nn::Result IsAnyoneAgreedWithAnalystics() NN_NOEXCEPT;

    /*!
        @brief      指定したユーザーが NA EULA に同意しているか調べて、
                    同意している場合は成功を返します。

        @param[in]  nsaId   ネットワークサービスアカウント。

        @return     処理結果。
    */
    nn::Result IsUserAgreedWithNaEula(const nn::account::NetworkServiceAccountId& nsaId) NN_NOEXCEPT;

    /*!
        @brief      いずれかのユーザーが NA EULA に同意しているか調べて、
                    同意している場合は成功を返します。

        @return     処理結果。
    */
    nn::Result IsAnyoneAgreedWithNaEula() NN_NOEXCEPT;

    /*!
        @brief      内部的に保持している、情報送信に同意しているユーザーのリストを更新します。

        @return     処理結果。
    */
    nn::Result Update() NN_NOEXCEPT;

    /*!
        @brief      ニンテンドーアカウントの情報のキャッシュの更新処理の結果を通知します。

        @param[in]  uid     ユーザーアカウント。
        @param[in]  result  更新処理の結果。
    */
    void NotifyUpdateNintendoAccountCacheResult(const nn::account::Uid& uid, nn::Result result) NN_NOEXCEPT;

private:
    //
    nn::Result IsEnabledImpl() NN_NOEXCEPT;
    //
    nn::Result SetEnabledImpl(bool isEnabled) NN_NOEXCEPT;

private:
    //
    nn::os::SdkMutex m_EnableFlagMutex;
    nn::os::SdkMutex m_AgreementInfoMutex;
    //
    bool m_IsEnabled;
    //
    UserManager m_NintendoAccountCacheExpiredUsers;
    //
    struct Account
    {
        nn::account::NetworkServiceAccountId nsaId;
        bool isAgreed;
    } m_Accounts[nn::account::UserCountMax];
    int m_AccountCount = 0;
};

}}}}}
