﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/prepo/detail/service/prepo_Common.h>

namespace nn { namespace prepo { namespace detail { namespace service { namespace core {

/*!
    @brief      統計情報を管理するためのクラスです。
*/
class StatisticsManager
{
private:
    NN_DISALLOW_COPY(StatisticsManager);
    NN_DISALLOW_MOVE(StatisticsManager);

private:
    /*!
        @brief      コンストラクタです。
    */
    StatisticsManager() NN_NOEXCEPT;

public:
    /*!
        @brief      インスタンスを取得します。

        @return     インスタンス。
    */
    static StatisticsManager& GetInstance() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(StatisticsManager, s_Instance);
        return s_Instance;
    }

public:
    /*!
        @brief      統計情報のファイルパスです。
    */
    static const char* FilePath;

public:
    /*!
        @brief      統計情報に、サーバにアップロードされたレポートのバイトサイズと個数を追加します。

        @param[in]  category    レポートの分類。
        @param[in]  size        バイトサイズ。
        @param[in]  count       個数。

        @pre
            - size > 0
            - count > 0
    */
    void AddUploaded(ReportCategory category, size_t size, int count) NN_NOEXCEPT;

    /*!
        @brief      統計情報に、バッファの空きが足りずに失われたレポートのバイトサイズと個数を追加します。

        @param[in]  category    レポートの分類。
        @param[in]  size        バイトサイズ。
        @param[in]  count       個数。

        @pre
            - size > 0
            - count > 0
    */
    void AddLostByBufferShortage(ReportCategory category, size_t size, int count) NN_NOEXCEPT;

    /*!
        @brief      統計情報に、ストレージの空きが足りずに失われたレポートのバイトサイズと個数を追加します。

        @param[in]  category    レポートの分類。
        @param[in]  size        バイトサイズ。
        @param[in]  count       個数。

        @pre
            - size > 0
            - count > 0
    */
    void AddLostByStorageShortage(ReportCategory category, size_t size, int count) NN_NOEXCEPT;

    /*!
        @brief      統計情報に、ユーザー同意がないため失われたレポートのバイトサイズと個数を追加します。

        @param[in]  category    レポートの分類。
        @param[in]  size        バイトサイズ。
        @param[in]  count       個数。

        @pre
            - size > 0
            - count > 0
    */
    void AddLostByDisagree(ReportCategory category, size_t size, int count) NN_NOEXCEPT;

    /*!
        @brief      統計情報を取得します。

        @param[out] out     統計情報。

        @pre
            - out != nullptr
    */
    void GetStatistics(Statistics* out) const NN_NOEXCEPT;

    /*!
        @brief      統計情報をストレージに永続化します。

        @return     処理結果。
    */
    nn::Result Save() NN_NOEXCEPT;

    /*!
        @brief      ストレージに永続化した統計情報を読み込みます。

        @return     処理結果。

        @details
                    ストレージに永続化した統計情報がないときは、何もせずに成功します。
    */
    nn::Result Load() NN_NOEXCEPT;

    /*!
        @brief      ストレージに永続化した統計情報を削除します。

        @return     処理結果。

        @details
                    ストレージに永続化した統計情報がないときは、何もせずに成功します。
    */
    nn::Result Clear() NN_NOEXCEPT;

    /*!
        @brief      統計情報が更新されたときにシグナルされるイベントを登録します。

        @param[in]  event   登録するイベント。

        @pre
            - event != nullptr
    */
    void RegisterUpdateEvent(nn::os::EventType* event) NN_NOEXCEPT;

    /*!
        @brief      統計情報が更新されたときにシグナルされるイベントの登録を解除します。

        @return     登録解除されたイベント。
    */
    nn::os::EventType* UnregisterUpdateEvent() NN_NOEXCEPT;

    /*!
        @brief      統計情報をレポートバッファに追加します。

        @return     処理結果。
    */
    nn::Result Post() NN_NOEXCEPT;

private:
    nn::Result SaveImpl(const Statistics& statistics) NN_NOEXCEPT;
    nn::Result LoadImpl(Statistics* statistics) NN_NOEXCEPT;
    void SignalUpdateEvent() NN_NOEXCEPT;

private:
    mutable nn::os::SdkMutex m_Mutex;
    nn::os::EventType* m_Event;
    Statistics m_Statistics;
};

}}}}}
