﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/fs.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/pdm/detail/pdm_Util.h>
#include <nn/util/util_LockGuard.h>

namespace nn { namespace pdm { namespace detail {

/**
 * @brief   簡易的な直列エントリ記録クラスです。
 *
 * @details 排他制御は行われません。
 */
template<typename TEntry, typename TCounter = uint16_t>
class SimpleEntryRecorder
{
public:
    /**
     * @brief   デフォルトコンストラクタです。正式な利用の前に @ref Initialize() をしてください。
     */
    explicit SimpleEntryRecorder() NN_NOEXCEPT
        : m_pEntryBuffer(nullptr)
        , m_FlushedByteSize(0)
        , m_CommitCount(0)
        , m_TotalCommit(0)
        , m_Capacity(0)
        , m_Filled(0) {}

    /**
     * @brief   コンストラクタです。
     */
    explicit SimpleEntryRecorder(void* pBuffer, const size_t bufferByteSize) NN_NOEXCEPT
    {
        Initialize(pBuffer, bufferByteSize);
    }

    /**
     * @brief   エントリバッファの初期化を行います。
     */
    NN_FORCEINLINE void Initialize(void* pBuffer, const size_t bufferByteSize) NN_NOEXCEPT
    {
        m_pEntryBuffer = reinterpret_cast<TEntry*>(pBuffer);
        m_Capacity = static_cast<TCounter>(bufferByteSize / sizeof(TEntry));
        m_FlushedByteSize = 0;
        m_CommitCount = 0;
        m_TotalCommit = 0;
        m_Filled = 0;
    }

    /**
     * @brief   累計情報、及び割り当てバッファ情報を残して、エントリバッファの初期化を行います。
     */
    NN_FORCEINLINE void Reset() NN_NOEXCEPT
    {
        m_TotalCommit += m_CommitCount;
        m_FlushedByteSize = 0;
        m_CommitCount = 0;
        m_Filled = 0;
    }

    /**
     * @brief   エントリの追加。
     *
     * @return  追加に成功した場合 true が返ります。
     */
    NN_FORCEINLINE bool Add(const TEntry& entry) NN_NOEXCEPT
    {
        if (HasFullEntry())
        {
            return false;
        }
        m_pEntryBuffer[m_Filled++] = entry;
        return true;
    }

    /**
     * @brief   追加済エントリ数をフラッシュします。
     *
     * @return  現在までのフラッシュバイトサイズ(フラッシュしたエントリの総バイト数)を返します。
     *          現在までのフラッシュバイトサイズは、@ref Reset() 、及び @ref Commit() でクリアされます。
     */
    NN_FORCEINLINE const size_t Flush() NN_NOEXCEPT
    {
        const uint32_t flushByte = m_FlushedByteSize + (m_Filled * sizeof(TEntry));
        m_FlushedByteSize = flushByte;
        m_Filled = 0;
        return flushByte;
    }

    /**
     * @brief   コミットカウントをインクリメントします。
     *
     * @details フラッシュバイトサイズをクリーン( 0 )します。
     */
    NN_FORCEINLINE void Commit() NN_NOEXCEPT
    {
        m_FlushedByteSize = 0;
        ++m_CommitCount;
    }

    /**
     * @brief   エントリバッファの追加済エントリ数が上限に達しているか確認します。
     *
     * @return  エントリ数が上限に達している場合 true が返ります。
     */
    NN_FORCEINLINE bool HasFullEntry() const NN_NOEXCEPT
    {
        return (m_Filled >= m_Capacity);
    }

    /**
     * @brief   エントリバッファにエントリが保持されている状態か確認します。
     *
     * @return  追加済エントリ数が 0 でない場合 true が返ります。
     */
    NN_FORCEINLINE bool NeedsFlush() const NN_NOEXCEPT
    {
        return (m_Filled > 0);
    }

    /**
     * @brief   未コミット状態のフラッシュ実績があるか確認します。
     *
     * @return  未コミット状態のフラッシュ実績がある場合 true が返ります。
     */
    NN_FORCEINLINE bool NeedsCommit() const NN_NOEXCEPT
    {
        return (m_FlushedByteSize > 0);
    }

    /**
     * @brief   エントリバッファ先頭を取得します。
     */
    NN_FORCEINLINE const TEntry* GetEntryBuffer() const NN_NOEXCEPT
    {
        return m_pEntryBuffer;
    }

    /**
     * @brief   現在の追加済エントリ数を返します。
     *
     * @details @ref Reset() 及び @ref Flush() でクリアされます。
     */
    NN_FORCEINLINE const TCounter GetFilledCount() const NN_NOEXCEPT
    {
        return m_Filled;
    }

    /**
     * @brief   １回のフラッシュで計上される総バイト数を返します。
     */
    NN_FORCEINLINE const size_t GetFullByteSize() const NN_NOEXCEPT
    {
        return m_Capacity * sizeof(TEntry);
    }

    /**
     * @brief   現在のコミット回数( @ref Commit() コール数 )を返します。
     *
     * @details @ref Reset() でクリアされます。
     */
    NN_FORCEINLINE const uint16_t GetCommitCount() const NN_NOEXCEPT
    {
        return m_CommitCount;
    }

    /**
     * @brief   コミット回数( @ref Commit() コール数 )の累計数を返します。
     */
    NN_FORCEINLINE const uint16_t GetTotalCommitCount() const NN_NOEXCEPT
    {
        return m_TotalCommit;
    }

private:
    TEntry*     m_pEntryBuffer;     //!< エントリバッファ( 外部指定 )
    uint32_t    m_FlushedByteSize;  //!< フラッシュしたエントリ総バイト数
    uint16_t    m_CommitCount;      //!< コミット回数
    uint16_t    m_TotalCommit;      //!< 総コミット回数
    TCounter    m_Capacity;         //!< 追加可能エントリ上限数
    TCounter    m_Filled;           //!< 追加済エントリ数
};

/**
 * @brief   イベントエントリの追記型キャッシュバッファです。
 */
template<class TEventEntry, uint32_t TCapacity = 32>
struct EventEntryCache
{
    static const uint32_t Capacity = TCapacity;
    TEventEntry playEvent[Capacity];
    uint32_t    filledCount;

    EventEntryCache() NN_NOEXCEPT : filledCount(0) {}

    void Add(const TEventEntry& newEntry) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(filledCount < Capacity);
        playEvent[filledCount] = newEntry;
        filledCount++;
    }

    uint32_t GetAvailableSpaceCount() const NN_NOEXCEPT
    {
        return Capacity - filledCount;
    }

    //!----------------------------------------------
    //! @brief 追記型キャッシュの切り替え型ダブルバッファ管理クラスです。
    class Switcher
    {
        NN_DISALLOW_COPY(Switcher);
        NN_DISALLOW_MOVE(Switcher);

    public:
        typedef EventEntryCache<TEventEntry, TCapacity> EntryCache;

        Switcher() NN_NOEXCEPT
            : m_pAddBuffer(&m_Buffers[0])
            , m_pFlushBuffer(&m_Buffers[1])
        {
        }

        void Clear() NN_NOEXCEPT
        {
            NN_UTIL_LOCK_GUARD(m_AddedMutex);
            m_pAddBuffer->filledCount = 0;
        }

        template<class predicateOverflow>
        LockGuard Add(const TEventEntry& newEntry, predicateOverflow onOverflow) NN_NOEXCEPT
        {
            LockGuard lock(m_AddedMutex);
            if (m_pAddBuffer->GetAvailableSpaceCount() == 0)
            {
                onOverflow();
            }
            m_pAddBuffer->Add(newEntry);
            return lock;
        }

        LockGuard SwapBuffer(const uint32_t excessCount = 0) NN_NOEXCEPT
        {
            NN_UTIL_LOCK_GUARD(m_AddedMutex);
            if (m_pAddBuffer->filledCount <= excessCount)
            {
                // 指定超過値以下の場合はなにもしない。
                return LockGuard();
            }
            LockGuard flushLock(m_FlushMutex); // flush 完了前に次の swap が要求された時の排他用.
            std::swap(m_pAddBuffer, m_pFlushBuffer);
            m_pAddBuffer->filledCount = 0;
            return flushLock;
        }

        uint32_t GetCachedEntryCount() const NN_NOEXCEPT
        {
            NN_UTIL_LOCK_GUARD(m_AddedMutex);
            return m_pAddBuffer->filledCount;
        }

        EntryCache* GetFlushBuffer() const NN_NOEXCEPT
        {
            return m_pFlushBuffer;
        }

        LockGuard AcquireLockForAdded() const NN_NOEXCEPT
        {
            return LockGuard(m_AddedMutex);
        }

        LockGuard AcquireLockForFlush() const NN_NOEXCEPT
        {
            return LockGuard(m_FlushMutex);
        }

    private:
        mutable os::SdkRecursiveMutex m_AddedMutex;
        mutable os::SdkRecursiveMutex m_FlushMutex;
        EntryCache*                   m_pAddBuffer;
        EntryCache*                   m_pFlushBuffer;
        EntryCache                    m_Buffers[2];
    };
};

/**
 * @brief   イベントエントリ単位のファイルストリームへの書き込み・読み込みを管理するクラスです。
 *
 * @details コミット、マウント制御は行いません。別途制御が必要になります。
 *          これは、保存媒体としてセーブデータを利用するため、該当制御のファイル単位操作が不適切なためです。
 */
class EventEntryFileStream
{
    NN_DISALLOW_COPY(EventEntryFileStream);
    NN_DISALLOW_MOVE(EventEntryFileStream);

public:
    /**
     * @brief   デフォルトイベントエントリレコードファイル名定数です。
     */
    static const char* const DefaultPlayEventFileName;

    /**
     * @brief   マウントスキームに従ったファイルパスを生成します。
     *
     * @param[in] pOutValue     生成されたファイルパス文字列の受け取る格納コンテナの先頭アドレス。
     * @param[in] capacity      生成されたファイルパス文字列を受け取る格納コンテナの受信可能容量( byte数 )。
     * @param[in] pMountedName  マウントボリューム名。
     * @param[in] pOpenFileName オープン対象ファイル名。
     *
     * @return  生成されたファイルパス文字列の先頭アドレス。( 引数の @ref pOutValue と同じです )
     */
    static const char* MakeFilePath(char* pOutValue, int capacity, const char* pMountedName, const char* pOpenFileName) NN_NOEXCEPT;

protected:
    /**
     * @brief   @ref OpenFile() 関数の返り値の定義です。
     */
    enum class OpenFileResult : Bit8
    {
        Opened,         //!< 既存のファイルを開きました。
        NewlyCreated,   //!< ファイルを新規作成しました。
        Failed,         //!< ファイルを開けませんでした。
    };

    /**
     * @brief   @ref Open() 関数の返り値の定義です。
     */
    enum class OpenResult : Bit8
    {
        Error,          //!< オープンに失敗しました。( 現在返却されず、@ref Open() 内でABORTします )
        Opened,         //!< 既存ファイルでのストリームオープンに成功しました。
        NewlyCreated,   //!< 新規ファイルでのストリームオープンに成功しました。
        AlreadyOpened,  //!< 既にオープンしています。
    };

    /**
     * @brief   ストリーム共通ヘッダ
     */
    struct Header
    {
        uint32_t start;
        uint32_t count;

        Header() NN_NOEXCEPT : start(0), count(0) {}
        inline void Reset(uint32_t newStart = 0, uint32_t newCount = 0) NN_NOEXCEPT
        {
            start = newStart;
            count = newCount;
        }
    };

    //! @brief      ストリーム共通ヘッダの書き込み時に呼び出されるメソッドです。このメソッドをオーバーライドする事でカスタムヘッダを制御できます。
    virtual Result OnHeaderWrite(fs::FileHandle fileHandle, int64_t position, const fs::WriteOption inheritedOption, size_t* pOutWriteSize = nullptr) NN_NOEXCEPT;

    //! @brief      ストリーム共通ヘッダの読み込み時に呼び出されるメソッドです。このメソッドをオーバーライドする事でカスタムヘッダを制御できます。
    virtual Result OnHeaderRead(fs::FileHandle fileHandle, int64_t position, size_t* pOutReadSize = nullptr) NN_NOEXCEPT;

    //! @brief      ストリーム共通ヘッダのリセット時に呼び出されるメソッドです。このメソッドをオーバーライドする事でカスタムヘッダを制御できます。
    virtual void OnHeaderReset() NN_NOEXCEPT;

    /**
     * @brief       ストリーム共通ヘッダを取得します。( スレッドセーフ )
     *
     * @return      ストリーム共通ヘッダのコピー。
     *
     * @details     このメソッドの非スレッドセーフ版として @ref GetHeaderUnsafe() があります。
     */
    Header  GetHeader() const NN_NOEXCEPT;

    /**
     * @brief       ストリーム共通ヘッダを取得します。( 非スレッドセーフ )
     *
     * @return      ストリーム共通ヘッダのコピー。
     *
     * @details     このメソッドのスレッドセーフ版として @ref GetHeader() があります。
     */
    NN_FORCEINLINE Header  GetHeaderUnsafe() const NN_NOEXCEPT
    {
        return m_Header;
    }

    /**
     * @brief       記録のあるイベントの開始インデックスを取得します。( 非スレッドセーフ版 )
     *
     * @return      インデックス値。
     *
     * @details     このメソッドのスレッドセーフ版は @ref GetStartIndex() があります。@n
     *              ストリームオープン状態を確認しません。@n
     *              ストリームオープン状態に依存する用途では @ref IsOpenedUnsafe() もしくは @ref IsOpened() での確認後に利用してください。
     */
    NN_FORCEINLINE uint32_t GetStartIndexUnsafe() const NN_NOEXCEPT
    {
        return m_Header.start;
    }

    /**
     * @brief       記録のあるイベントの最終インデックスを取得します。( 非スレッドセーフ版 )
     *
     * @return      最終インデクス値。
     *
     * @details     このメソッドのスレッドセーフ版は @ref GetLastIndex() があります。@n
     *              ストリームオープン状態を確認しません。@n
     *              ストリームオープン状態に依存する用途では @ref IsOpenedUnsafe() もしくは @ref IsOpened() での確認後に利用してください。
     */
    NN_FORCEINLINE uint32_t GetLastIndexUnsafe() const NN_NOEXCEPT
    {
        const auto header = m_Header;
        return (header.count == 0u) ? 0u : (header.start + header.count - 1);
    }

    /**
     * @brief       記録のあるイベントの数を取得します。( 非スレッドセーフ版 )
     *
     * @return      イベントの数。
     *
     * @details     このメソッドのスレッドセーフ版は @ref GetCount() があります。@n
     *              ストリームオープン状態を確認しません。@n
     *              ストリームオープン状態に依存する用途では @ref IsOpenedUnsafe() もしくは @ref IsOpened() での確認後に利用してください。
     */
    NN_FORCEINLINE uint32_t GetCountUnsafe() const NN_NOEXCEPT
    {
        return m_Header.count;
    }

    /**
     * @brief       記録できるイベントの最大数を取得します。( 非スレッドセーフ版 )
     *
     * @return      イベントの最大数。
     *
     * @details     このメソッドのスレッドセーフ版は @ref GetCountMax() があります。@n
     *              ストリームオープン状態を確認しません。@n
     *              ストリームオープン状態に依存する用途では @ref IsOpenedUnsafe() もしくは @ref IsOpened() での確認後に利用してください。
     */
    NN_FORCEINLINE uint32_t GetCountMaxUnsafe() const NN_NOEXCEPT
    {
        return m_EventCountMax;
    }

    /**
     * @brief       ストリームがオープンしているか確認します。( スレッドセーフ )
     *
     * @return      ストリームがオープン済であれば true が返されます。
     *
     * @details     このメソッドの非スレッドセーフ版として @ref IsOpenedUnsafe() があります。
     */
    bool    IsOpened() const NN_NOEXCEPT;

    /**
     * @brief       ストリームがオープンしているか確認します。( 非スレッドセーフ )
     *
     * @return      ストリームがオープン済であれば true が返されます。
     *
     * @details     このメソッドのスレッドセーフ版として @ref IsOpened() があります。
     */
    NN_FORCEINLINE bool IsOpenedUnsafe() const NN_NOEXCEPT
    {
        return m_IsStreamOpened;
    }

    /**
     * @brief       ストリーム保存対象ファイルパスが準備されているか確認します。( 非スレッドセーフ )
     *
     * @return      準備済であれば true が返されます。
     */
    NN_FORCEINLINE bool IsPreparedUnsafe() const NN_NOEXCEPT
    {
        return m_FilePath[0] != '\0';
    }

    /**
     * @brief       コンストラクタです。
     *
     * @param[in]   headerSize      ファイルに保存するヘッダ領域サイズ( byte )。@ref EventEntryFileStream::Header 以上が必要です。
     * @param[in]   entryUnitSize   ファイルに保存するエントリ単位サイズ( byte )。
     *
     * @pre
     *              - headerSize >= sizeof(EventEntryFileStream::Header)
     *              - entryUnitSize > 0
     *
     */
    explicit EventEntryFileStream(uint16_t headerSize, uint16_t entryUnitSize) NN_NOEXCEPT;

    /**
     * @brief       記録対象データベースファイルパスを準備します。
     *
     * @param[in]   pFileMountedVolumeName  ストリーム運用時にマウント確立が保証されるボリューム名。
     *                                      ポインタで保持するため参照先メモリは維持される必要があります。
     * @param[in]   pOpenFileName           ストリームを保存するストレージファイル名。
     *                                      nullptr の場合は、"PlayEvent.dat" が適用されます。
     *
     * @pre
     *              - pFileMountedVolumeName != nullptr
     */
    void PrepareDatabaseFilePath(const char* pFileMountedVolumeName, const char* pOpenFileName = nullptr) NN_NOEXCEPT;

    /**
     * @brief       ファイルを読み込み、記録されたイベント数などの情報を初期化します。他の関数を呼ぶ前に必ず一度呼ぶ必要があります。
     *
     * @param[in]   eventCountMax       ファイルに保存できるイベントの最大数。
     * @param[in]   isExistOnly         true の場合、ファイルが存在する場合のみストリームオープンの成功とします。@n
     *                                  false の場合、ファイルが存在しない場合生成します。
     *
     * @return      ストリームオープンの結果を返します。
     *              - Opened            既存ファイルでのストリームオープンに成功しました。
     *              - NewlyCreated      新規ファイルでのストリームオープンに成功しました。
     *              - AlreadyOpened     既にオープンしています。
     *              - Error             オープンに失敗しました。( true == isExistOnly 時のみ )
     *
     * @pre
     *              - eventCountMax > 0
     *              - @ref PrepareDatabaseFilePath() でデータベースファイルパスが準備されている。
     *
     * @details     イベント最大数指定引数の存在は、PlayEventBuffer::Initialize との互換性のためです。
     */
    OpenResult OpenUnsafe(uint32_t eventCountMax, bool isExistOnly) NN_NOEXCEPT;

    /**
     * @brief       ファイルを読み込み、記録されたイベント数などの情報を初期化します。@n
     *              他の関数を呼ぶ前に必ず一度呼ぶ必要があります。@n
     *              @ref PlayEventBuffer クラス用です。
     *
     * @param[in]   eventCountMax           ファイルに保存できるイベントの最大数。
     * @param[in]   pFileMountedVolumeName  ストリーム運用時にマウント確立が保証されるボリューム名。
     *                                      ポインタで保持するため参照先メモリは維持される必要があります。
     *
     * @return      ストリームオープンの結果を返します。
     *              - Opened            既存ファイルでのストリームオープンに成功しました。
     *              - NewlyCreated      新規ファイルでのストリームオープンに成功しました。
     *              - AlreadyOpened     既にオープンしています。
     *
     * @pre
     *              - eventCountMax > 0
     *
     * @details     イベント最大数指定引数の存在は、PlayEventBuffer::Initialize との互換性のためです。
     */
    NN_FORCEINLINE OpenResult Open(uint32_t eventCountMax, const char* pFileMountedVolumeName) NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(m_FileMutex);
        PrepareDatabaseFilePath(pFileMountedVolumeName);
        return OpenUnsafe(eventCountMax, false);
    }

    /**
     * @brief       ストリームを閉じます。再利用には @ref Open() の実施が必要です。
     */
    void Close() NN_NOEXCEPT;

    /**
     * @brief       ストリームに保存している記録を破棄します。ストレージ保存されたファイルも削除されます。
     *
     * @details     ストリームオープン状態で実施した場合、以後 @ref Read() 系APIが使えなくなります。
     *              利用するには @ref Close() 後に @ref Open を呼んで再オープンするか、
     *              @ref Write() によって何かしら書き込みが必要です。
     *
     */
    void Discard() NN_NOEXCEPT;

    /**
     * @brief       イベントエントリを書き込みます。
     *              本メソッドで書き込まれた内容はコミットが完了するまでストレージには反映しません。
     *
     * @param[in]   pWriteEntrySource   書き込み元ソース先頭メモリアドレス。
     * @param[in]   writeEntryCount     書き込み元ソース先頭からの有効なイベントエントリ数。
     *
     * @return      ファイル書き込みの排他ロックハンドル。スコープアウトでロック解除されます。
     *
     * @pre
     *              - pWriteEntrySource != nullptr
     *              - writeEntryCount > 0
     */
    LockGuard Write(const void* pWriteEntrySource, uint32_t writeEntryCount) NN_NOEXCEPT;

    /**
     * @brief       記録されたイベントを読み込みます。事前に @ref BeginRead() を呼ぶ必要があります。
     *
     * @param[out]  pOutValue 読み込み先。コンストラクタで指定したエントリ単位サイズを１ベントとして outCountMax 分受け取れる容量が必要です。
     * @param[in]   startIndex 読み込みを開始するイベントのインデックス。
     * @param[in]   outCountMax 読み込むイベントの最大数。
     *
     * @return      読み込んだイベントの数。
     *
     * @pre
     *              - pOutValue != nullptr
     *              - pOutValue >= entryUnitSize(コンストラクタで指定する引数値) * outCountMax
     *              - outCountMax > 0
     *              - startIndex > @ref GetStartIndex()
     *              - @ref BeginRead() を呼んで読み込み環境を開いている。
     *              - @ref BeginRead() 後に @ref EndRead() で読み込み環境を閉じていない。
     */
    uint32_t ReadImpl(void* pOutValue, uint32_t startIndex, uint32_t outCountMax) const NN_NOEXCEPT;

    /**
     * @brief       イベントエントリを書き込みます。( 非スレッドセーフ版 )
     *              本メソッドで書き込まれた内容はコミットが完了するまでストレージには反映しません。
     *
     * @param[in]   pWriteEntrySource   書き込み元ソース先頭メモリアドレス。
     * @param[in]   writeEntryCount     書き込み元ソース先頭からの有効なイベントエントリ数。
     *
     * @pre
     *              - pWriteEntrySource != nullptr
     *              - writeEntryCount > 0
     */
    void WriteUnsafe(const void* pWriteEntrySource, uint32_t writeEntryCount) NN_NOEXCEPT;

    /**
     * @brief       ヘッダのみを書き込みます。( 非スレッドセーフ版 )
     *              本メソッドで書き込まれた内容はコミットが完了するまでストレージには反映しません。
     */
    void WriteHeaderOnlyUnsafe() NN_NOEXCEPT;

    /**
     * @brief       強制的に無効化状態( ファイルアクセスが出来ない )で @ref BeginRead() を行うエイリアスです。
     *
     * @details     ファイル操作の排他制御を行う mutex をロックします。
     */
    void BeginReadWithForceInvalidation() NN_NOEXCEPT;

public:
    /**
     * @brief       ファイル読み込みを開始します。
     *
     * @details     ファイル操作の排他制御を行う mutex をロックし、読み込み用にファイルを開きます。
     */
    virtual void BeginRead() NN_NOEXCEPT;

    /**
     * @brief       ファイル読み込みを終了します。
     *
     * @details     BeginRead で開いたファイルをクローズし、ファイル操作の排他制御を行う mutex を Unlock します。
     */
    virtual void EndRead() NN_NOEXCEPT;

    /**
     * @brief       記録のあるイベントの開始インデックスを取得します。
     *
     * @return      インデックス値。
     *
     * @details     この関数で得られたインデックス値は @ref Read() 関数で取得可能なイベントのインデックスの最小値と一致します。@n
     *              記録されたイベントの数が記録できるイベントの最大数を超えた場合には、古いイベントを新しいイベントで置き換えていくため、
     *              0 より大きな値になることがあります。
     */
    uint32_t GetStartIndex() const NN_NOEXCEPT;

    /**
     * @brief       記録のあるイベントの最終インデックスを取得します。
     *
     * @return      インデックス値。
     *
     * @details     この関数で得られたインデックス値は @ref Read() 関数で取得可能なイベントのインデックスの最大値と一致します。
     *              記録されたイベントの数が記録できるイベントの最大数を超えた場合には、古いイベントを新しいイベントで置き換えていくため、
     *              @ref GetCountMax() で得られる値 - 1 よりも大きな値になる事があります。
     */
    uint32_t GetLastIndex() const NN_NOEXCEPT;

    /**
     * @brief       記録のあるイベントの数を取得します。
     *
     * @return      イベントの数。
     */
    uint32_t GetCount() const NN_NOEXCEPT;

    /**
     * @brief       記録できるイベントの最大数を取得します。
     *
     * @return      イベントの最大数。
     */
    uint32_t GetCountMax() const NN_NOEXCEPT;

    /**
     * @brief       ファイル操作の排他制御を行う mutex を取得します。
     *
     * @return      ファイル操作の排他制御を行う mutex
     */
    inline os::SdkRecursiveMutex& GetFileMutex() NN_NOEXCEPT
    {
        return m_FileMutex;
    }

    /**
     * @brief       ファイル操作の排他制御を行う lock_guard mutex を取得します。
     *
     * @return      ファイル操作の排他制御を行う lock_guard mutex。
     */
    inline LockGuard AcquireLockForFile() const NN_NOEXCEPT
    {
        return LockGuard(m_FileMutex);
    }

private:
    /**
     * @brief       PlayEvent を記録したファイルを開きます。ファイルが存在しない場合は新しく作成します。
     *
     * @param[out]  pOutHandle 開いたファイルのハンドル。
     */
    OpenFileResult OpenFile(fs::FileHandle* pOutHandle) NN_NOEXCEPT;

    /**
     * @brief       PlayEvent を記録したファイルを開きます。@n
     *              ファイルが存在した場合はヘッダを読み込みます。@n
     *              ファイルが存在しない場合は、新規作成を行わずヘッダ情報はリセットされます。
     *
     * @param[out]  pOutHandle 開いたファイルのハンドル。
     */
    OpenFileResult OpenFileForExistOnly(fs::FileHandle* pOutHandle) NN_NOEXCEPT;

    /**
     * @brief       index 番目のイベントのファイル中でのオフセット位置を取得します。
     *
     * @param[in]   eventIndex インデックス値。
     *
     * @return      index 番目イベントに対応するファイル先頭からのシークオフセット( byte )。
     */
    int64_t ComputeFilePosition(uint32_t eventIndex) const NN_NOEXCEPT
    {
        return m_HeaderSize + (eventIndex % m_EventCountMax) * static_cast<int64_t>(m_EntrySize);
    }

    mutable os::SdkRecursiveMutex m_FileMutex;
    fs::FileHandle          m_ReadFileHandle;
    Header                  m_Header;
    uint32_t                m_EventCountMax;    // const にしたいぃぃ
    const uint16_t          m_HeaderSize;
    const uint16_t          m_EntrySize;
    char                    m_FilePath[32];
    bool                    m_ReadFileOpened;
    bool                    m_IsStreamOpened;
};

}}}
