﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief 抽象化された電源を操作する関数の宣言。
 */

#pragma once

#include <nn/nn_Common.h>

#include <nn/pcv/pcv_Types.h>

namespace nn { namespace pcv { namespace driver {

//! @name 抽象化された電源を操作する関数。
//! @{

/**
 * @brief 電源を指定した電圧で ON にします。
 *
 * @param[in]   powerControlTarget 電源。
 * @param[in]   microVolt 電圧(uV)。
 * @return      処理の結果を返します。
 * @retval      ResultSuccess 成功しました。
 *
 * @pre nn::pcv::driver::Initialize 関数を実行し PCV ドライバライブラリを初期化する必要があります。
 *
 * @detail 電源を指定した電圧で ON にします。@n
 * 本関数の電源は物理的な電源ドメインと必ずしも対応しません。
 */
Result PowerOn(PowerControlTarget powerControlTarget, MicroVolt microVolt) NN_NOEXCEPT;

/**
 * @brief 電源を OFF にします。
 *
 * @param[in]   powerControlTarget 電源。
 * @return      処理の結果を返します。
 * @retval      ResultSuccess 成功しました。
 *
 * @pre nn::pcv::driver::Initialize 関数を実行し PCV ドライバライブラリを初期化する必要があります。
 *
 * @detail 電源を OFF にします。@n
 * 本関数の電源は物理的な電源ドメインと必ずしも対応しません。
 */
Result PowerOff(PowerControlTarget powerControlTarget) NN_NOEXCEPT;

/**
 * @brief 電源を指定した電圧に変更します。
 *
 * @param[in]   powerControlTarget 電源。
 * @param[in]   microVolt 電圧(uV)。
 * @return      処理の結果を返します。
 * @retval      ResultSuccess 成功しました。
 *
 * @pre nn::pcv::driver::Initialize 関数を実行し PCV ドライバライブラリを初期化する必要があります。
 *
 * @detail 電源を指定した電圧に変更します。@n
 * 本関数の電源は物理的な電源ドメインと必ずしも対応しません。
 */
Result ChangeVoltage(PowerControlTarget powerControlTarget, MicroVolt microVolt) NN_NOEXCEPT;

/**
 * @brief 電源が ON であるかを取得します。
 *
 * @param[out]  pOutIsPoweredOn 電源が ON である場合 true。
 * @param[in]   powerControlTarget 電源。
 * @return      処理の結果を返します。
 * @retval      ResultSuccess 成功しました。
 *
 * @pre nn::pcv::Initialize 関数を実行し PCV ドライバライブラリを初期化する必要があります。
 *
 * @detail 電源が ON であるかを取得します。@n
 * 本関数の電源は物理的な電源ドメインと必ずしも対応しません。
 */
Result IsPoweredOn(bool* pOutIsPoweredOn, PowerControlTarget powerControlTarget) NN_NOEXCEPT;

//! @}

}}} // namespace nn::pcv::driver
