﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief PCV ドライバライブラリのエラーレポート用 API 宣言。
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/os.h>

#include "../pcv_ErrorReportTypes.h"

namespace nn {

namespace pcv {
namespace driver {

/**
 * @brief       エラーレポート用のシステムイベントを取得します。
 * @return      エラーレポート用のシステムイベントへのポインタ。
 */
nn::os::SystemEventType* GetPowerClockInfoEvent() NN_NOEXCEPT;

/**
 * @brief       OSC のクロックを取得します。
 * @return      OSC のクロック周波数 (Hz)
 */
uint32_t GetOscillatorClock() NN_NOEXCEPT;

/**
 * @brief       DVFS テーブルを取得します。
 *
 * @param[out]  pOutClocks    クロック周波数を格納する配列バッファ
 * @param[out]  pOutVoltages  電圧を格納する配列バッファ
 * @param[out]  pOutCount     pOutCloks および pOutVoltages に返り値として格納された要素数
 * @param[in]   moduleId      DVFS テーブル取得対象のモジュール
 * @param[in]   maxCount      pOutCloks および pOutVoltages に格納可能な最大要素数
 */
Result GetDvfsTable(uint32_t* pOutClocks, int32_t* pOutVoltages, int32_t* pOutCount, Module moduleId, int32_t maxCount) NN_NOEXCEPT;

/**
 * @brief       モジュールごとの情報を取得します。
 *
 * @param[out]  pOutModuleStates      ModuleState を格納する配列バッファ
 * @param[out]  pOutCount             pOutModuleStates に返り値として格納された要素数
 * @param[in]   maxCount              pOutModuleStates に格納可能な最大要素数
 */
void GetModuleStateTable(ModuleState* pOutModuleStates, int32_t* pOutCount, int32_t maxCount) NN_NOEXCEPT;

/**
 * @brief       電源ドメインごとの情報を取得します。
 *
 * @param[out]  pOutPowerDomainStates PowerDomainState を格納する配列バッファ
 * @param[out]  pOutCount             pOutPowerDomainStates に返り値として格納された要素数
 * @param[in]   maxCount              pOutPowerDomainStates に格納可能な最大要素数
 */
void GetPowerDomainStateTable(PowerDomainState* pOutPowerDomainStates, int32_t* pOutCount, int32_t maxCount) NN_NOEXCEPT;

/**
 * @brief       チップ情報を取得します。
 *
 * @param[out]  pOutFuseInfo          チップ情報の値を格納する配列バッファ
 * @param[out]  pOutCount             pOutFuseInfo に返り値として格納された要素数
 * @param[in]   maxCount              pOutFuseInfo に格納可能な最大要素数
 */
void GetFuseInfo(uint32_t* pOutValues, int32_t* pOutCount, int32_t maxCount) NN_NOEXCEPT;

/**
 * @brief       Dram ID を取得します。
 *
 * @param[out]  pOutDramId          Dram ID の値を格納する配列バッファ
 */
void GetDramId(uint32_t* pOutDramId) NN_NOEXCEPT;

} // driver
} // pcv
} // nn
