﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief ペアレンタルコントロールの設定取得に関する pctl ライブラリの関数を提供します。
 */

#include <nn/pctl/pctl_TypesSettings.h>

namespace nn { namespace pctl {

/**
 * @name ペアレンタルコントロール設定取得関数
 * @{
 */

/**
 * @brief ペアレンタルコントロールによる制限が有効であるかどうかを返します。
 * @return 制限が有効である場合は true
 *
 * @details
 * 本関数は既にペアレンタルコントロールの設定がなされているかどうかで
 * 処理を分岐したい場合に用います。ペアレンタルコントロールによる制限が行われる
 * 機能を利用する場面で、その機能が制限されているかどうかを確認したい場合は、
 * 本関数ではなく @ref CheckFreeCommunicationPermission などの
 * 各場面に応じた関数を用いるようにしてください。
 *
 * 本関数はペアレンタルコントロールの暗証番号が設定されている場合に true、
 * 設定されていない場合に false を返します。
 */
bool IsRestrictionEnabled() NN_NOEXCEPT;

/**
 * @brief ペアレンタルコントロール設定の制限レベルを取得します。
 * @return 設定された制限レベル(@ref SafetyLevel)を表す値
 *
 * @details
 * ペアレンタルコントロールが一度も設定されていない場合(システム初期状態から、
 * または設定が削除されてから一度も制限レベルの設定が行われていない場合)、
 * 本関数は @ref SafetyLevel_None を返します。
 *
 * なお、SafetyLevel_None は制限レベルにおける「制限なし」設定を示しており、
 * 暗証番号が設定されている場合にも SafetyLevel_None が返る場合があります。
 */
SafetyLevel GetSafetyLevel() NN_NOEXCEPT;

/**
 * @brief 現在のペアレンタルコントロールによる制限設定を一括で取得します。
 * @param[out] pSettings 設定値を受け取るポインター
 * @pre
 *  - pSettings != nullptr
 *
 * @details
 * ペアレンタルコントロールが一度も設定されていない場合、
 * 本関数はすべて 0 に相当する設定値を返します。
 * これには、システム初期状態、および設定が削除されてから
 * 一度も制限レベルの設定が行われていない場合が含まれます。
 */
void GetCurrentSettings(RestrictionSettings* pSettings) NN_NOEXCEPT;

/**
 * @}
 */

}}
