﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/pctl/pctl_TypesSystem.h>
#include <nn/pctl/pctl_ResultSystem.h>
#include <nn/pctl/pctl_ApiSystemReadOnly.h>
#include <nn/pctl/detail/service/watcher/pctl_EventLog.h>

namespace nn { namespace pctl {

/**
 * @}
 */
/**
 * @name ペアコン設定変更関数（デバッグ用）
 * @{
 */

/**
 * @brief (デバッグ用) アプリケーション別の「他の人との自由なコミュニケーション」の利用制限リストからアプリケーションを削除します。
 * @param[in] applicationId アプリケーションのアプリケーションID
 * @pre
 *  - applicationId != nn::ncm::ApplicationId::GetInvalidId()
 *
 * @details
 * 本関数はデバッグ用です。製品では本体初期化時にリストが全削除されます(個別削除はありません)。
 *
 * @note
 * - 他のリスト更新を伴う関数と異なり、本関数はストレージへの反映が終わるまで処理をブロックします。
 */
void DeleteFromFreeCommunicationApplicationListForDebug(nn::ncm::ApplicationId applicationId) NN_NOEXCEPT;

/**
 * @brief (デバッグ用) アプリケーション別の「他の人との自由なコミュニケーション」の利用制限リストを全削除します。
 *
 * @details
 * 本関数はデバッグ用です。製品では本体初期化時にリストが全削除されます。
 *
 * @note
 * - 他のリスト更新を伴う関数と異なり、本関数はストレージへの反映が終わるまで処理をブロックします。
 */
void ClearFreeCommunicationApplicationListForDebug() NN_NOEXCEPT;

/**
 * @brief (デバッグ用) 制限対象外リストに追加されたアプリケーションの総数を返します。
 * @return アプリケーションの数
 *
 * @details
 * 本関数はデバッグ用です。
 * 本関数が返すアプリケーションの数は、呼び出し時点における @ref GetExemptApplicationListForDebug で
 * 得られるデータの総数に一致します。
 *
 * @ref AddToExemptApplicationListForDebug の非同期処理が実行中である場合、
 * 本関数はその処理(数10ms程度)が終わるまで呼び出し元をブロックします。
 */
int GetExemptApplicationListCountForDebug() NN_NOEXCEPT;

/**
 * @brief (デバッグ用) アプリケーション別の制限対象外リストを取得します。
 * @param[out] pOutInfoArray 制限対象外データを受け取るための配列
 * @param[in] offset データの取得を開始する内部配列のオフセット
 * @param[in] count pOutInfoArray で受け取れる最大の要素数
 * @return pOutInfoArray に出力された設定値の数(要素数)
 * @pre
 *  - pOutInfoArray != nullptr
 *  - offset >= 0
 *  - count >= 0
 * @post
 *  - 戻り値 <= count
 *  - 戻り値が 0 より大きい場合、pOutInfoArray は 0 から (戻り値 - 1) のインデックスに対応する要素が有効である
 *
 * @details
 * 本関数はデバッグ用です。
 * offset から数えた内部配列の要素数が count より小さい場合、本関数の戻り値は count より少なくなります。
 * 特に、offset が内部配列の全体の要素数以上になっている場合、戻り値は 0 となります。
 *
 * @ref AddToExemptApplicationListForDebug の非同期処理が実行中である場合、
 * 本関数はその処理(数10ms程度)が終わるまで呼び出し元をブロックします。
 */
int GetExemptApplicationListForDebug(ExemptApplicationInfo* pOutInfoArray, int offset, int count) NN_NOEXCEPT;

/**
 * @brief (デバッグ用) アプリケーション別の制限対象外リストを更新します。
 * @param[in] pInfoArray アプリケーション別の制限対象外データを保持する配列
 * @param[in] count pInfoArray の要素数
 * @pre
 *  - pInfoArray != nullptr
 *  - count > 0
 *  - pInfoArray の各要素はすべて内部配列に登録されているアプリケーションである
 *    (@ref GetExemptApplicationList で取得されるリストに含まれるアプリケーションである)
 *
 * @details
 * 本関数はデバッグ用です。
 * 本関数に指定する pInfoArray の要素数は必ずしも @ref GetExemptApplicationListCountForDebug で
 * 得られる数と一致する必要はありません。内部配列に存在するものの pInfoArray に
 * 含まれていないアプリケーションに対しては、設定値の削除は行われずそのまま内部配列に残ります。
 *
 * pInfoArray の要素として同じアプリケーションに対する設定値が複数含まれていた場合の挙動は未定義です。
 *
 * @note
 * 本関数は設定値をストレージへ書き込む処理を行いますが、
 * この書き込み処理は並列(非同期)で行うため、その処理の間ブロックすることはありません。
 */
void UpdateExemptApplicationListForDebug(const ExemptApplicationInfo* pInfoArray, int count) NN_NOEXCEPT;

/**
 * @brief (デバッグ用) アプリケーション別の制限対象外リストにアプリケーションを追加します。
 * @param[in] applicationId アプリケーションのアプリケーションID
 * @pre
 *  - applicationId != nn::ncm::ApplicationId::GetInvalidId()
 *
 * @details
 * 本関数はデバッグ用です。
 * 本関数は、指定されたアプリケーションIDをシステム内部で管理する配列に追加します。
 * 既に内部配列にアプリケーションが含まれている場合は何も処理を行いません。
 *
 * メモリ上の内部配列への追加処理は同期的に行われるため、本関数は
 * その処理の間呼び出し元をブロックします。
 *
 * @note
 * - リスト追加への同期処理は内部のメモリ上のリストに追加する処理がメインとなります。
 *   ストレージへの反映(書き込み)は並列に行うため、その処理の間
 *   追加で呼び出し元をブロックすることはありません。
 */
void AddToExemptApplicationListForDebug(nn::ncm::ApplicationId applicationId) NN_NOEXCEPT;

/**
 * @brief (デバッグ用) アプリケーション別の制限対象外リストからアプリケーションを削除します。
 * @param[in] applicationId アプリケーションのアプリケーションID
 * @pre
 *  - applicationId != nn::ncm::ApplicationId::GetInvalidId()
 *
 * @details
 * 本関数はデバッグ用です。製品では本体初期化時にリストが全削除されます(個別削除はありません)。
 *
 * @note
 * - 他のリスト更新を伴う関数と異なり、本関数はストレージへの反映が終わるまで処理をブロックします。
 */
void DeleteFromExemptApplicationListForDebug(nn::ncm::ApplicationId applicationId) NN_NOEXCEPT;

/**
 * @brief (デバッグ用) アプリケーション別の制限対象外リストを全削除します。
 *
 * @details
 * 本関数はデバッグ用です。製品では本体初期化時にリストが全削除されます。
 *
 * @note
 * - 他のリスト更新を伴う関数と異なり、本関数はストレージへの反映が終わるまで処理をブロックします。
 */
void ClearExemptApplicationListForDebug() NN_NOEXCEPT;


/**
 * @brief (デバッグ用) イベントの送信処理を行います。
 *
 * @param[out] pOutCount     送信されたイベントの個数
 * @param[out] pOutEventData 送信されたイベントが格納される配列
 * @param[in]  eventDataSize \a pOutEventData で与えられる配列の長さ
 *
 * @details
 * 本関数はデバッグ用です。
 * \a pOutEventData 配列の長さが十分でない場合、 \a eventDataSize よりも大きい値が \a pOutCount として返されることがあります。
 */
nn::Result RequestPostEvent(int* pOutCount, detail::service::watcher::EventData* pOutEventData, int eventDataSize) NN_NOEXCEPT;

/**
 * @}
 */

}}
