﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/pctl/detail/pctl_Log.h>
#include <nn/pctl/detail/service/pctl_PlayState.h>
#include <nn/pctl/detail/service/watcher/pctl_EventLog.h>
#include <nn/time/time_TimeZoneRule.h>
#include <nn/util/util_LockGuard.h>

namespace nn { namespace time {
struct CalendarAdditionalInfo;
}}

namespace nn { namespace pctl { namespace detail { namespace service { namespace watcher {

struct EventLogDataHeader
{
    uint16_t startIndex;      //!< 記録を開始した位置を表すインデックス番号
    uint16_t lastIndex;       //!< 最後に記録した位置を表すインデックス番号(startIndex 未満の場合はループ)
    uint16_t lastIndexOnLoad; //!< ファイル読み込み時点の lastIndex の値
    bool isAnyRecorded;       //!< 何か記録されていれば true
    bool isAnyRecordedOnLoad; //!< ファイル読み込み時点で何か記録されていれば true
};

struct EventLogFileData
{
    EventLogDataHeader header;
    EventData storage[MaxEventCount];
};

class WatcherEventStorage
{
public:
    WatcherEventStorage() NN_NOEXCEPT;

    // @brief インスタンスの初期化を行います。
    void Initialize() NN_NOEXCEPT;

    // @brief インスタンスを再初期化します。(テスト用)
    inline void ReinitializeForTest() NN_NOEXCEPT
    {
        Initialize();
    }

    // @brief イベントを追加します。
    // @param[in] eventType イベントの種類
    inline void AddEvent(EventType eventType) NN_NOEXCEPT
    {
        AddEvent(eventType, nullptr);
    }

    // @brief イベントを追加します。
    // @param[in] eventType イベントの種類
    // @param[in] pPayload 追加情報データ(不要な場合は nullptr)
    void AddEvent(EventType eventType, const EventData::Payload* pPayload) NN_NOEXCEPT;

    // @brief イベントを追加します。
    // @param[in] eventType イベントの種類
    // @param[in] timestamp 発生時刻
    // @param[in] pPayload 追加情報データ(不要な場合は nullptr)
    void AddEvent(EventType eventType, const nn::time::PosixTime& timestamp, const EventData::Payload* pPayload) NN_NOEXCEPT;

    // @brief イベントを挿入追加します。
    // @param[in] eventType イベントの種類
    // @param[in] pPayload 追加情報データ(不要な場合は nullptr)
    // @param[in] timeAt イベントの発生時刻
    // @details timeAt の値に応じて挿入場所が決定されます。
    void InsertEvent(EventType eventType, const EventData::Payload* pPayload, const nn::time::PosixTime& timeAt) NN_NOEXCEPT;

    // @brief イベントを挿入追加します。
    // @param[in] eventType イベントの種類
    // @param[in] locationName タイムゾーン地域名
    // @param[in] pPayload 追加情報データ(不要な場合は nullptr)
    // @param[in] timeAt イベントの発生時刻
    // @details timeAt の値に応じて挿入場所が決定されます。
    void InsertEvent(EventType eventType, const nn::time::LocationName& locationName, const EventData::Payload* pPayload, const nn::time::PosixTime& timeAt) NN_NOEXCEPT;

    // @brief 本体起動イベントを追加します。
    inline void AddDeviceLaunchEvent(const nn::time::PosixTime& timestamp) NN_NOEXCEPT
    {
        AddEvent(EventType::DidDeviceLaunch, timestamp, nullptr);
    }

    // @brief 本体起床イベントを追加します。
    inline void AddWakeupEvent() NN_NOEXCEPT
    {
        AddEvent(EventType::DidWakeup);
    }

    // @brief 本体スリープイベントを追加します。
    inline void AddSleepEvent() NN_NOEXCEPT
    {
        AddEvent(EventType::DidSleep);
    }

    // @brief アプリケーション起動イベントを追加します。
    inline void InsertApplicationLaunchEvent(const nn::ncm::ApplicationId& applicationId, const nn::time::PosixTime& timeAt) NN_NOEXCEPT
    {
        EventData::Payload payload;
        payload.applicationLaunch.applicationId = applicationId;
        InsertEvent(EventType::DidApplicationLaunch, &payload, timeAt);
    }

    // @brief アプリケーション終了イベントを追加します。
    inline void InsertApplicationTerminateEvent(const nn::ncm::ApplicationId& applicationId, const nn::time::PosixTime& timeAt) NN_NOEXCEPT
    {
        NN_DETAIL_PCTL_TRACE("[pctl] InsertApplicationTerminateEvent: id = 0x%016llX\n",
            applicationId.value);
        EventData::Payload payload;
        payload.applicationTerminate.applicationId = applicationId;
        InsertEvent(EventType::DidApplicationTerminate, &payload, timeAt);
    }

    inline void AddApplicationRejectedEvent(const nn::ncm::ApplicationId& applicationId) NN_NOEXCEPT
    {
        EventData::Payload payload;
        payload.limitedApplicationLaunch.applicationId = applicationId;
        AddEvent(EventType::DidLimitedApplicationLaunch, &payload);
    }

    // @brief Idleイベントを追加します。
    inline void AddIdleEvent() NN_NOEXCEPT
    {
        AddEvent(EventType::Idle);
    }

    // @brief (一時)解除イベントを追加します。
    inline void AddUnlockEvent() NN_NOEXCEPT
    {
        AddEvent(EventType::DidUnlock);
    }

    // @brief (一時)解除イベントを指定時刻の情報として追加します。
    inline void InsertUnlockEvent(const nn::time::PosixTime& timeAt) NN_NOEXCEPT
    {
        InsertEvent(EventType::DidUnlock, nullptr, timeAt);
    }

    // @brief (再)ロックイベントを追加します。
    inline void AddLockEvent() NN_NOEXCEPT
    {
        AddEvent(EventType::DidLock);
    }

    // @brief 解除コード間違いイベントを追加します。
    inline void AddWrongPinCodeEvent() NN_NOEXCEPT
    {
        AddEvent(EventType::DidWrongUnlockCode);
    }

    // @brief オンラインに遷移したイベントを追加します。
    inline void AddOnlineEvent() NN_NOEXCEPT
    {
        AddEvent(EventType::DidComeOnline);
    }

    // @brief オフラインに遷移したイベントを追加します。
    inline void AddOfflineEvent() NN_NOEXCEPT
    {
        AddEvent(EventType::DidGoOffline);
    }

    // @brief 新規UGC制限タイトル追加イベントを追加します。
    // @details 蓄積済みイベントをさかのぼって既に同じイベントがあれば追加しません。
    inline void AddNewManagedApplicationEvent(const nn::ncm::ApplicationId& applicationId) NN_NOEXCEPT
    {
        AddManagedEvent(true, applicationId);
    }

    // @brief タイトルのUGC制限削除イベントを追加します。
    // @details 蓄積済みイベントをさかのぼって既に同じイベントがあれば追加しません。
    inline void AddRemoveManagedApplicationEvent(const nn::ncm::ApplicationId& applicationId) NN_NOEXCEPT
    {
        AddManagedEvent(false, applicationId);
    }

    // @brief 強制中断イベントを追加します。
    inline void AddTimerSuspensionEvent() NN_NOEXCEPT
    {
        AddEvent(EventType::DidInterruptPlaying);
    }

    // @brief アプリケーションが実行していたことを表すイベントを追加します。
    inline void AddApplicationPlayedEvent(const nn::ncm::ApplicationId& applicationId) NN_NOEXCEPT
    {
        NN_DETAIL_PCTL_TRACE("[pctl] AddApplicationPlayedEvent: id = 0x%016llX, playingTime = %lld\n",
            applicationId.value);
        EventData::Payload payload;
        payload.playedApplication.applicationId = applicationId;
        AddEvent(EventType::DidApplicationPlay, &payload);
    }

    // @brief アプリケーションが中断したことを表すイベントを追加します。
    inline void InsertApplicationSuspendedEvent(const nn::ncm::ApplicationId& applicationId, const nn::time::PosixTime& timeAt) NN_NOEXCEPT
    {
        NN_DETAIL_PCTL_TRACE("[pctl] InsertApplicationSuspendedEvent: id = 0x%016llX\n",
            applicationId.value);
        EventData::Payload payload;
        payload.suspendedApplication.applicationId = applicationId;
        InsertEvent(EventType::DidApplicationSuspend, &payload, timeAt);
    }

    // @brief アプリケーションが再開したことを表すイベントを追加します。
    inline void InsertApplicationResumedEvent(const nn::ncm::ApplicationId& applicationId, const nn::time::PosixTime& timeAt) NN_NOEXCEPT
    {
        NN_DETAIL_PCTL_TRACE("[pctl] InsertApplicationResumedEvent: id = 0x%016llX\n",
            applicationId.value);
        EventData::Payload payload;
        payload.resumedApplication.applicationId = applicationId;
        InsertEvent(EventType::DidApplicationResume, &payload, timeAt);
    }

    // @brief ユーザーが Open になったことを表すイベントを追加します。
    inline void InsertUserOpenedEvent(const nn::account::Uid& uid, const nn::time::PosixTime& timeAt) NN_NOEXCEPT
    {
        NN_DETAIL_PCTL_TRACE("[pctl] InsertUserOpenedEvent: uid = %08x_%08x_%08x_%08x\n",
            static_cast<uint32_t>(uid._data[0] >> 32),
            static_cast<uint32_t>(uid._data[0] & 0xFFFFFFFFull),
            static_cast<uint32_t>(uid._data[1] >> 32),
            static_cast<uint32_t>(uid._data[1] & 0xFFFFFFFFull));
        EventData::Payload payload;
        payload.openedUser.uid = uid;
        InsertEvent(EventType::DidUserOpen, &payload, timeAt);
    }

    // @brief ユーザーが Close になったことを表すイベントを追加します。
    inline void InsertUserClosedEvent(const nn::account::Uid& uid, const nn::time::PosixTime& timeAt) NN_NOEXCEPT
    {
        NN_DETAIL_PCTL_TRACE("[pctl] InsertUserClosedEvent: uid = %08x_%08x_%08x_%08x\n",
            static_cast<uint32_t>(uid._data[0] >> 32),
            static_cast<uint32_t>(uid._data[0] & 0xFFFFFFFFull),
            static_cast<uint32_t>(uid._data[1] >> 32),
            static_cast<uint32_t>(uid._data[1] & 0xFFFFFFFFull));
        EventData::Payload payload;
        payload.closedUser.uid = uid;
        InsertEvent(EventType::DidUserClose, &payload, timeAt);
    }

    // @brief アラーム一時無効イベントを追加します。
    inline void AddAlarmDisabledEvent() NN_NOEXCEPT
    {
        AddEvent(EventType::DidAlarmMakeInvisible);
    }

    // @brief アラーム一時無効解除イベントを追加します。
    inline void AddAlarmEnabledEvent() NN_NOEXCEPT
    {
        AddEvent(EventType::DidAlarmMakeVisible);
    }

    // @brief ダウンロード開始イベントを追加します。
    inline void AddDownloadStartedEvent(const nn::ncm::ApplicationId& applicationId) NN_NOEXCEPT
    {
        EventData::Payload payload;
        payload.startedApplicationDownload.applicationId = applicationId;
        AddEvent(EventType::DidApplicationDownloadStart, &payload);
    }

    // @brief 時間切れイベントを追加します。
    inline void AddLimitTimeReachedEvent() NN_NOEXCEPT
    {
        AddEvent(EventType::DidReachLimitTime);
    }

    // @brief シャットダウン(電源OFF)イベントを追加します。
    inline void AddShutdownEvent() NN_NOEXCEPT
    {
        AddEvent(EventType::DidShutdown);
    }

    // @brief プレイタイマー開始イベントを追加します。
    inline void AddPlayTimerStartedEvent() NN_NOEXCEPT
    {
        AddEvent(EventType::DidPlayTimerStart);
    }

    // @brief プレイタイマー開始イベントを指定時刻の情報として追加します。
    inline void InsertPlayTimerStartedEvent(const nn::time::PosixTime& timeAt) NN_NOEXCEPT
    {
        InsertEvent(EventType::DidPlayTimerStart, nullptr, timeAt);
    }

    // @brief プレイタイマー停止イベントを追加します。
    inline void AddPlayTimerStoppedEvent() NN_NOEXCEPT
    {
        AddEvent(EventType::DidPlayTimerStop);
    }

    // @brief 連携完了イベントを指定時刻の情報として追加します。
    inline void InsertPairingActivatedEvent(const nn::time::PosixTime& timeAt) NN_NOEXCEPT
    {
        InsertEvent(EventType::DidDeviceActivate, nullptr, timeAt);
    }

    // @brief 地域名が変更したイベントを追加します。
    inline void AddLocationNameChangedEvent() NN_NOEXCEPT
    {
        AddEvent(EventType::DidLocationNameChange);
    }

    // @brief 強制電源断が発生した(と思われるタイミングに)イベントを追加します。
    inline void AddUnexpectedShutdownOccurEvent(const nn::time::PosixTime& timeAt) NN_NOEXCEPT
    {
        AddEvent(EventType::DidUnexpectedShutdownOccur, timeAt, nullptr);
    }

    // @brief 送信用途のために蓄えられたイベントデータのうち古いものから指定した数だけ取得します。
    // @param[out] outDataArray イベントデータを受け取る配列
    // @param[in] maxDataCount 必要とするイベントデータの最大数
    // @return 利用できるデータの数(なければ 0)
    // @details
    // 終了したら FinishPostPartialData を呼び出してください。ただし戻り値が 0 の場合は不要です。
    uint16_t GetPartialDataForPost(EventData* outDataArray, uint16_t maxDataCount) NN_NOEXCEPT;
    // @brief GetPartialDataForPost で送信したデータの完了を通知します。
    // @details 送信した数のデータは無効扱いされ、データ情報の更新処理が行われます。
    void FinishPostPartialData(uint16_t postCount) NN_NOEXCEPT;

    // @brief 最後に記録されたイベントデータを取得します。
    // @return データがあれば true
    // @details 本メソッドは排他制御を行いません。排他制御が不要な
    //          プレイタイマー系初期化時の処理でのみ使用することを想定しています。
    bool GetLastEventData(EventData* outEventData) const NN_NOEXCEPT
    {
        if (!m_Data.header.isAnyRecorded)
        {
            return false;
        }
        NN_STATIC_ASSERT(std::is_pod<std::remove_pointer<decltype(outEventData)>::type>::value);
        *outEventData = m_Data.storage[m_Data.header.lastIndex];
        return true;
    }

    // @brief インスタンスが初期化されたタイミング以降に記録されたイベントを破棄します。
    void DiscardEventsSinceInitialize() NN_NOEXCEPT;

    // @brief 全イベントを破棄します。
    void DiscardAllEvents() NN_NOEXCEPT;

    // @brief ファイルシステムのコミット処理をスキップするかどうかを指定します。
    void SetCommitSkipped(bool isSkipped) NN_NOEXCEPT
    {
        m_IsCommitSkipped = isSkipped;
    }

    // @brief データをファイルに書き込みます。
    void WriteDataToFile() NN_NOEXCEPT;

    // @brief データが記録されている(イベントが蓄積されている)かどうかを返します。
    bool IsAnyDataRecorded() const NN_NOEXCEPT
    {
        return m_Data.header.isAnyRecorded;
    }

    // @brief 現在保持しているイベントの数を返します。(テスト用)
    int GetEventCountForTest() const NN_NOEXCEPT
    {
        if (!m_Data.header.isAnyRecorded)
        {
            return 0;
        }
        int count = static_cast<int>(m_Data.header.lastIndex) - static_cast<int>(m_Data.header.startIndex) + 1;
        if (count <= 0)
        {
            count += MaxEventCount;
        }
        return count;
    }

    // @brief 起動してから今までに記録したイベントの数(単調増加)を返します。(テスト用)
    uint32_t GetTotalEventCountFromBootForTest() const NN_NOEXCEPT
    {
        return m_TotalEventCountFromBoot;
    }

    // @brief 起動してから今までに記録したイベントの数を上書きして設定します。(テスト用)
    void SetTotalEventCountFromBootForTest(uint32_t totalEventCountFromBoot) NN_NOEXCEPT
    {
        m_TotalEventCountFromBoot = totalEventCountFromBoot;
    }

    // @brief 蓄積済みのイベントを PlayState に反映させます。
    // @details WatcherEventStorage の内部データへの変更は行いません。
    void ApplyStoredEventsToPlayState(PlayState* pPlayState) NN_NOEXCEPT;

private:
    // @brief 新規UGC制限タイトル追加イベントまたはタイトルのUGC制限削除イベントを追加します。
    // @details 蓄積済みイベントをさかのぼって既に同じイベントがあれば追加しません。
    void AddManagedEvent(bool isAdd, const nn::ncm::ApplicationId& applicationId) NN_NOEXCEPT;

    // @brief イベントを追加します。m_MutexData のロック済みを前提に処理を行います。
    void AddEventInternalNoLock(EventType eventType, const nn::time::PosixTime& timestamp, const nn::time::LocationName& locationName, const EventData::Payload* pPayload) NN_NOEXCEPT;

    // @brief イベントが時刻順になるように挿入追加します。m_MutexData のロック済みを前提に処理を行います。
    // @details 挿入位置は timestamp の値(timestamp を超える時刻を持つイベントの手前、または末尾)で決定されます。
    void InsertEventInternalNoLock(EventType eventType, const nn::time::PosixTime& timestamp, const nn::time::LocationName& locationName, const EventData::Payload* pPayload) NN_NOEXCEPT;

    void InsertEventInternalNoLock(uint16_t insertPosition, EventType eventType, const nn::time::PosixTime& timestamp, const nn::time::LocationName& locationName, const EventData::Payload* pPayload) NN_NOEXCEPT;

    static void StoreEventData(EventData* pOutData, EventType eventType, const nn::time::PosixTime& timestamp, const nn::time::LocationName& locationName,
        const nn::time::CalendarAdditionalInfo& calInfo, const EventData::Payload* pPayload) NN_NOEXCEPT;

    // @brief ファイルからデータを読み込みます。
    void LoadDataFromFile() NN_NOEXCEPT;

    // @brief データをファイルに書き込む要求を発行します。
    void PostWriteDataToFileNoLock(bool isHeaderOnly, uint16_t writeStartIndex, uint16_t writeLastIndex) NN_NOEXCEPT;

    // @brief データをファイルに書き込みます。ヘッダー部分は必ず書き込み、
    //     それ以外の部分は writeStartIndex と writeLastIndex に対応する箇所を書き込みます。
    //     (isHeaderOnly が true であるか、ヘッダーの isAnyRecorded が false であれば
    //      writeStartIndex と writeLastIndex は無視されヘッダー以外の書き込みを行いません。)
    void WriteDataToFileNoLock(bool isHeaderOnly, uint16_t writeStartIndex, uint16_t writeLastIndex) NN_NOEXCEPT;

    nn::os::SdkMutex m_MutexData;
    // 一時変数
    nn::time::TimeZoneRule m_TimeZoneRule;
    // fsのCommitをスキップするかどうか(テスト用)
    bool m_IsCommitSkipped;
    // GetPartialDataForPost が呼び出されてまだ FinishPostPartialData が呼び出されていないかどうか
    bool m_IsProcessingPostData;
    // プロセスを起動してから蓄えたイベントの数(単調増加でオーバーフローを許容)
    uint32_t m_TotalEventCountFromBoot;
    // GetPartialDataForPost 呼び出し時点の m_TotalEventCountFromBoot - (GetPartialDataForPost の戻り値) の値
    uint32_t m_EventCountOnPost;
    // WriteDataToFile で書き込みを行う際に用いる書き込み範囲の情報
    struct WriteDataInfo
    {
        uint16_t writeStartIndex;
        uint16_t writeLastIndex;
        bool isWritingEventsNecessary; // ヘッダーだけでなくイベント本体も書き込むかどうか
    } m_WriteDataInfo;
    // (最大数を決めているので固定長のバッファーで持つ)
    EventLogFileData m_Data;
    // 事前に見積もった 384KiB 以下に収まっていることの確認
    NN_STATIC_ASSERT(sizeof(EventLogFileData) <= 384 * 1024);
};

}}}}}
