﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_SdkThread.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/pctl/detail/service/watcher/pctl_PlayTimer.h>
#include <nn/pctl/detail/service/watcher/pctl_WatcherEventStorage.h>
#include <nn/time/time_CalendarAdditionalInfo.h>
#include <nn/time/time_CalendarTime.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_LockGuard.h>
#include <atomic>

namespace nn { namespace pctl { namespace detail { namespace service { namespace watcher {

// @brief 電源遷移状態を表す定数
enum PowerTransition
{
    PowerTransition_None = 0,                //!< なし、あるいは下記以外(pctl では利用しないイベントの場合など)
    PowerTransition_FullAwakeToMinimumAwake, //!< FullAwake から MinimumAwake へ遷移(FullAwake 以外からの遷移は含まず)
    PowerTransition_ToFullAwake,             //!< FullAwake へ遷移
    PowerTransition_ToShutdown               //!< ShutdownReady へ遷移
};

class WatcherEventManager
{
public:
    static const size_t ThreadStackSize = 16 * 1024;

    // @brief 何秒おきにチェックを行うかを示す定数です。(秒単位)
    static const int16_t TimeSpanForCheck = 10;
    // @brief 時間切れ状態から強制中断するまでの猶予時間を示す定数です。(秒単位)
    static const int16_t TimeSpanForSuspension = 10;
    // @brief 残ったイベントを何秒後に送信するか(Oneshot)を示す定数です。(秒単位)
    static const int16_t TimeSpanForPostRemainingEvents = 30;

    // @brief 設定を指定した構造体に読み込みます。
    // @return データを読み込めた場合は true
    // @details スレッドでは処理を行わず、呼び出し元スレッドでそのまま処理を行います。
    static bool LoadSettings(PlayTimerStatus* outStatus) NN_NOEXCEPT;

    // @brief 構造体のデータを設定として書き込みます。
    // @details スレッドでは処理を行わず、呼び出し元スレッドでそのまま処理を行います。
    static void SaveSettings(const PlayTimerStatus& status) NN_NOEXCEPT;

    // @brief プレイタイマーの設定をセーブデータから削除します。
    static void ClearPlayTimerSettingsFromSaveData() NN_NOEXCEPT;

    WatcherEventManager(void* threadStackBuffer, size_t stackBufferSize) NN_NOEXCEPT;
    ~WatcherEventManager() NN_NOEXCEPT;

    // @brief Watcher系処理の初期化を行います。
    void Initialize() NN_NOEXCEPT;

    // @brief スレッドを開始します。
    // @details 終了は StopThread またはデストラクターで行います。
    void StartThread() NN_NOEXCEPT;

    // @brief スレッドを終了します。
    // @details 終了時にタイマー停止処理も行います。
    //   また、StopThread を呼び出した後は StartThread できません。
    void StopThread() NN_NOEXCEPT;

    // @brief プレイ状態の保存を行う非同期処理のトリガーを発行します。
    void PostSavePlayState() NN_NOEXCEPT;

    // @brief デバイスユーザー情報の保存を行う非同期処理のトリガーを発行します。
    void PostSaveDeviceUserData() NN_NOEXCEPT;

    // @brief デバイスユーザー情報の削除を行います。(同期処理)
    void ClearDeviceUserData() NN_NOEXCEPT;

    // @brief 残り時間などの状態をリセットします。
    // @details 経過時間(遊んだ時間)などが初期化されます。設定値は初期化されません。
    void ResetTimeStatus() NN_NOEXCEPT;

    // @brief イベント待ちを解いて残り時間を再計算します。
    void RefreshTimeStatus() NN_NOEXCEPT;

    // @brief プレイタイマー設定を更新し、残り時間を再計算します。(RefreshTimeStatus の処理を含みます。)
    // @return 変更された設定があれば true
    bool UpdatePlayTimerSettings(const PlayTimerSettings& settings) NN_NOEXCEPT;

    // @brief プレイタイマー設定を消去します。
    void ClearPlayTimerSettings() NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(m_MutexMessage);
        ClearPlayTimerSettingsNoLock();
    }

    // @brief 一日の経過時間をリセットします。
    void ClearSpentTime() NN_NOEXCEPT;

    // @brief プレイタイマーの開始・停止状態(有効・無効状態)を切り替えます。
    // @details 設定の有効・無効とは異なります。また、初期状態は無効状態です。
    void SetPlayTimerEnabled(bool isEnabled) NN_NOEXCEPT;

    // @brief プレイタイマーの開始・停止状態(有効・無効状態)を返します。
    // @details 一時解除などによってタイマーがロックされている場合は false が返ります。
    bool IsPlayTimerEnabled() const NN_NOEXCEPT;

    // @brief アラーム無効状態のリセットを行うタイミングを取得します。
    static nn::Result GetTimeToResetAlarmDisable(nn::time::PosixTime* outTime, const nn::time::PosixTime& now) NN_NOEXCEPT;

    // @brief アラーム無効状態(残り時間通知や強制中断が発生しない)かどうかを取得します。
    bool IsAlarmDisabled() const NN_NOEXCEPT;

    // @brief アラーム無効状態(残り時間通知や強制中断が発生しない)を設定します。
    // @return 実際に設定された無効状態(isDisabled と異なる場合があります)
    // @details 日付を跨いだタイミングで状態が元に戻ります。
    bool SetAlarmDisabled(bool isDisabled, const nn::time::PosixTime& now) NN_NOEXCEPT;

    // @brief 設定変更通知用のシステムイベントを返します。
    nn::os::SystemEvent* GetSynchronizationEvent() NN_NOEXCEPT
    {
        return &m_SysEventSynchronization;
    }

    // @brief 解除コード変更通知用のシステムイベントを返します。
    nn::os::SystemEvent* GetPinCodeChangedEvent() NN_NOEXCEPT
    {
        return &m_SysEventPinCodeChanged;
    }

    // @brief 強制中断通知用のシステムイベントを返します。
    nn::os::SystemEvent* GetPlayTimerEventToRequestSuspension() NN_NOEXCEPT
    {
        return &m_SysEventSuspend;
    }

    // @brief 現在の残り時間を返します。
    inline nn::TimeSpan GetRemainingTime() const NN_NOEXCEPT
    {
        return GetRemainingTime(nullptr);
    }

    // @brief 現在の残り時間を返します。
    nn::TimeSpan GetRemainingTime(PlayTimerMode* outTimerMode) const NN_NOEXCEPT;

    // @brief 現在の状態を返します。
    void GetPlayTimerStatus(PlayTimerStatus* outStatus) const NN_NOEXCEPT;

    // @brief 現在の設定値を返します。
    void GetPlayTimerSettings(PlayTimerSettings* outSettings) const NN_NOEXCEPT;

    // @brief 強制中断が必要かどうかを返します。
    bool IsRestrictedByPlayTimer() NN_NOEXCEPT;

    // @brief タイマーをロックするかどうかを設定します。
    // @details ロックした場合はタイマーが停止している状態と同等に扱われますが、
    //     内部的な開始・停止状態は維持され、ロックを解除したタイミングで復元されます。
    //     また、この値は永続化されず、初期値は false になります。
    void UpdateUnlockRestrictionStatus() NN_NOEXCEPT;

    // @brief イベント処理を停止し、無効化します。
    // @details 溜められたイベントも無効化されます。また、再起動するまで有効化できません。
    //   (再起動後既に無効化されていた場合はイベント処理も無効化のままになります。)
    void DisableEventProcess() NN_NOEXCEPT;

    // @brief 現在の日の消費時間(タイマー有効時の連続稼働時間)を返します。
    nn::TimeSpan GetSpentTime() const NN_NOEXCEPT;

    // @brief 別スレッドのイベント処理を待ちます。(テスト用)
    void WaitForEventsForTest() NN_NOEXCEPT;

    // @brief プロセスが保持するタイムゾーン地域名を取得します。
    // @details システム設定とずれがある場合を許容する場合(イベントログ生成など)に使用します。
    nn::time::LocationName& GetTimeLocationName() NN_NOEXCEPT
    {
        return m_PlayTimerStatus.lastTimeLocationName;
    }

    // @brief 電源管理に関するイベントのハンドル処理を行います。
    // @details 呼び出し前に m_MutexMessage をロックしません。
    void HandlePowerStateEvent(PowerTransition transition) NN_NOEXCEPT;

    // @brief イベント書き込みを行う通知を発行します。
    void PostSerializeEvents() NN_NOEXCEPT;

    // @brief バックグラウンドでの設定取得を要求します。
    //     tryCount によって即時開始するか(リトライ用に)遅延開始するか分岐します。
    void RequestRetrieveSettingsBackground(int tryCount, bool fromNpns) NN_NOEXCEPT;

    // @brief 残ったイベントを遅延送信するようにイベントを発行します。
    void RequestPostRemainingEvents() NN_NOEXCEPT;

    // @brief バックグラウンドでの設定取得待ち状態をクリアします。
    void ClearWaitingForRetrieveSettingsBackground() NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(m_MutexMessage);
        ClearWaitingForRetrieveSettingsBackgroundNoLock();
    }

    // @brief サーバー連携が完了(確定)した際に必要となる処理を行います。
    void ProcessPairingActivated(nn::time::PosixTime timeCurrent) NN_NOEXCEPT;

    // @brief 連携解除が行われた際に必要となるイベント待ち受け関連の処理を実行します。
    void ProcessPairingDeleted() NN_NOEXCEPT;

    // @brief 間欠起動のタスク開始を要求します。nn::ResultSuccess のときに実行が開始されます。
    nn::Result RequestStartIntermittentTask() NN_NOEXCEPT;

    // @brief 間欠起動のタスクをキャンセルします。
    void CancelIntermittentTask() NN_NOEXCEPT;

    // @brief 各種データの書き込みが必要な状態である場合、それを同期的に行います。
    void ProcessSerializeData() NN_NOEXCEPT;

    // @brief イベント書き込みが必要な状態になっていればそれを即時実行します。(テスト用)
    void ProcessSerializeEventsIfNeededForTest() NN_NOEXCEPT;

    // @brief プレイ時間のデータ(状態)を反映させます。
    void ApplyPlayTimerStatus(bool isIgnorableTime,
        const nn::time::PosixTime& lastTimestampForElapsedTime,
        int64_t lastElapsedTimeSeconds,
        const nn::time::PosixTime& lastTimerStoppedTime) NN_NOEXCEPT;

    // @brief イベント送信前の処理を行います。
    void PreparePostEvent() NN_NOEXCEPT;

private:
    // @brief 設定を読み込みます。
    // @details スレッドでは処理を行わず、呼び出し元スレッドでそのまま処理を行います。
    void LoadSettings() NN_NOEXCEPT;

    // @brief グローバルに定義されたプレイ状態を初期化します。
    // @param[out] outPlayState 初期化したデータと同じデータを受け取る構造体へのポインター
    // @details
    // * ファイルに保存されている場合、その内容の読み込みも行います。
    // * スレッドでは処理を行わず、呼び出し元スレッドでそのまま処理を行います。
    void InitializePlayState(PlayState* outPlayState) NN_NOEXCEPT;

    // @brief デバイスユーザー情報データを初期化します。
    // @details
    // * ファイルに保存されている場合、その内容の読み込みも行います。
    // * スレッドでは処理を行わず、呼び出し元スレッドでそのまま処理を行います。
    void InitializeDeviceUserData() NN_NOEXCEPT;

    // @brief イベント送信に対するタイマーの初期化を行います。
    void InitializeTimerForPostEvents() NN_NOEXCEPT;

    // @brief デバイス起動時の処理を行います。
    void OnDeviceLaunch(PlayState* pTempPlayState) NN_NOEXCEPT;

    // @brief タイマーイベント更新のパターンを表す定数です。
    enum class RefreshTimerPattern : uint8_t
    {
        Normal,           //!< 通常更新
        SettingChanged,   //!< 設定変更による更新
        DayChanged        //!< 日付変更による更新
    };

    // @brief bedtime の確認に利用する設定値を取得します。
    // @details 曜日別設定の取得になる場合、特定の時間帯は前日の設定を取得する場合があります。
    const PlayTimerDaySettings* GetDaySettingsForBedtime(const nn::time::CalendarTime& calTimeNow, Week dayOfWeek) const NN_NOEXCEPT;
    // @brief limitTime の確認に利用する設定値を取得します。
    const PlayTimerDaySettings* GetDaySettingsForLimitTime(Week dayOfWeek) const NN_NOEXCEPT;

    // @brief 現在の残り時間を返します。
    // @return 設定に基づいた残り時間が計算できたら true
    // @details 戻り値にかかわらず *outRemainingTime と *outTimerMode は更新されます。
    bool GetRemainingTimeNoLock(nn::TimeSpan* outRemainingTime, PlayTimerMode* outTimerMode, nn::time::PosixTime now) const NN_NOEXCEPT;

    // @brief プレイタイマー設定を消去します。
    void ClearPlayTimerSettingsNoLock() NN_NOEXCEPT;

    // @brief バックグラウンドでの設定取得待ち状態をクリアします。
    void ClearWaitingForRetrieveSettingsBackgroundNoLock() NN_NOEXCEPT
    {
        m_TimerEventRetrieveSettings.m_Event.Stop();
        m_TimerEventRetrieveSettings.m_Event.Clear();
    }

    // @brief 次の時刻チェックのタイミングを計算してタイマーイベントを更新します。
    void RefreshTimerEvents(RefreshTimerPattern refreshTimerPattern) NN_NOEXCEPT;

    // @brief 日付変更と消費時間を計算します。
    void CalculateSpentTime(nn::TimeSpan* outSpentTime, const nn::time::CalendarTime& calTime, nn::time::PosixTime now) const NN_NOEXCEPT;

    // @brief 制限時刻に基づいてタイマーイベントを設定します。
    void SetNextTimerEventForLimit(nn::time::PosixTime now, const nn::time::CalendarTime& calTime, Week dayOfWeek, RefreshTimerPattern refreshTimerPattern) NN_NOEXCEPT;

    // @brief 次のオーバーレイ通知を行うためのタイマーイベントを設定します。
    void SetNextNotifyTimeEvent(nn::TimeSpan timeSpan, const nn::util::optional<int16_t>& lastNotifiedTime) NN_NOEXCEPT;

    // @brief 直近の制限時刻を返します。
    // @return 制限時刻があれば true
    bool GetNextLimitTime(nn::time::PosixTime* outLimitTime, nn::time::PosixTime now, const nn::time::CalendarTime& calTime, Week dayOfWeek, nn::TimeSpan spentTime) const NN_NOEXCEPT;

    // @brief 有効・無効の状態が変わったときの処理を行います。
    void OnTimerRunningStatusChanged(bool isEnabled) NN_NOEXCEPT;

    // @brief 消費時間を更新します。
    void UpdateSpendTime(nn::time::PosixTime now) NN_NOEXCEPT;

    // @brief 消費時間を更新します。
    void UpdateSpendTime(nn::time::PosixTime now, nn::time::PosixTime startTime, nn::time::PosixTime stopTime, const nn::time::CalendarTime& calTimeChecked) NN_NOEXCEPT;

    // @brief タイムゾーン地域名設定が変わったかどうかをチェックし、変わっていればイベントをトリガーします。
    // @return 変わっていれば true
    bool CheckLocationNameChanged() NN_NOEXCEPT;

    // @brief デバイス設定が変わったかどうかをチェックし、変わっていれば通信をトリガーします。
    // @details 内部で CheckLocationNameChanged も呼び出します。
    void CheckDeviceSettingsChanged() NN_NOEXCEPT;

    // @brief オンライン状態(インターネット接続状態)が変わったかどうかをチェックします。
    void CheckOnlineStatus() NN_NOEXCEPT;

    // @brief 日付が変わったかどうかをチェックします。
    void CheckDayChangedNoLock() NN_NOEXCEPT;

    // @brief 日付が変わったかどうかをチェックします。
    void CheckDayChangedNoLock(nn::time::PosixTime timeNow, const nn::time::CalendarTime& calTimeNow, Week dayOfWeek) NN_NOEXCEPT;

    // @brief 日付が変わった際に必要な、通知イベントなどの更新処理を行います。
    void RefreshEventsForDayChangedNoLock(nn::time::PosixTime timeNow, const nn::time::CalendarTime& calTimeNow, Week dayOfWeek) NN_NOEXCEPT;

    // @brief 曜日が変わって有効・無効状態に変化があるかどうかをチェックします。
    void CheckDayEnabled(bool notifyIfChanged) NN_NOEXCEPT;

    // @brief 曜日が変わって有効・無効状態に変化があるかどうかをチェックします。
    void CheckDayEnabled(nn::time::PosixTime timeNow, const nn::time::CalendarTime& calTimeNow, Week dayOfWeek, bool notifyIfChanged) NN_NOEXCEPT;

    // @brief RefreshTimer の処理を行うためにイベントをトリガーします。
    void SignalRefreshTimerEventNoLock(RefreshTimerPattern pattern) NN_NOEXCEPT
    {
        m_RefreshTimerPattern = pattern;
        m_EventHolderRefreshTimer.m_Event.Signal();
    }

    void OnExit() NN_NOEXCEPT;

    void OnRefreshTimer() NN_NOEXCEPT;

    void OnUpdatePlayTimerSettings() NN_NOEXCEPT;

    void OnNextCheck() NN_NOEXCEPT;

    void OnPostEvents() NN_NOEXCEPT;

    void OnPostRemainingEvents() NN_NOEXCEPT;

    void OnNotifyToOverlay() NN_NOEXCEPT;

    void OnNotifyExceededToOverlay() NN_NOEXCEPT;

    // @brief 強制中断直前の通知処理を行います。
    void OnPreSuspend() NN_NOEXCEPT;

    // @brief 強制中断の処理を行います。
    void OnSuspend() NN_NOEXCEPT;

    // @brief 日付変更時の処理を行います。
    void OnDayChanged() NN_NOEXCEPT;

    // @brief 日付変更時の処理を行います。
    void OnBedtimeDayChanged() NN_NOEXCEPT;

    // @brief タイムゾーンが変わったときの処理を行います。
    void OnLocationNameChanged() NN_NOEXCEPT;

    void OnSaveDeviceEvents() NN_NOEXCEPT;

    void OnSaveTimerStatus() NN_NOEXCEPT;

    void OnPowerAwake() NN_NOEXCEPT;

    void OnRetrieveSettingsEvents(bool fromNpns) NN_NOEXCEPT;

    void OnStartIntermittentTask() NN_NOEXCEPT;

    void OnSavePlayState() NN_NOEXCEPT;

    void OnSaveDeviceUserData() NN_NOEXCEPT;

    void SaveSettingsNoLock() NN_NOEXCEPT;
    void SavePlayState() NN_NOEXCEPT;
    void SaveDeviceUserData() NN_NOEXCEPT;

    // 各種イベント発行のラッパー関数
    // NOTE: m_MutexMessage はロックしたまま呼び出してはいけない
    void AddDeviceLaunchEvent(const nn::time::PosixTime& timestamp) NN_NOEXCEPT;
    void AddTimerSuspensionEvent() NN_NOEXCEPT;
    void AddSleepEvent() NN_NOEXCEPT;
    void AddWakeupEvent() NN_NOEXCEPT;
    void AddOnlineEvent() NN_NOEXCEPT;
    void AddOfflineEvent() NN_NOEXCEPT;
    void AddIdleEvent() NN_NOEXCEPT;
    void AddApplicationPlayedEvent() NN_NOEXCEPT;
    void AddAlarmDisabledEvent() NN_NOEXCEPT;
    void AddAlarmEnabledEvent() NN_NOEXCEPT;
    void AddLimitTimeReachedEvent() NN_NOEXCEPT;
    void AddPlayTimerStartedEvent() NN_NOEXCEPT;
    void AddPlayTimerStoppedEvent() NN_NOEXCEPT;
    void AddLocationNameChangedEvent() NN_NOEXCEPT;
    void AddUnexpectedShutdownOccurEvent(const nn::time::PosixTime& timeAt) NN_NOEXCEPT;

    void ThreadProc() NN_NOEXCEPT;
    static void StaticThreadProc(void* pThis) NN_NOEXCEPT
    {
        reinterpret_cast<WatcherEventManager*>(pThis)->ThreadProc();
    }

    template <typename TEvent>
    class EventHolderT
    {
    public:
        EventHolderT() NN_NOEXCEPT :
            m_Event(nn::os::EventClearMode::EventClearMode_ManualClear)
        {
            nn::os::InitializeMultiWaitHolder(&m_WaitHolder, m_Event.GetBase());
        }
        ~EventHolderT() NN_NOEXCEPT
        {
            nn::os::FinalizeMultiWaitHolder(&m_WaitHolder);
        }

        TEvent m_Event;
        nn::os::MultiWaitHolderType m_WaitHolder;
    };
    typedef EventHolderT<nn::os::Event> EventHolder;
    typedef EventHolderT<nn::os::TimerEvent> TimerEventHolder;

    nn::os::ThreadType m_TaskThread;
    nn::os::MultiWaitType m_MultiWait;
    // スレッド終了イベント
    EventHolder m_EventHolderExit;
    // タイマー更新イベント
    EventHolder m_EventHolderRefreshTimer;
    // タイマー設定変更・保存＆更新イベント
    EventHolder m_EventHolderUpdateTimerSettings;
    // 定期チェックイベント
    TimerEventHolder m_TimerEventNextCheck;
    // 定期イベント送信イベント
    TimerEventHolder m_TimerEventPostEvents;
    // 残イベント送信イベント(Postしきれなかった分を遅延送信)
    TimerEventHolder m_TimerEventPostRemainingEvents;
    // 通知イベント
    TimerEventHolder m_TimerEventNotify;
    // 超過時間通知イベント
    EventHolder m_EventHolderNotifyExceeded;
    // 強制中断イベント
    TimerEventHolder m_TimerEventPreSuspend;
    // 強制中断イベント
    TimerEventHolder m_TimerEventSuspend;
    // 日付変更イベント
    TimerEventHolder m_TimerEventDayChanged;
    // Bedtime用日付変更イベント
    TimerEventHolder m_TimerEventBedtimeDayChanged;
    // タイムゾーン地域名変更イベント
    EventHolder m_EventLocationNameChanged;
    // プレイイベント保存イベント
    EventHolder m_EventHolderSaveDeviceEvents;
    // プレイタイマー状態保存イベント
    EventHolder m_EventHolderSaveTimerStatus;
    // スリープ解除イベント
    EventHolder m_EventHolderPowerAwake;
    // 設定取得イベント
    TimerEventHolder m_TimerEventRetrieveSettings;
    // 設定取得イベント(NPNSトリガー専用)
    EventHolder m_EventRetrieveSettingsFromNpns;
    // 間欠起動タスク開始イベント
    EventHolder m_EventHolderStartIntermittentTask;
    // プレイ状態保存イベント
    EventHolder m_EventHolderSavePlayState;
    // デバイスユーザー情報保存イベント
    EventHolder m_EventHolderSaveDeviceUserData;
    nn::os::SystemEvent m_SysEventSynchronization;
    nn::os::SystemEvent m_SysEventPinCodeChanged;
    nn::os::SystemEvent m_SysEventSuspend;
    // m_Message や m_GeneralSettingsData, m_GeneralSettingsMask の読み書き時にロックを行うための Mutex
    mutable nn::os::SdkMutex m_MutexMessage;
    PlayTimerStatusVer3 m_PlayTimerStatus;
    nn::time::PosixTime m_LastPlayTimerStatusUpdated;
    nn::util::optional<int16_t> m_TimeSecondsValueForLastNotification;
    int16_t m_TimeSecondsValueForNextNotification;
    int64_t m_TimeSecondsValueForExceededNotification;
    // m_TimerEventRetrieveSettings の試行回数
    int m_TryCountForRetrieveSettings;
    // 直前の定期チェックにおいてネットワーク時計が使用できたかどうか
    bool m_IsLastNetworkTimeAvailable;
    // 現在の日において時間切れを通知したかどうか(設定変更で残り時間が復活した場合はリセット)
    bool m_IsTimeExceedNotified;
    // 強制中断イベントを発行するためのイベントをトリガーしたかどうか
    bool m_IsSuspendEventTriggered;
    // 強制中断した(強制中断通知を発行した)かどうか
    bool m_IsSuspended;
    // m_EventHolderRefreshTimer のイベント処理におけるパターン
    // (m_RefreshTimerPattern == RefreshTimerPattern::Normal の場合 m_TimerEventNextCheck と m_TimerEventPreSuspend はそのまま)
    RefreshTimerPattern m_RefreshTimerPattern;
    // タイマーが一度でも稼働したことがあるかどうか
    // (セーブデータから読み込まれた場合も含む)
    bool m_IsTimerOnceRun;
    // タイマー処理が SetPlayTimerEnabled によって開始されたかどうか
    // (ペアコンの一時解除などの影響は除く)
    bool m_IsTimerStartedManually;
    // スレッドが動作中かどうか
    bool m_IsThreadRunning;
    // オンライン状態かどうか(イベント発行にのみ利用)
    bool m_IsOnline;
    // スリープ状態かどうか
    std::atomic<bool> m_IsSleeping;
};

}}}}}
