﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Common.h>
#include <nn/account/account_Config.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/pctl/pctl_TypesSystem.h>
#include <nn/pctl/pctl_TypesWatcher.h>
#include <nn/pctl/detail/service/pctl_ServiceConfig.h>
#include <nn/pctl/detail/service/watcher/pctl_EventLog.h>
#include <nn/settings/system/settings_Region.h>

#include <cstring>

namespace nn { namespace pctl { namespace detail { namespace service { namespace watcher {

static const size_t MaxEtagLength = 96; // 最低でも66文字(64 + strlen("\"\""))
static const size_t MaxMiiUriLength = 256;

typedef uint64_t ServerDeviceId;
typedef uint64_t NintendoAccountId;

// 文字列リテラルから文字数を計算するために用いるクラス
// MaxLengthCheck > 0 の場合はこの長さ以下になるかどうかのチェックを static_assert で行う
template <typename T, size_t MaxLengthCheck = 0> class StringLengthCalculator
{
public:
    //static const size_t Length = 0;
};
template <size_t N, size_t MaxLengthCheck> class StringLengthCalculator<const char(&)[N], MaxLengthCheck>
{
public:
    static const size_t Length = N - 1;
    NN_STATIC_ASSERT(Length <= MaxLengthCheck);
};
template <size_t N> class StringLengthCalculator<const char(&)[N], 0>
{
public:
    static const size_t Length = N - 1;
};

struct StringValueDefinition
{
    const char* value;
    size_t length;
};
#define DEFINE_STRING_VALUE(text) \
    { text, StringLengthCalculator<decltype(text), 0>::Length }
// 定義と同時に text の長さチェックを行う
#define DEFINE_STRING_VALUE_WITH_LENGTH_CHECK(text, maxLength) \
    { text, StringLengthCalculator<decltype(text), (maxLength)>::Length }
#define DEFINE_EMPTY_STRING_VALUE() \
    { nullptr, 0 }

static const StringValueDefinition SafetyLevelValues[SafetyLevel::SafetyLevel_Max] = {
    DEFINE_STRING_VALUE("NONE"),
    DEFINE_STRING_VALUE("CUSTOM"),
    DEFINE_STRING_VALUE("CHILDREN"),
    DEFINE_STRING_VALUE("YOUNG_TEENS"),
    DEFINE_STRING_VALUE("OLDER_TEENS")
};

static const StringValueDefinition RestrictionValues[] = {
    DEFINE_STRING_VALUE("UNRESTRICTED"),
    DEFINE_STRING_VALUE("RESTRICTED")
};

static const StringValueDefinition SafeLaunchValues[] = {
    DEFINE_STRING_VALUE("NONE"),
    DEFINE_STRING_VALUE("ALLOW")
};

static const StringValueDefinition RatingOrganizationValues[] = {
    DEFINE_STRING_VALUE("CERO"),          // RatingOrganization::CERO
    DEFINE_STRING_VALUE("GRAC_GCRB"),     // RatingOrganization::GRACGCRB
    DEFINE_STRING_VALUE("GSRMR"),         // RatingOrganization::GSRMR
    DEFINE_STRING_VALUE("ESRB"),          // RatingOrganization::ESRB
    DEFINE_STRING_VALUE("CLASS_IND"),     // RatingOrganization::ClassInd
    DEFINE_STRING_VALUE("USK"),           // RatingOrganization::USK
    DEFINE_STRING_VALUE("PEGI"),          // RatingOrganization::PEGI
    DEFINE_STRING_VALUE("PEGI_PORTUGAL"), // RatingOrganization::PEGIPortugal
    DEFINE_STRING_VALUE("PEGI_BBFC"),     // RatingOrganization::PEGIBBFC
    DEFINE_STRING_VALUE("RUSSIAN"),       // RatingOrganization::Russian
    DEFINE_STRING_VALUE("ACB"),           // RatingOrganization::ACB
    DEFINE_STRING_VALUE("OFLC")           // RatingOrganization::OFLC
};
NN_STATIC_ASSERT(static_cast<int>(nn::ns::RatingOrganization::OFLC) < std::extent<decltype(RatingOrganizationValues)>::value);

static const StringValueDefinition RegionValues[] = {
    DEFINE_STRING_VALUE("JAPAN"),         // RegionCode::RegionCode_Japan
    DEFINE_STRING_VALUE("USA"),           // RegionCode::RegionCode_Usa
    DEFINE_STRING_VALUE("EUROPE"),        // RegionCode::RegionCode_Europe
    DEFINE_STRING_VALUE("AUSTRALIA"),     // RegionCode::RegionCode_Australia
    DEFINE_STRING_VALUE("CHINA"),         // RegionCode::RegionCode_China
    DEFINE_STRING_VALUE("KOREA"),         // RegionCode::RegionCode_Korea
    DEFINE_STRING_VALUE("TAIWAN"),        // RegionCode::RegionCode_Taiwan
    DEFINE_STRING_VALUE("UNDEFINED")
};
NN_STATIC_ASSERT(static_cast<int>(nn::settings::system::RegionCode::RegionCode_Taiwan) < std::extent<decltype(RegionValues)>::value);

// ※ 7文字以下で定義する(セーブデータ上もこの文字列を使うため)
static const StringValueDefinition LanguageValues[] = {
    DEFINE_STRING_VALUE_WITH_LENGTH_CHECK("ja-JP", 7),       // Language::Language_Japanese
    DEFINE_STRING_VALUE_WITH_LENGTH_CHECK("en-US", 7),       // Language::Language_AmericanEnglish
    DEFINE_STRING_VALUE_WITH_LENGTH_CHECK("fr-FR", 7),       // Language::Language_French
    DEFINE_STRING_VALUE_WITH_LENGTH_CHECK("de-DE", 7),       // Language::Language_German
    DEFINE_STRING_VALUE_WITH_LENGTH_CHECK("it-IT", 7),       // Language::Language_Italian
    DEFINE_STRING_VALUE_WITH_LENGTH_CHECK("es-ES", 7),       // Language::Language_Spanish
    DEFINE_STRING_VALUE_WITH_LENGTH_CHECK("zh-Hans", 7),     // Language::Language_Chinese (zh-CN --> zh-Hans として扱う)
    DEFINE_STRING_VALUE_WITH_LENGTH_CHECK("ko-KR", 7),       // Language::Language_Korean
    DEFINE_STRING_VALUE_WITH_LENGTH_CHECK("nl-NL", 7),       // Language::Language_Dutch
    DEFINE_STRING_VALUE_WITH_LENGTH_CHECK("pt-PT", 7),       // Language::Language_Portuguese
    DEFINE_STRING_VALUE_WITH_LENGTH_CHECK("ru-RU", 7),       // Language::Language_Russian
    DEFINE_STRING_VALUE_WITH_LENGTH_CHECK("zh-Hant", 7),     // Language::Language_Taiwanese (zh-TW --> zh-Hant として扱う)
    DEFINE_STRING_VALUE_WITH_LENGTH_CHECK("en-GB", 7),       // Language::Language_BritishEnglish
    DEFINE_STRING_VALUE_WITH_LENGTH_CHECK("fr-CA", 7),       // Language::Language_CanadianFrench
    DEFINE_STRING_VALUE_WITH_LENGTH_CHECK("es-MX", 7),       // Language::Language_LatinAmericanSpanish
    DEFINE_STRING_VALUE_WITH_LENGTH_CHECK("zh-Hans", 7),     // Language::Language_SimplifiedChinese
    DEFINE_STRING_VALUE_WITH_LENGTH_CHECK("zh-Hant", 7)      // Language::Language_TraditionalChinese
}; // unused: "pt-BR"
NN_STATIC_ASSERT(static_cast<int>(nn::settings::Language::Language_TraditionalChinese) < std::extent<decltype(LanguageValues)>::value);

static const StringValueDefinition PlayTimerDayWeekMode[] = {
    DEFINE_STRING_VALUE("DAILY"),                // isWeekSettingsUsed == false
    DEFINE_STRING_VALUE("EACH_DAY_OF_THE_WEEK")  // isWeekSettingsUsed == true
};

static const StringValueDefinition PlayTimerRestrictionMode[] = {
    DEFINE_STRING_VALUE("ALARM"),                // PlayTimerMode::PlayTimerMode_Alarm
    DEFINE_STRING_VALUE("FORCED_TERMINATION")    // PlayTimerMode::PlayTimerMode_Suspend
};
NN_STATIC_ASSERT(static_cast<int>(PlayTimerMode::PlayTimerMode_Suspend) < std::extent<decltype(PlayTimerRestrictionMode)>::value);

// device
struct ServerDeviceSimple
{
    ServerDeviceId id;
};

// user
struct UserData
{
    NintendoAccountId accountId;
    char nickname[nn::account::NicknameBytesMax];
    char miiUri[MaxMiiUriLength];
};

// device_owner
struct ServerDeviceOwner
{
    ServerDeviceId owningDeviceId;
    UserData user;
};

struct EtagInfo
{
    uint64_t lastUpdatedAt;
    char etag[MaxEtagLength];
};

// event id for notification data
enum class NotificationEventId
{
    // DID_CHANGE_PARENTAL_CONTROL_SETTING
    ChangedParentalControlSetting = 0,
    // DID_UNLINK_NX
    Unlinked,
    // NX_ONLINE_CHECK
    OnlineCheck,
    // UPDATE_ALARM_SETTING_STATE
    UpdateAlarmSettingState,
    EventIdCount
};
static const StringValueDefinition NotificationEventIdName[] = {
    DEFINE_STRING_VALUE("DID_CHANGE_PARENTAL_CONTROL_SETTING"), // NotificationEventId::ChangedParentalControlSetting
    DEFINE_STRING_VALUE("DID_UNLINK_NX"),                       // NotificationEventId::Unlinked
    DEFINE_STRING_VALUE("NX_ONLINE_CHECK"),                     // NotificationEventId::OnlineCheck
    DEFINE_STRING_VALUE("UPDATE_ALARM_SETTING_STATE"),          // NotificationEventId::UpdateAlarmSettingState
};
NN_STATIC_ASSERT(static_cast<int>(NotificationEventId::EventIdCount) == std::extent<decltype(NotificationEventIdName)>::value);

enum class AlarmSettingState
{
    Success = 0,
    ToVisible,
    ToInvisible,
    Failed,
    Count,
    Undefined = -1
};
static const StringValueDefinition AlarmSettingStateName[] = {
    DEFINE_STRING_VALUE("SUCCESS"),      // AlarmSettingState::Success
    DEFINE_STRING_VALUE("TO_VISIBLE"),   // AlarmSettingState::ToVisible
    DEFINE_STRING_VALUE("TO_INVISIBLE"), // AlarmSettingState::ToInvisible
    DEFINE_STRING_VALUE("FAILED"),       // AlarmSettingState::Failed
};
NN_STATIC_ASSERT(static_cast<int>(AlarmSettingState::Count) == std::extent<decltype(AlarmSettingStateName)>::value);

static const StringValueDefinition DeviceAlarmSettingVisibilityName[] = {
    DEFINE_STRING_VALUE("VISIBLE"),   // enabled
    DEFINE_STRING_VALUE("INVISIBLE"), // disabled
};

// (※ サーバーに送信しないデータには DEFINE_EMPTY_STRING_VALUE による空定義を追加)
static const StringValueDefinition DeviceEventTypeName[] = {
    DEFINE_STRING_VALUE(""),                                // EventType::Invalid
    DEFINE_STRING_VALUE("DID_DEVICE_LAUNCH"),               // EventType::DidDeviceLaunch
    DEFINE_STRING_VALUE("DID_WAKEUP"),                      // EventType::DidWakeup
    DEFINE_STRING_VALUE("DID_SLEEP"),                       // EventType::DidSleep
    DEFINE_STRING_VALUE("DID_APP_LAUNCH"),                  // EventType::DidApplicationLaunch
    DEFINE_STRING_VALUE("DID_APP_TERMINATE"),               // EventType::DidApplicationTerminate
    DEFINE_STRING_VALUE("IDLE"),                            // EventType::Idle
    DEFINE_STRING_VALUE("DID_UNLOCK"),                      // EventType::DidUnlock
    DEFINE_STRING_VALUE("DID_LOCK"),                        // EventType::DidLock
    DEFINE_STRING_VALUE("DID_WRONG_UNLOCK_CODE"),           // EventType::DidWrongUnlockCode
    DEFINE_STRING_VALUE("DID_COME_ONLINE"),                 // EventType::DidComeOnline
    DEFINE_STRING_VALUE("DID_GO_OFFLINE"),                  // EventType::DidGoOffline
    DEFINE_STRING_VALUE("DID_ADD_MANAGED_APP"),             // EventType::DidAddNewManagedApplication
    DEFINE_STRING_VALUE("DID_REMOVE_MANAGED_APP"),          // EventType::DidRemoveManagedApplication
    DEFINE_STRING_VALUE("DID_INTERRUPT_PLAYING"),           // EventType::DidInterruptPlaying
    DEFINE_STRING_VALUE("DID_APP_PLAY"),                    // EventType::DidApplicationPlay
    DEFINE_STRING_VALUE("DID_ALARM_MAKE_INVISIBLE"),        // EventType::DidAlarmMakeInvisible
    DEFINE_STRING_VALUE("DID_ALARM_MAKE_VISIBLE"),          // EventType::DidAlarmMakeVisible
    DEFINE_STRING_VALUE("DID_APP_DOWNLOAD_START"),          // EventType::DidApplicationDownloadStart
    DEFINE_STRING_VALUE("DID_REACH_LIMIT_TIME"),            // EventType::DidReachLimitTime
    DEFINE_EMPTY_STRING_VALUE(),                            // EventType::DidApplicationSuspend
    DEFINE_EMPTY_STRING_VALUE(),                            // EventType::DidApplicationResume
    DEFINE_EMPTY_STRING_VALUE(),                            // EventType::DidUserOpen
    DEFINE_EMPTY_STRING_VALUE(),                            // EventType::DidUserClose
    DEFINE_EMPTY_STRING_VALUE(),                            // EventType::DidShutdown
    DEFINE_EMPTY_STRING_VALUE(),                            // EventType::DidPlayTimerStart
    DEFINE_EMPTY_STRING_VALUE(),                            // EventType::DidPlayTimerStop
    DEFINE_EMPTY_STRING_VALUE(),                            // EventType::DidDeviceActivate
    DEFINE_EMPTY_STRING_VALUE(),                            // EventType::DidLocationNameChange
    DEFINE_STRING_VALUE("DID_FORCE_POWER_OFF"),             // EventType::DidUnexpectedShutdownOccur
    DEFINE_STRING_VALUE("DID_LIMITED_APP_LAUNCH"),          // EventType::DidLimitedApplicationLaunch
};
NN_STATIC_ASSERT(static_cast<int>(EventType::EventTypeMaxCount) == std::extent<decltype(DeviceEventTypeName)>::value);

// 一時解除中でもサーバーに送信するデータには false、送信しない(無視する)データには true を設定
// (true であっても DEFINE_EMPTY_STRING_VALUE で定義されているものは送信されない)
static const bool IgnorableDeviceEventsOnUnlocked[] = {
    true,  // EventType::Invalid
    false, // EventType::DidDeviceLaunch
    false, // EventType::DidWakeup
    false, // EventType::DidSleep
    true,  // EventType::DidApplicationLaunch
    true,  // EventType::DidApplicationTerminate
    false, // EventType::Idle
    false, // EventType::DidUnlock
    false, // EventType::DidLock
    true,  // EventType::DidWrongUnlockCode
    false, // EventType::DidComeOnline
    false, // EventType::DidGoOffline
    false, // EventType::DidAddNewManagedApplication
    false, // EventType::DidRemoveManagedApplication
    true,  // EventType::DidInterruptPlaying
    true,  // EventType::DidApplicationPlay
    false, // EventType::DidAlarmMakeInvisible
    false, // EventType::DidAlarmMakeVisible
    false, // EventType::DidApplicationDownloadStart
    true,  // EventType::DidReachLimitTime
    true,  // EventType::DidApplicationSuspend
    true,  // EventType::DidApplicationResume
    true,  // EventType::DidUserOpen
    true,  // EventType::DidUserClose
    false, // EventType::DidShutdown
    true,  // EventType::DidPlayTimerStart
    true,  // EventType::DidPlayTimerStop
    false, // EventType::DidDeviceActivate
    false, // EventType::DidLocationNameChange
    false, // EventType::DidUnexpectedShutdownOccur
    false, // EventType::DidLimitedApplicationLaunch
};
NN_STATIC_ASSERT(static_cast<int>(EventType::EventTypeMaxCount) == std::extent<decltype(IgnorableDeviceEventsOnUnlocked)>::value);

// @brief ID文字列を uint64_t 型に変換します。(主にJSON解析用)
bool HandleIdString(uint64_t* outValue, const char* valueString, size_t valueLength) NN_NOEXCEPT;

// @brief 長さの決まっている文字列と文字列リテラルの文字列が一致するかどうかを返します。
template <typename T, size_t N>
inline bool IsEqualLiteralStringWithLength(const T* string, size_t stringLen, const T(&literalString)[N]) NN_NOEXCEPT
{
    return (stringLen == N - 1) && (std::strncmp(string, literalString, stringLen) == 0);
}

// StringValueDefinition のデータを元に string を index 値に変換します。
int GetIndexFromString(const char* string, size_t stringLen, const StringValueDefinition* definitions, int count) NN_NOEXCEPT;

template <int N>
inline int GetIndexFromString(const char* string, size_t stringLen, const StringValueDefinition(&definitions)[N]) NN_NOEXCEPT
{
    return GetIndexFromString(string, stringLen, definitions, N);
}

// @brief StringValueDefinition が空定義(DEFINE_EMPTY_STRING_VALUE を使用している)かどうかを返します。
inline bool IsEmptyStringValueDefinitionData(const StringValueDefinition& data) NN_NOEXCEPT
{
    return data.value == nullptr || data.length == 0;
}

}}}}}
