﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/pctl/detail/service/common/pctl_NetworkBuffer.h>

#include <atomic>

namespace nn { namespace pctl { namespace detail { namespace service { namespace common {

/**
 * @brief 非同期処理の呼び出し元で利用する状態管理クラス(抽象クラス)です。
 */
class AsyncContext
{
protected:
    AsyncContext() NN_NOEXCEPT :
        m_Result(nn::ResultSuccess()),
        m_SystemEvent(),
        m_IsFinished(false),
        m_IsBackgroundMode(false),
        m_IsSharableConnectionNecessary(true),
        m_IsSystemEventInitialized(false)
    {
    }

public:
    virtual ~AsyncContext() NN_NOEXCEPT
    {
        if (m_IsSystemEventInitialized)
        {
            nn::os::DestroySystemEvent(&m_SystemEvent);
        }
    }

    /**
     * @brief 状態管理を閉じます。
     * @pre
     *  - IsFinished の戻り値が true である
     * @details
     * 必要に応じてインターフェイス自身が使用するメモリが解放されるため、
     * 以降同一インスタンスに対するメソッド呼び出しを行うことができません。
     */
    virtual void CloseContext() NN_NOEXCEPT = 0;
    /**
     * @brief 現在のスレッドで処理を行います。
     *
     * @details
     * Cancel() とは別スレッドからの呼び出しになります。
     * また、SetFinished は自動では呼び出されません。
     */
    virtual nn::Result Execute(NetworkBuffer& bufferInfo) NN_NOEXCEPT = 0;
    /**
     * @brief 処理を中断します。
     *
     * @details
     * 処理を中断した場合、IsFinished は true と nn::pctl::ResultCanceled を返します。
     */
    virtual void Cancel() NN_NOEXCEPT = 0;
    /**
     * @brief 別の要因で処理の中断がなされたかどうかを返します。
     *
     * @details
     * 内部で Cancelable などを保持する場合はそれが返す値を返します。
     * 保持していない場合は常に false を返します。
     */
    virtual bool IsCancelTriggered() const NN_NOEXCEPT = 0;
    /**
     * @brief 処理が終わっているかどうかを返し、終わっていれば pResult に処理結果を返します。
     * @param[out] pResult 処理が終わっている場合、その処理結果を受け取るポインター
     * @return 処理が終わっていれば true
     */
    bool IsFinished(nn::Result* pResult) const NN_NOEXCEPT;
    /**
     * @brief 処理が終わっているかどうかを返します。
     * @return 処理が終わっていれば true
     */
    bool IsFinished() const NN_NOEXCEPT
    {
        return m_IsFinished;
    }
    /**
     * @brief 処理が終わったことを設定します。
     * @param[in] result 処理結果の nn::Result 値
     */
    void SetFinished(nn::Result result) NN_NOEXCEPT
    {
        m_Result = result;
        m_IsFinished = true;
        if (m_IsSystemEventInitialized)
        {
            nn::os::SignalSystemEvent(&m_SystemEvent);
        }
    }
    /**
     * @brief バックグラウンドで動作することを意図したインスタンスかどうかを返します。
     */
    bool IsBackgroundMode() const NN_NOEXCEPT
    {
        return m_IsBackgroundMode;
    }
    /**
     * @brief バックグラウンドで動作することを意図しているかどうかを設定します。
     * @details
     * バックグラウンド動作を意図している場合、処理が終わったらそのバックグラウンドスレッドで CloseContext が呼び出されます。
     */
    void SetBackgroundMode(bool backgroundMode) NN_NOEXCEPT
    {
        m_IsBackgroundMode = backgroundMode;
    }
    /**
     * @brief 他との接続と共有可能な接続要求を行うモードが必要かどうかを返します。
     */
    bool IsSharableConnectionNecessary() const NN_NOEXCEPT
    {
        return m_IsSharableConnectionNecessary;
    }
    /**
     * @brief 他との接続と共有可能な接続要求を行うモードが必要かどうかを設定します。
     * @details
     * true の場合は接続要求時に RequirementPreset_InternetForSystemProcessSharable を利用します。
     * (既定は true で、false の場合は RequirementPreset_InternetForSystemProcess を利用します。)
     */
    void SetSharableConnectionNecessary(bool sharableConnectionNecessary) NN_NOEXCEPT
    {
        m_IsSharableConnectionNecessary = sharableConnectionNecessary;
    }
    /**
     * @brief 処理終了時にイベントをトリガーさせる目的で SystemEvent を初期化します。
     */
    nn::Result InitializeSystemEvent() NN_NOEXCEPT;
    /**
     * @brief 内部で保持している SystemEvent を返します。未初期化の場合は nullptr が返ります。
     */
    nn::os::SystemEventType* GetSystemEvent() NN_NOEXCEPT
    {
        return m_IsSystemEventInitialized ? &m_SystemEvent : nullptr;
    }

    /**
     * @brief 終了している状態をリセットします。
     */
    void ResetFinished() NN_NOEXCEPT
    {
        m_IsFinished = false;
        m_Result = nn::ResultSuccess();
    }

private:
    nn::Result m_Result;
    nn::os::SystemEventType m_SystemEvent;
    bool m_IsFinished;
    bool m_IsBackgroundMode;
    bool m_IsSharableConnectionNecessary;
    bool m_IsSystemEventInitialized;
};

}}}}}
