﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_ConditionVariable.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/omm/omm_Api.h>
#include <nn/omm/srv/omm_Common.private.h>
#include <nn/hdcp/hdcp.h>
#include <mutex>

#ifdef NN_OMM_AS_VI_MANAGER
#include <nn/vi/sf/vi_ServiceTypes.h>
#else
#include <nn/vi/vi_Display.h>
#endif

namespace nn { namespace omm { namespace srv {
    class OperationModeManager;

    class DisplayManager
    {
    public:
#ifdef NN_OMM_AS_VI_MANAGER
        typedef vi::DisplayId DisplayType;
#else
        typedef vi::Display* DisplayType;
#endif
    public:
        DisplayManager() NN_NOEXCEPT : m_IsTvEnabled(), m_IsLcdEnabled(), m_Mutex(true),
            m_StopHotplugHandlerEvent(os::EventClearMode_ManualClear),
            m_DefaultDisplayResolutionChangeEvent(os::EventClearMode_AutoClear, true),
            m_DefaultDisplayResolutionWidth(), m_DefaultDisplayResolutionHeight(),
            m_ConsoleStyle(){}
        void Initialize(OperationModeManager* pOperationModeManager, ConsoleStyle consoleStyle) NN_NOEXCEPT;
        void EnableHandheldDisplay() NN_NOEXCEPT;
        void EnableConsoleDisplay() NN_NOEXCEPT;
        void DisableAllDisplay() NN_NOEXCEPT;
        void SwitchBacklightOn() NN_NOEXCEPT;
        void SwitchBacklightOff() NN_NOEXCEPT;
        void FadeLcdAndBacklight(bool isIn, TimeSpan span) NN_NOEXCEPT;
        void FadeLcd(bool isIn, TimeSpan span) NN_NOEXCEPT;
        void FadeBacklight(bool isIn, TimeSpan span) NN_NOEXCEPT;
        void FadeTv(bool isIn, TimeSpan span) NN_NOEXCEPT;
        // workaround for SIGLO-33313
        nn::Result UpdatePowerState() NN_NOEXCEPT;

        void StartDrawingLayer() NN_NOEXCEPT;
        void StopDrawingLayer() NN_NOEXCEPT;
        void SetDrawingLayerVisible(bool isOn) NN_NOEXCEPT;
        void SetDrawingLayerTexture(int x, int y, int width, int height, const void* buffer, size_t size) NN_NOEXCEPT;
        void SetStartupLogoLayerVisible(bool isOn) NN_NOEXCEPT;

        void StartHotplugHandler() NN_NOEXCEPT;
        void StopHotplugHandler() NN_NOEXCEPT;

        void GetDefaultDisplayResolution(int* pWidth, int* pHeight) NN_NOEXCEPT;
        os::SystemEvent* GetDefaultDisplayResolutionChangeEvent() NN_NOEXCEPT;
        void UpdateDefaultDisplayResolution(bool isLcd) NN_NOEXCEPT;

        class CecController
        {
            friend DisplayManager;
        private:

            void Initialize(ConsoleStyle consoleStyle) NN_NOEXCEPT;

        public:

            void NotifyCecSettingsChanged() NN_NOEXCEPT;
            void SetApplicationCecSettingsAndNotifyChanged(TvPowerStateMatchingMode mode) NN_NOEXCEPT;

            void Enable() NN_NOEXCEPT;
            void Disable() NN_NOEXCEPT;

            void BeforeSleep()  NN_NOEXCEPT;
            void AfterSleep() NN_NOEXCEPT;

            void TriggerOneTouchPlay() NN_NOEXCEPT;

        private:

            bool ReadCecSettings() NN_NOEXCEPT;
            bool IsCurrentlyCecEnabled() NN_NOEXCEPT;
            Result RunOneImpl(uint64_t sequenceNumber) NN_NOEXCEPT;
            void RunImpl() NN_NOEXCEPT;
            void ManageCecTimerEvent() NN_NOEXCEPT;
            void CecEventMonitor() NN_NOEXCEPT;
            bool ShouldBeActive() NN_NOEXCEPT;
            Result PerformOneTouchPlayImpl(bool* pOut) NN_NOEXCEPT;
            Result PerformOneTouchPlayConsoleOnlyImpl(bool* pOut) NN_NOEXCEPT;
            Result PerformOneTouchPlayDefaultImpl(bool* pOut) NN_NOEXCEPT;
            void CancelImpl(std::unique_lock<os::Mutex> lk) NN_NOEXCEPT;

            os::ThreadType m_CecThread;
            os::ThreadType m_CecMonitor;
            os::Mutex m_Mutex{false};
            os::ConditionVariable m_Condition;
            os::SystemEventType m_CecSystemEvent;
            os::TimerEventType m_CecHpdEvent;
            bool m_HpdState;
            bool m_NeedToStop;
            bool m_IsConsoleOnly;
            bool m_IsSettingsEnabled; // cec 設定が有効かどうか
            bool m_IsEnabled; // cec 機能を有効化するかどうか
            bool m_IsResumed; // スリープによって無効化されていないかどうか
            int m_OneTouchPlayRetryCount = 0;
            uint64_t m_CurrentConnectionSequenceNumber = 0; // 現在の接続のシーケンス番号

            uint64_t m_FaliedConnectionSequenceNumber = 0; // 接続に失敗したシーケンス番号
            TvPowerStateMatchingMode m_ApplicationPowerStateMatchingMode;

            bool TryToRestart() NN_NOEXCEPT;
            void MarkFailed(uint64_t sequenceNumber) NN_NOEXCEPT;
            void MarkNewConnection() NN_NOEXCEPT;

            os::Event m_SuspendedEvent{os::EventClearMode_ManualClear};
            bool m_NeedsCancel = false;
        };

        CecController& GetCecController() NN_NOEXCEPT
        {
            return m_CecController;
        }

        class HdcpManager
        {
            friend DisplayManager;

        private:
            struct TriggerStatus
            {
                OperationMode operationMode;
                hdcp::HdcpAuthenticationState hdcpAuthenticationState;
                bool isHdcpApplicationRunning;
                bool isHdcpApplicationForeground;
                bool isHdmiConneted;
            };

        public:
            HdcpManager() NN_NOEXCEPT :
                m_TriggerStatusMutex{false},
                m_HdcpAuthenticationStateChangeEvent(os::EventClearMode_AutoClear, true),
                m_HdcpAuthenticationFailedEvent(os::EventClearMode_ManualClear, true){}

        private:
            void Initialize(OperationModeManager* pOperationModeManager) NN_NOEXCEPT;

        public:
            os::SystemEvent* GetHdcpAuthenticationFailedEvent() NN_NOEXCEPT;
            HdcpState GetHdcpState() NN_NOEXCEPT;
            os::SystemEvent* GetHdcpStateChangeEvent() NN_NOEXCEPT;
            void NotifyHdcpApplicationExecutionStarted() NN_NOEXCEPT;
            void NotifyHdcpApplicationExecutionFinished() NN_NOEXCEPT;
            void NotifyHdcpApplicationDrawingStarted() NN_NOEXCEPT;
            void NotifyHdcpApplicationDrawingFinished() NN_NOEXCEPT;
            void NotifyOperationModeChanged() NN_NOEXCEPT;
            void NotifyHdmiHotplugConnected() NN_NOEXCEPT;
            void NotifyHdmiHotplugDisconnected() NN_NOEXCEPT;
            bool GetHdcpAuthenticationFailedEmulationEnabled() NN_NOEXCEPT;
            Result SetHdcpAuthenticationFailedEmulation(bool enable) NN_NOEXCEPT;

        private:
            void NotifyHdcpAuthenticationStateChanged() NN_NOEXCEPT;
            void RunImpl() NN_NOEXCEPT;
            bool IsHdcpAuthenticationFailed(hdcp::HdcpAuthenticationState state) const NN_NOEXCEPT;
            HdcpState GetNextHdcpState() NN_NOEXCEPT;
            void UpdateHdcpState() NN_NOEXCEPT;

        private:
            OperationModeManager* m_pOperationModeManager;
            os::ThreadType m_HdcpThread;
            HdcpState m_HdcpState;
            TriggerStatus m_TriggerStatus;
            os::EventType m_TriggerStatusChangeEvent;
            os::Mutex m_TriggerStatusMutex;
            os::SystemEventType m_HdcpAuthenticationTransitionEvent; // hdcp ライブラリから omm がシグナルを受ける
            os::SystemEvent m_HdcpAuthenticationStateChangeEvent;    // omm がユーザ向けにシグナルする
            os::SystemEvent m_HdcpAuthenticationFailedEvent;
            bool m_IsHdcpFailedEmulationAllowed;
            bool m_IsHdcpFailedEmulationEnabled;
        };

        HdcpManager& GetHdcpManager() NN_NOEXCEPT
        {
            return m_HdcpManager;
        }

    private:
        void SetLcdEnabled(bool isOn) NN_NOEXCEPT;
        void SetTvEnabled(bool isOn) NN_NOEXCEPT;
        void HandleTvEnabled() NN_NOEXCEPT;
        void HotplugHandler() NN_NOEXCEPT;
        void UpdateDefaultDisplayResolution(DisplayType display) NN_NOEXCEPT;

        DisplayType m_Tv;
        DisplayType m_Lcd;
        bool m_IsTvEnabled;
        bool m_IsLcdEnabled;

        os::Mutex m_Mutex;
        os::ThreadType m_HotplugHandleThread;
        os::SystemEventType m_HotplugEvent;
        os::Event m_StopHotplugHandlerEvent;
        os::SystemEvent m_DefaultDisplayResolutionChangeEvent;
        int m_DefaultDisplayResolutionWidth;
        int m_DefaultDisplayResolutionHeight;

        CecController m_CecController;
        HdcpManager m_HdcpManager;

        ConsoleStyle m_ConsoleStyle;
    };
}}}
