﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 *  @file
 *  @brief  サーバーに対して発行する API です。
 */

#include <nn/nn_ApplicationId.h>

#include <nn/fs/fs_SaveDataTransferVersion2.h>

#include <nn/olsc/srv/olsc_InternalTypes.h>
#include <nn/olsc/srv/adaptor/olsc_AdaptorBase.h>
#include <nn/olsc/srv/database/olsc_SaveDataArchiveInfoCache.h>
#include <nn/util/util_Execution.h>

#include <curl/curl.h>

namespace nn { namespace olsc { namespace srv { namespace transfer {

    /**
    *       @brief      SaveDataArchiveInfo のリストの取得を要求します。
    *
    *       @param[in]   pOutCount   サーバーから返ってくる SaveDataArchiveInfo の数。
    *       @param[in]   saveDataArchiveInfoList   サーバーから返ってくる SaveDataArchiveInfo のリスト。
    *       @param[in]   listCount   saveDataArchiveInfoList の配列の数。
    *       @param[in]   nsaIdToken   使用する NSA ID トークン。
    *       @param[in]   curlHandle   内部の通信処理に使用する libcURL のハンドル。
    *       @param[in]   workBuffer   ワークバッファ。
    *       @param[in]   workBufferSize   ワークバッファのサイズ。
    *       @param[in]   pCancelable   中止を検知するためのオブジェクト。
    *
    *       @return     処理結果を返します。
    *
    */
    Result RequestSaveDataArchiveInfoList(int* pOutCount, SaveDataArchiveInfo saveDataArchiveInfoList[], size_t listCount, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT;

    /**
    *       @brief      SaveDataArchiveInfo のリストの取得を要求します。取得した SaveDataArchiveInfo のリストは与えられた sdaInfoCache に Add します。
    *
    *       @param[in]   sdaInfoCache   サーバーから返ってくる SaveDataArchiveInfo を出力する sdaInfoCache。
    *       @param[in]   nsaIdToken   使用する NSA ID トークン。
    *       @param[in]   curlHandle   内部の通信処理に使用する libcURL のハンドル。
    *       @param[in]   workBuffer   ワークバッファ。
    *       @param[in]   workBufferSize   ワークバッファのサイズ。
    *       @param[in]   pCancelable   中止を検知するためのオブジェクト。
    *
    *       @return     処理結果を返します。
    *
    */
    Result RequestSaveDataArchiveInfoList(database::SaveDataArchiveInfoCache& sdaInfoCache, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT;

    /**
    *       @brief      与えられた appId に合致した SaveDataArchiveInfo の取得を要求します。
    *
    *       @param[in]   pOutCount   サーバーから返ってくる SaveDataArchiveInfo の数。
    *       @param[in]   saveDataArchiveInfoList   サーバーから返ってくる SaveDataArchiveInfo のリスト。
    *       @param[in]   listCount   saveDataArchiveInfoList の配列の数。
    *       @param[in]   nsaIdToken   使用する NSA ID トークン。
    *       @param[in]   appId   Sda を特定するための applicationId。
    *       @param[in]   curlHandle   内部の通信処理に使用する libcURL のハンドル。
    *       @param[in]   workBuffer   ワークバッファ。
    *       @param[in]   workBufferSize   ワークバッファのサイズ。
    *       @param[in]   pCancelable   中止を検知するためのオブジェクト。
    *
    *       @return     処理結果を返します。
    *
    */
    Result RequestSaveDataArchiveInfoList(int* pOutCount, SaveDataArchiveInfo saveDataArchiveList[], size_t listCount, const NsaIdToken& nsaIdToken, ApplicationId appId, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT;

    /**
    *       @brief      与えられた appId に合致した SaveDataArchiveInfo の取得を要求し、得られた SaveDataArchiveInfo から Fixed であるものを出力します。
    *
    *       @param[in]   pOutValue   サーバーから返ってくる Fixed な SaveDataArchiveInfo。
    *       @param[in]   nsaIdToken   使用する NSA ID トークン。
    *       @param[in]   appId   Sda を特定するための applicationId。
    *       @param[in]   curlHandle   内部の通信処理に使用する libcURL のハンドル。
    *       @param[in]   workBuffer   ワークバッファ。
    *       @param[in]   workBufferSize   ワークバッファのサイズ。
    *       @param[in]   pCancelable   中止を検知するためのオブジェクト。
    *
    *       @return     処理結果を返します。
    *
    */
    Result RequestFixedSaveDataArchiveInfo(nn::util::optional<SaveDataArchiveInfo>* pOutValue, const NsaIdToken& nsaIdToken, ApplicationId appId, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT;

    /**
    *       @brief      与えられた sdaId に合致した SaveDataArchiveInfo の取得を要求します。
    *
    *       @param[in]   pOutValue   取得した SaveDataArchiveInfo。
    *       @param[in]   nsaIdToken   使用する NSA ID トークン。
    *       @param[in]   sdaId    SaveDataArchive の ID。
    *       @param[in]   curlHandle   内部の通信処理に使用する libcURL のハンドル。
    *       @param[in]   workBuffer   ワークバッファ。
    *       @param[in]   workBufferSize   ワークバッファのサイズ。
    *       @param[in]   pCancelable   中止を検知するためのオブジェクト。
    *
    *       @return     処理結果を返します。
    *
    */
    Result RequestSaveDataArchiveInfo(nn::util::optional<SaveDataArchiveInfo>* pOutValue, const NsaIdToken& nsaIdToken, SaveDataArchiveId sdaId, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT;

    /**
    *       @brief      指定した SaveDataArchive の ComponentFile の取得を要求します。
    *                   ただし、この API で取得できる ComponentFileInfo には、S3 の URL は入っていません。
    *
    *       @param[in]   pOutCount   サーバーから返ってくる ComponentFileInfo の数。
    *       @param[in]   componentFileInfoList   サーバーから返ってくる ComponentFileInfo のリスト。
    *       @param[in]   listCount   componentFileInfoList の配列の数。
    *       @param[in]   id   SaveDataArchive の ID。
    *       @param[in]   nsaIdToken   使用する NSA ID トークン。
    *       @param[in]   curlHandle   内部の通信処理に使用する libcURL のハンドル。
    *       @param[in]   workBuffer   ワークバッファ。
    *       @param[in]   workBufferSize   ワークバッファのサイズ。
    *       @param[in]   pCancelable   中止を検知するためのオブジェクト。
    *
    *       @return     処理結果を返します。
    *
    */
    Result RequestComponentFileInfoList(int* pOutCount, ComponentFileInfo componentFileList[], size_t listCount, SaveDataArchiveId id, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT;

    /**
    *       @brief      SaveDataArchive のダウンロードを開始します。
    *
    *       @param[in]   pOutCount   サーバーから返ってくる ComponentFileInfo の数。
    *       @param[in]   componentFileInfoList   サーバーから返ってくる ComponentFileInfo のリスト。
    *       @param[in]   listCount   componentFileInfoList の配列の数。
    *       @param[in]   sdaId   SaveDataArchive の ID。
    *       @param[in]   nsaIdToken   ダウンロードに使用する NSA ID トークン。
    *       @param[in]   curlHandle   内部の通信処理に使用する libcURL のハンドル。
    *       @param[in]   workBuffer   ワークバッファ。
    *       @param[in]   workBufferSize   ワークバッファのサイズ。
    *       @param[in]   pCancelable   中止を検知するためのオブジェクト。
    *
    *       @return     処理結果を返します。
    *
    */
    Result RequestStartDownloadSaveDataArchive(int* pOutCount, ComponentFileInfo componentFileList[], size_t listCount, SaveDataArchiveId sdaId, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT;

    /**
    *       @brief      SaveDataArchive のダウンロードを終了します。
    *
    *       @param[in]   sdaId   SaveDataArchive の ID。
    *       @param[in]   nsaIdToken   使用する NSA ID トークン。
    *       @param[in]   curlHandle   内部の通信処理に使用する libcURL のハンドル。
    *       @param[in]   workBuffer   ワークバッファ。
    *       @param[in]   workBufferSize   ワークバッファのサイズ。
    *       @param[in]   pCancelable   中止を検知するためのオブジェクト。
    *
    *       @return     処理結果を返します。
    *
    */
    Result RequestFinishDownloadSaveDataArchive(SaveDataArchiveId sdaId, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT;

    /**
    *       @brief      指定した SaveDataArchive の KeySeedPackage を取得します。
    *
    *       @param[out]  pOutKsp   サーバーで生成された KeySeedPackage。
    *       @param[in]   sdaId   SaveDataArchive の ID。
    *       @param[in]   challenge   KeySeedPackage 生成用の challenge。
    *       @param[in]   nsaIdToken   使用する NSA ID トークン。
    *       @param[in]   curlHandle   内部の通信処理に使用する libcURL のハンドル。
    *       @param[in]   workBuffer   ワークバッファ。
    *       @param[in]   workBufferSize   ワークバッファのサイズ。
    *       @param[in]   pCancelable   中止を検知するためのオブジェクト。
    *
    *       @return     処理結果を返します。
    *
    */
    Result GetKeySeedPackage(fs::SaveDataTransferManagerVersion2::KeySeedPackage* pOutKsp, SaveDataArchiveId sdaId, const fs::SaveDataTransferManagerForCloudBackUp::Challenge& challenge, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT;

    // TODO Util などに移植予定

    // SCSI ストレージサーバーに対して GET を行う API
    Result GetImpl(nn::olsc::srv::adaptor::AdaptorBase<8,128>& adaptor, const NsaIdToken& nsaIdToken, CURL* curlHandle, const char* url, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT;
    // SCSI ストレージサーバーに対して POST を行う API
    Result PostImpl(nn::olsc::srv::adaptor::AdaptorBase<8,128>& adaptor, const NsaIdToken& nsaIdToken, const char* url, const char* postData, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT;

}}}} // namespace nn::olsc::srv::transfer
